﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/settings/system/settings_Network.h>
#include <nn/settings/system/settings_WirelessLan.h>
#include <nnt/nntest.h>

namespace {

//!< テスト用ネットワーク設定 A
const ::nn::settings::system::NetworkSettings NetworkSettingsA = {
    { 'S', 'e', 't', 't', 'i', 'n', 'g', 's', 'A' },
    { 95, 60, 67, 49, 25, 83, 23, 57, 85, 30, 54, 97, 94, 79, 27, 64 },
    {},
    static_cast<int32_t>(::nn::settings::system::NicType_Ieee80211),
    {
        {},
        { '"', 'f', 'o', 'o', 'b', 'a', 'r', '"' },
        8,
    },
    {
        ::nn::settings::system::AuthenticationType_Shared,
        ::nn::settings::system::EncryptionMode_Wep,
        "opensesame123",
        13
    },
    {
        {},
        { 10, 11, 12, 13 },
        { 20, 21, 22, 23 },
        { 30, 31, 32, 33 },
        { 40, 41, 42, 43 },
        { 50, 51, 52, 53 },
    },
    {
        {},
        "localhost",
        static_cast<uint16_t>(8080),
        "user0",
        "password",
    },
    static_cast<uint16_t>(1000),
};

//!< テスト用ネットワーク設定 B
const ::nn::settings::system::NetworkSettings NetworkSettingsB = {
    { 'S', 'e', 't', 't', 'i', 'n', 'g', 's', 'B' },
    { 82, 92, 93, 76, 22, 46, 83, 49, 83, 49, 77, 10, 28, 76, 15, 95 },
    {},
    static_cast<int32_t>(::nn::settings::system::NicType_Ethernet),
    {},
    {},
    {
        {},
        { 60, 61, 62, 63 },
        { 70, 71, 72, 73 },
        { 80, 81, 82, 83 },
        { 90, 91, 92, 93 },
        { 10, 11, 12, 13 },
    },
    {
        {},
        "localhost",
        static_cast<uint16_t>(80),
        "user1",
        "password",
    },
    static_cast<uint16_t>(1500),
};

::nn::settings::system::NetworkSettings g_NetworkSettings[
    ::nn::settings::system::NetworkSettingsCountMax];

//!< ネットワーク設定が等しいことを期待します。
void ExpectEqualNetworkSettings(
    const ::nn::settings::system::NetworkSettings& lhs,
    const ::nn::settings::system::NetworkSettings& rhs) NN_NOEXCEPT;

} // namespace

//!< 無線 LAN が有効か否かを表す値の設定を取得可能か
TEST(NetworkSuite, WirelessLanEnableFlagTest1)
{
    ::nn::settings::system::IsWirelessLanEnabled();
}

//!< 無線 LAN が有効か否かを表す値の設定を変更可能か
TEST(NetworkSuite, WirelessLanEnableFlagTest2)
{
    ::nn::settings::system::SetWirelessLanEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsWirelessLanEnabled());

    ::nn::settings::system::SetWirelessLanEnabled(false);

    EXPECT_FALSE(::nn::settings::system::IsWirelessLanEnabled());

    ::nn::settings::system::SetWirelessLanEnabled(true);

    EXPECT_TRUE(::nn::settings::system::IsWirelessLanEnabled());
}

//!< ネットワーク設定を取得可能か
TEST(NetworkSuite, NetworkSettingsTest1)
{
    ::nn::settings::system::GetNetworkSettings(&g_NetworkSettings[0], 1);
}

//!< ネットワーク設定を変更可能か
TEST(NetworkSuite, NetworkSettingsTest2)
{
    ::nn::settings::system::SetNetworkSettings(&NetworkSettingsA, 1);

    ::nn::settings::system::NetworkSettings* settings = &g_NetworkSettings[0];

    const auto countMax = ::nn::settings::system::NetworkSettingsCountMax;

    EXPECT_EQ(1,
              ::nn::settings::system::GetNetworkSettings(settings, countMax));

    {
        SCOPED_TRACE("");
        ExpectEqualNetworkSettings(NetworkSettingsA, settings[0]);
    }

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = (i % 2 == 1) ? NetworkSettingsA : NetworkSettingsB;
    }

    ::nn::settings::system::SetNetworkSettings(settings, countMax);

    for (size_t i = 0; i < countMax; ++i)
    {
        settings[i] = ::nn::settings::system::NetworkSettings();
    }

    EXPECT_EQ(countMax,
              ::nn::settings::system::GetNetworkSettings(settings, countMax));

    for (size_t i = 0; i < countMax; ++i)
    {
        SCOPED_TRACE("");
        ExpectEqualNetworkSettings(
            (i % 2 == 1) ? NetworkSettingsA : NetworkSettingsB, settings[i]);
    }

    ::nn::settings::system::SetNetworkSettings(nullptr, 0);

    EXPECT_EQ(0,
              ::nn::settings::system::GetNetworkSettings(settings, countMax));
}

namespace {

void ExpectEqualNetworkSettings(
    const ::nn::settings::system::NetworkSettings& lhs,
    const ::nn::settings::system::NetworkSettings& rhs) NN_NOEXCEPT
{
    for (size_t i = 0; i < sizeof(lhs.name) / sizeof(lhs.name[0]); ++i)
    {
        EXPECT_EQ(lhs.name[i], rhs.name[i]);
    }

    for (size_t i = 0; i < sizeof(lhs.id); ++i)
    {
        EXPECT_EQ(lhs.id[i], rhs.id[i]);
    }

    EXPECT_EQ(lhs.flags, rhs.flags);
    EXPECT_EQ(lhs.nicType, rhs.nicType);
    EXPECT_EQ(lhs.ssidSettings.flags, rhs.ssidSettings.flags);

    for (size_t i = 0; i < sizeof(lhs.ssidSettings.ssid); ++i)
    {
        EXPECT_EQ(lhs.ssidSettings.ssid[i], rhs.ssidSettings.ssid[i]);
    }

    EXPECT_EQ(lhs.ssidSettings.ssidLength, rhs.ssidSettings.ssidLength);

    EXPECT_EQ(lhs.securitySettings.authenticationMode,
              rhs.securitySettings.authenticationMode);
    EXPECT_EQ(lhs.securitySettings.encryptionMode,
              rhs.securitySettings.encryptionMode);

    for (size_t i = 0; i < sizeof(lhs.securitySettings.keyMaterial); ++i)
    {
        EXPECT_EQ(lhs.securitySettings.keyMaterial[i],
                  rhs.securitySettings.keyMaterial[i]);
    }

    EXPECT_EQ(lhs.securitySettings.keyMaterialLength,
              rhs.securitySettings.keyMaterialLength);

    EXPECT_EQ(lhs.ipSettings.flags, rhs.ipSettings.flags);
    EXPECT_EQ(lhs.ipSettings.ipAddress[0], rhs.ipSettings.ipAddress[0]);
    EXPECT_EQ(lhs.ipSettings.ipAddress[1], rhs.ipSettings.ipAddress[1]);
    EXPECT_EQ(lhs.ipSettings.ipAddress[2], rhs.ipSettings.ipAddress[2]);
    EXPECT_EQ(lhs.ipSettings.ipAddress[3], rhs.ipSettings.ipAddress[3]);
    EXPECT_EQ(lhs.ipSettings.subnetMask[0], rhs.ipSettings.subnetMask[0]);
    EXPECT_EQ(lhs.ipSettings.subnetMask[1], rhs.ipSettings.subnetMask[1]);
    EXPECT_EQ(lhs.ipSettings.subnetMask[2], rhs.ipSettings.subnetMask[2]);
    EXPECT_EQ(lhs.ipSettings.subnetMask[3], rhs.ipSettings.subnetMask[3]);
    EXPECT_EQ(lhs.ipSettings.defaultGateway[0],
              rhs.ipSettings.defaultGateway[0]);
    EXPECT_EQ(lhs.ipSettings.defaultGateway[1],
              rhs.ipSettings.defaultGateway[1]);
    EXPECT_EQ(lhs.ipSettings.defaultGateway[2],
              rhs.ipSettings.defaultGateway[2]);
    EXPECT_EQ(lhs.ipSettings.defaultGateway[3],
              rhs.ipSettings.defaultGateway[3]);
    EXPECT_EQ(lhs.ipSettings.preferredDns[0], rhs.ipSettings.preferredDns[0]);
    EXPECT_EQ(lhs.ipSettings.preferredDns[1], rhs.ipSettings.preferredDns[1]);
    EXPECT_EQ(lhs.ipSettings.preferredDns[2], rhs.ipSettings.preferredDns[2]);
    EXPECT_EQ(lhs.ipSettings.preferredDns[3], rhs.ipSettings.preferredDns[3]);
    EXPECT_EQ(lhs.ipSettings.alternateDns[0], rhs.ipSettings.alternateDns[0]);
    EXPECT_EQ(lhs.ipSettings.alternateDns[1], rhs.ipSettings.alternateDns[1]);
    EXPECT_EQ(lhs.ipSettings.alternateDns[2], rhs.ipSettings.alternateDns[2]);
    EXPECT_EQ(lhs.ipSettings.alternateDns[3], rhs.ipSettings.alternateDns[3]);

    EXPECT_EQ(lhs.proxySettings.flags, rhs.proxySettings.flags);

    for (size_t i = 0; i < sizeof(lhs.proxySettings.hostName); ++i)
    {
        EXPECT_EQ(lhs.proxySettings.hostName[i], rhs.proxySettings.hostName[i]);
    }

    EXPECT_EQ(lhs.proxySettings.portNumber, rhs.proxySettings.portNumber);

    for (size_t i = 0; i < sizeof(lhs.proxySettings.userName); ++i)
    {
        EXPECT_EQ(lhs.proxySettings.userName[i], rhs.proxySettings.userName[i]);
    }

    for (size_t i = 0; i < sizeof(lhs.proxySettings.password); ++i)
    {
        EXPECT_EQ(lhs.proxySettings.password[i], rhs.proxySettings.password[i]);
    }

    EXPECT_EQ(lhs.mtu, rhs.mtu);
}

} // namespace
