﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/settings/settings_Language.h>
#include <nn/settings/system/settings_Language.h>
#include <nnt/nntest.h>

namespace {

//!< 言語を表す構造体です。
struct Language final
{
    ::nn::settings::Language value; //!< 言語
    char string[8];                 //!< IETF 言語タグ形式の言語コード
};

const Language Languages[] =
{
    { ::nn::settings::Language_Japanese,                "ja"      },
    { ::nn::settings::Language_AmericanEnglish,         "en-US"   },
    { ::nn::settings::Language_French,                  "fr"      },
    { ::nn::settings::Language_German,                  "de"      },
    { ::nn::settings::Language_Italian,                 "it"      },
    { ::nn::settings::Language_Spanish,                 "es"      },
    { ::nn::settings::Language_Chinese,                 "zh-CN"   },
    { ::nn::settings::Language_Korean,                  "ko"      },
    { ::nn::settings::Language_Dutch,                   "nl"      },
    { ::nn::settings::Language_Portuguese,              "pt"      },
    { ::nn::settings::Language_Russian,                 "ru"      },
    { ::nn::settings::Language_Taiwanese,               "zh-TW"   },
    { ::nn::settings::Language_BritishEnglish,          "en-GB"   },
    { ::nn::settings::Language_CanadianFrench,          "fr-CA"   },
    { ::nn::settings::Language_LatinAmericanSpanish,    "es-419"  },
    { ::nn::settings::Language_SimplifiedChinese,       "zh-Hans" },
    { ::nn::settings::Language_TraditionalChinese,      "zh-Hant" },
};

} // namespace

//!< 全ての言語の書き込みは正しくできるか
TEST(LanguageCodeSuite, WriteAllLanguageCode)
{
    for (const Language& language : Languages)
    {
        const auto code = ::nn::settings::LanguageCode::Make(language.value);

        EXPECT_STREQ(code.string, language.string);

        ::nn::settings::system::SetLanguageCode(code);

        ::nn::settings::LanguageCode value = {};

        ::nn::settings::GetLanguageCode(&value);

        EXPECT_STREQ(code.string, value.string);
    }
}

//!< 異なった言語コードは、別々と認識されるか
TEST(LanguageCodeSuite, CheckNotEqualAnotherLanguage)
{
    for (const Language& language1 : Languages)
    {
        const auto code = ::nn::settings::LanguageCode::Make(language1.value);

        for (const Language& language2 : Languages)
        {
            if (language1.value == language2.value)
            {
                EXPECT_EQ(language2.value, code);

                EXPECT_EQ(
                    ::nn::settings::LanguageCode::Make(language2.value), code);
            }
            else
            {
                EXPECT_NE(language2.value, code);

                EXPECT_NE(
                    ::nn::settings::LanguageCode::Make(language2.value), code);
            }
        }
    }
}

//!< 言語コードの一覧は正しく取得できるか
TEST(LanguageCodeSuite, CheckLanguageCodeList)
{
    const size_t count = sizeof(Languages) / sizeof(Languages[0]);

    ASSERT_EQ(static_cast<int>(count),
              ::nn::settings::GetAvailableLanguageCodeCount());

    ::nn::settings::LanguageCode codes[count + 1] = {};

    // 取得できる言語数と、取得した言語数は等しいか
    ASSERT_EQ(static_cast<int>(count),
              ::nn::settings::GetAvailableLanguageCodes(
                  codes, static_cast<int>(sizeof(codes) / sizeof(codes[0]))));

    // 順序は正しいか
    for (size_t i = 0; i < count; ++i)
    {
        EXPECT_EQ(Languages[i].value, codes[i]);
    }
}
