﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nn/settings/settings_Xpad.h>
#include <nnt/nntest.h>

const ::nn::hid::NpadIdType SupportedNpadIds[] =
{
    ::nn::hid::NpadId::No1,
    ::nn::hid::NpadId::No2,
    ::nn::hid::NpadId::No3,
    ::nn::hid::NpadId::No4
};

const int SupportedNpadIdCount = NN_ARRAY_SIZE(SupportedNpadIds);

class XpadSuiteWithParam : public ::testing::TestWithParam<int>
{
protected:

    static ::nn::settings::BasicXpadGenericPadMap
        g_BasicXpadGenericPadMaps1[SupportedNpadIdCount];

    static ::nn::settings::BasicXpadGenericPadMap
        g_BasicXpadGenericPadMaps2[SupportedNpadIdCount];

    static ::nn::settings::BasicXpadKeyboardMap
        g_BasicXpadKeyboardMaps1[SupportedNpadIdCount];

    static ::nn::settings::BasicXpadKeyboardMap
        g_BasicXpadKeyboardMaps2[SupportedNpadIdCount];

    //!< 割り当てをファジングします。
    template<typename T>
    void FuzzMap(T* outValue) NN_NOEXCEPT;

    //!< Xpad に対する汎用コントローラの割り当てをファジングします。
    void FuzzXpadGenericPadMap(
        ::nn::settings::BasicXpadGenericPadMap* outValue) NN_NOEXCEPT;

    //!< Xpad に対する汎用コントローラの割り当てが初期値であることを期待します。
    void ExpectInitialXpadGenericPadMap(
        const ::nn::settings::BasicXpadGenericPadMap& map) NN_NOEXCEPT;

    //!< Xpad に対するキーボードの割り当てが初期値であることを期待します。
    void ExpectInitialXpadKeyboardMap(
        const ::nn::settings::BasicXpadKeyboardMap& map) NN_NOEXCEPT;

    //!< 割り当てのファジングを期待します。
    template<typename T>
    void ExpectFuzzedMap(const T& map) NN_NOEXCEPT;

    //!< Xpad に対する汎用コントローラの割り当てのファジングを期待します。
    void ExpectFuzzedXpadGenericPadMap(
        const ::nn::settings::BasicXpadGenericPadMap& map) NN_NOEXCEPT;

    //!< 割り当てが等しいことを期待します。
    template<typename T>
    void ExpectSameMap(const T& lhs, const T& rhs) NN_NOEXCEPT;

    //!< Xpad に対する汎用コントローラの割り当てが等しいことを期待します。
    void ExpectSameXpadGenericPadMap(
        const ::nn::settings::BasicXpadGenericPadMap& lhs,
        const ::nn::settings::BasicXpadGenericPadMap& rhs) NN_NOEXCEPT;
};

INSTANTIATE_TEST_CASE_P(XpadSuite, XpadSuiteWithParam, ::testing::Range(0, 4));

TEST_P(XpadSuiteWithParam, XpadGenericPadMapTest)
{
    for (int i = 0; i < SupportedNpadIdCount; ++i)
    {
        SCOPED_TRACE(::testing::Message() << "i = " << i);

        ::nn::settings::GetXpadGenericPadMap(
            &g_BasicXpadGenericPadMaps1[i],
            SupportedNpadIds[i]);
        ExpectInitialXpadGenericPadMap(g_BasicXpadGenericPadMaps1[i]);

        if (GetParam() == 0)
        {
            for (int j = 0; j < i; ++j)
            {
                SCOPED_TRACE(::testing::Message() << "j = " << j);
                EXPECT_NE(g_BasicXpadGenericPadMaps1[j].genericPadId,
                          g_BasicXpadGenericPadMaps1[i].genericPadId);
            }
        }
    }

    ::nn::settings::BasicXpadGenericPadMap& map =
          g_BasicXpadGenericPadMaps1[GetParam()];
    FuzzXpadGenericPadMap(&map);
    map.genericPadId = 10;
    ::nn::settings::SetXpadGenericPadMap(
        map,
        SupportedNpadIds[GetParam()]);

    for (int i = 0; i < SupportedNpadIdCount; ++i)
    {
        SCOPED_TRACE(::testing::Message() << "i = " << i);

        ::nn::settings::GetXpadGenericPadMap(
            &g_BasicXpadGenericPadMaps2[i],
            SupportedNpadIds[i]);

        if (i == GetParam())
        {
            EXPECT_EQ(10, g_BasicXpadGenericPadMaps2[i].genericPadId);
            ExpectFuzzedXpadGenericPadMap(g_BasicXpadGenericPadMaps2[i]);
        }
        else
        {
            ExpectSameXpadGenericPadMap(g_BasicXpadGenericPadMaps1[i],
                                        g_BasicXpadGenericPadMaps2[i]);
        }
    }

    for (int i = 0; i < SupportedNpadIdCount; ++i)
    {
        ::nn::settings::ResetXpadGenericPadMap(
            SupportedNpadIds[i]);
        ::nn::settings::GetXpadGenericPadMap(
            &g_BasicXpadGenericPadMaps1[i],
            SupportedNpadIds[i]);
        ExpectInitialXpadGenericPadMap(g_BasicXpadGenericPadMaps1[i]);
    }
}

TEST_P(XpadSuiteWithParam, XpadKeyboardMapTest)
{
    for (int i = 0; i < SupportedNpadIdCount; ++i)
    {
        SCOPED_TRACE(::testing::Message() << "i = " << i);

        ::nn::settings::GetXpadKeyboardMap(
            &g_BasicXpadKeyboardMaps1[i],
            SupportedNpadIds[i]);
        ExpectInitialXpadKeyboardMap(g_BasicXpadKeyboardMaps1[i]);
    }

    ::nn::settings::BasicXpadKeyboardMap& map =
          g_BasicXpadKeyboardMaps1[GetParam()];
    FuzzMap(&map);
    ::nn::settings::SetXpadKeyboardMap(
        map,
        SupportedNpadIds[GetParam()]);

    for (int i = 0; i < SupportedNpadIdCount; ++i)
    {
        SCOPED_TRACE(::testing::Message() << "i = " << i);

        ::nn::settings::GetXpadKeyboardMap(
            &g_BasicXpadKeyboardMaps2[i],
            SupportedNpadIds[i]);

        if (i == GetParam())
        {
            ExpectFuzzedMap(g_BasicXpadKeyboardMaps2[i]);
        }
        else
        {
            ExpectSameMap(g_BasicXpadKeyboardMaps1[i],
                          g_BasicXpadKeyboardMaps2[i]);
        }
    }

    for (int i = 0; i < SupportedNpadIdCount; ++i)
    {
        ::nn::settings::ResetXpadKeyboardMap(
            SupportedNpadIds[i]);
        ::nn::settings::GetXpadKeyboardMap(
            &g_BasicXpadKeyboardMaps1[i],
            SupportedNpadIds[i]);
        ExpectInitialXpadKeyboardMap(g_BasicXpadKeyboardMaps1[i]);
    }
}

::nn::settings::BasicXpadGenericPadMap
    XpadSuiteWithParam::g_BasicXpadGenericPadMaps1[SupportedNpadIdCount];

::nn::settings::BasicXpadGenericPadMap
    XpadSuiteWithParam::g_BasicXpadGenericPadMaps2[SupportedNpadIdCount];

::nn::settings::BasicXpadKeyboardMap
    XpadSuiteWithParam::g_BasicXpadKeyboardMaps1[SupportedNpadIdCount];

::nn::settings::BasicXpadKeyboardMap
    XpadSuiteWithParam::g_BasicXpadKeyboardMaps2[SupportedNpadIdCount];

template<typename T>
void XpadSuiteWithParam::FuzzMap(T* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    outValue->buttonA = 'A';
    outValue->buttonB = 'B';
    outValue->buttonX = 'X';
    outValue->buttonY = 'Y';
    outValue->buttonStickL = '+';
    outValue->buttonStickR = '-';
    outValue->buttonL = 'L';
    outValue->buttonR = 'R';
    outValue->buttonZL = '1';
    outValue->buttonZR = '2';
    outValue->buttonStart = '3';
    outValue->buttonSelect = '4';
    outValue->buttonLeft = '5';
    outValue->buttonUp = '6';
    outValue->buttonRight = '7';
    outValue->buttonDown = '8';
}

void XpadSuiteWithParam::FuzzXpadGenericPadMap(
    ::nn::settings::BasicXpadGenericPadMap* outValue) NN_NOEXCEPT
{
    FuzzMap(outValue);
    outValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>(true);
    outValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>(false);
    outValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>(true);
    outValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>(false);
    outValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>(true);
    outValue->attributes.Set<
        ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>(true);
}

void XpadSuiteWithParam::ExpectInitialXpadGenericPadMap(
    const ::nn::settings::BasicXpadGenericPadMap& map) NN_NOEXCEPT
{
    EXPECT_EQ(::nn::settings::GenericPadButton_4, map.buttonA);
    EXPECT_EQ(::nn::settings::GenericPadButton_3, map.buttonB);
    EXPECT_EQ(::nn::settings::GenericPadButton_2, map.buttonX);
    EXPECT_EQ(::nn::settings::GenericPadButton_1, map.buttonY);
    EXPECT_EQ(::nn::settings::GenericPadButton_5, map.buttonL);
    EXPECT_EQ(::nn::settings::GenericPadButton_6, map.buttonR);
    EXPECT_EQ(::nn::settings::GenericPadButton_7, map.buttonZL);
    EXPECT_EQ(::nn::settings::GenericPadButton_8, map.buttonZR);
    EXPECT_EQ(::nn::settings::GenericPadButton_10, map.buttonStart);
    EXPECT_EQ(::nn::settings::GenericPadButton_9, map.buttonSelect);
    EXPECT_EQ(::nn::settings::GenericPadButton_11, map.buttonStickL);
    EXPECT_EQ(::nn::settings::GenericPadButton_12, map.buttonStickR);
    EXPECT_EQ(::nn::settings::GenericPadButton_Left, map.buttonLeft);
    EXPECT_EQ(::nn::settings::GenericPadButton_Up, map.buttonUp);
    EXPECT_EQ(::nn::settings::GenericPadButton_Right, map.buttonRight);
    EXPECT_EQ(::nn::settings::GenericPadButton_Down, map.buttonDown);
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>());
}

void XpadSuiteWithParam::ExpectInitialXpadKeyboardMap(
    const ::nn::settings::BasicXpadKeyboardMap& map) NN_NOEXCEPT
{
    EXPECT_EQ(0, map.buttonA);
    EXPECT_EQ(0, map.buttonB);
    EXPECT_EQ(0, map.buttonX);
    EXPECT_EQ(0, map.buttonY);
    EXPECT_EQ(0, map.buttonStickL);
    EXPECT_EQ(0, map.buttonStickR);
    EXPECT_EQ(0, map.buttonL);
    EXPECT_EQ(0, map.buttonR);
    EXPECT_EQ(0, map.buttonZL);
    EXPECT_EQ(0, map.buttonZR);
    EXPECT_EQ(0, map.buttonStart);
    EXPECT_EQ(0, map.buttonSelect);
    EXPECT_EQ(0, map.buttonLeft);
    EXPECT_EQ(0, map.buttonUp);
    EXPECT_EQ(0, map.buttonRight);
    EXPECT_EQ(0, map.buttonDown);
}

template<typename T>
void XpadSuiteWithParam::ExpectFuzzedMap(const T& map) NN_NOEXCEPT
{
    EXPECT_EQ('A', map.buttonA);
    EXPECT_EQ('B', map.buttonB);
    EXPECT_EQ('X', map.buttonX);
    EXPECT_EQ('Y', map.buttonY);
    EXPECT_EQ('+', map.buttonStickL);
    EXPECT_EQ('-', map.buttonStickR);
    EXPECT_EQ('L', map.buttonL);
    EXPECT_EQ('R', map.buttonR);
    EXPECT_EQ('1', map.buttonZL);
    EXPECT_EQ('2', map.buttonZR);
    EXPECT_EQ('3', map.buttonStart);
    EXPECT_EQ('4', map.buttonSelect);
    EXPECT_EQ('5', map.buttonLeft);
    EXPECT_EQ('6', map.buttonUp);
    EXPECT_EQ('7', map.buttonRight);
    EXPECT_EQ('8', map.buttonDown);
}

void XpadSuiteWithParam::ExpectFuzzedXpadGenericPadMap(
    const ::nn::settings::BasicXpadGenericPadMap& map) NN_NOEXCEPT
{
    ExpectFuzzedMap(map);
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>());
    EXPECT_FALSE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>());
    EXPECT_TRUE(
        map.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>());
}

template<typename T>
void XpadSuiteWithParam::ExpectSameMap(const T& lhs, const T& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(lhs.buttonA, rhs.buttonA);
    EXPECT_EQ(lhs.buttonB, rhs.buttonB);
    EXPECT_EQ(lhs.buttonX, rhs.buttonX);
    EXPECT_EQ(lhs.buttonY, rhs.buttonY);
    EXPECT_EQ(lhs.buttonStickL, rhs.buttonStickL);
    EXPECT_EQ(lhs.buttonStickR, rhs.buttonStickR);
    EXPECT_EQ(lhs.buttonL, rhs.buttonL);
    EXPECT_EQ(lhs.buttonR, rhs.buttonR);
    EXPECT_EQ(lhs.buttonZL, rhs.buttonZL);
    EXPECT_EQ(lhs.buttonZR, rhs.buttonZR);
    EXPECT_EQ(lhs.buttonStart, rhs.buttonStart);
    EXPECT_EQ(lhs.buttonSelect, rhs.buttonSelect);
    EXPECT_EQ(lhs.buttonLeft, rhs.buttonLeft);
    EXPECT_EQ(lhs.buttonUp, rhs.buttonUp);
    EXPECT_EQ(lhs.buttonRight, rhs.buttonRight);
    EXPECT_EQ(lhs.buttonDown, rhs.buttonDown);
}

void XpadSuiteWithParam::ExpectSameXpadGenericPadMap(
    const ::nn::settings::BasicXpadGenericPadMap& lhs,
    const ::nn::settings::BasicXpadGenericPadMap& rhs) NN_NOEXCEPT
{
    ExpectSameMap(lhs, rhs);
    EXPECT_EQ(
        lhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>(),
        rhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsXAxisInverted>());
    EXPECT_EQ(
        lhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>(),
        rhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsYAxisInverted>());
    EXPECT_EQ(
        lhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>(),
        rhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsZAxisInverted>());
    EXPECT_EQ(
        lhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>(),
        rhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsRAxisInverted>());
    EXPECT_EQ(
        lhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>(),
        rhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsUAxisInverted>());
    EXPECT_EQ(
        lhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>(),
        rhs.attributes.Test<
            ::nn::settings::GenericPadAxisAttribute::IsVAxisInverted>());
}
