﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <type_traits>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Windows.h>
#include <nn/os/os_Thread.h>
#include <nn/settings/fwdbg/settings_SettingsCommon.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include "testSettings_FileSystem.h"

namespace nn { namespace settings { namespace detail {

::nn::Result ReloadKeyValueStoreForDebug() NN_NOEXCEPT;

}}} // namespace nn::settings::detail

namespace nnt { namespace settings { namespace fwdbg {

namespace {

//!< 既定のディレクトリパス
const char DefaultDirectoryPath[] = ".\\";

//!< システムデータのファイル名
const char SystemDataFileName[] = "Nintendo_SystemData_FwdbgSettingsD";

//!< システムセーブデータのファイル名
const char SystemSaveDataFileName[] = "Nintendo_SystemSaveData_FwdbgSettingsS";

//!< 試行回数
const int TrialCount = 1000;

//!< 試行間隔
const ::nn::TimeSpan TrialInterval = ::nn::TimeSpan::FromMilliSeconds(10);

//!< システムデータのファイルパス
char g_SystemDataFilePath[MAX_PATH] = {};

//!< システムセーブデータのファイルパス
char g_SystemSaveDataFilePath[MAX_PATH] = {};

//!< ファイルのハンドルを返します。
HANDLE CreateFile(const char* path,
                  DWORD accessMode,
                  DWORD shareMode,
                  DWORD creationMode) NN_NOEXCEPT;

//!< 開いているファイルのファイルポインタを移動します。
void SetFilePointer(HANDLE fileHandle, LONG count, DWORD method) NN_NOEXCEPT;

//!< ファイルにデータを書き込みます。
void WriteFile(HANDLE fileHandle, LPCVOID buffer, DWORD size) NN_NOEXCEPT;

} // namespace

void MountFileSystem() NN_NOEXCEPT
{
    int position = ::GetTempPathA(MAX_PATH, g_SystemDataFilePath);

    if (position == 0)
    {
        position = ::nn::util::Strlcpy(
            g_SystemDataFilePath,
            DefaultDirectoryPath,
            static_cast<int>(
                ::std::extent<decltype(DefaultDirectoryPath)>::value));
    }

    ::nn::util::Strlcpy(
        g_SystemSaveDataFilePath, g_SystemDataFilePath, position + 1);

    ::nn::util::Strlcpy(
        &g_SystemDataFilePath[position],
        SystemDataFileName,
        static_cast<int>(
            ::std::extent<decltype(SystemDataFileName)>::value));

    ::nn::util::Strlcpy(
        &g_SystemSaveDataFilePath[position],
        SystemSaveDataFileName,
        static_cast<int>(
            ::std::extent<decltype(SystemSaveDataFileName)>::value));
}

void InitializeSettingsRootDirectory() NN_NOEXCEPT
{
    const char* const filePaths[] = {
        g_SystemDataFilePath,
        g_SystemSaveDataFilePath,
    };

    for (const char* filePath : filePaths)
    {
        HANDLE fileHandle = CreateFile(filePath, GENERIC_WRITE, 0,
                                       CREATE_ALWAYS);

        NN_UTIL_SCOPE_EXIT
        {
            ::CloseHandle(fileHandle);
        };

        const uint32_t totalSize = sizeof(uint32_t);

        WriteFile(fileHandle, &totalSize, sizeof(totalSize));
    }
}

void CreateSettingsDirectory(const char* name) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(name);
    NN_UNUSED(name);
}

void CreateSettingsItemFile(const char* name,
                            const char* key,
                            const void* value,
                            size_t size) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(name);
    NN_ASSERT_NOT_NULL(key);
    NN_ASSERT_NOT_NULL(value);

    {
        const HANDLE fileHandle =
            CreateFile(g_SystemDataFilePath, GENERIC_WRITE, 0, OPEN_EXISTING);

        NN_UTIL_SCOPE_EXIT
        {
            ::CloseHandle(fileHandle);
        };

        SetFilePointer(fileHandle, 0, FILE_END);

        char mapKey[::nn::settings::fwdbg::SettingsNameLengthMax +
                    ::nn::settings::fwdbg::SettingsItemKeyLengthMax + 2] = {};

        ::nn::util::SNPrintf(mapKey, sizeof(mapKey), "%s!%s", name, key);

        const auto mapKeySize =
            static_cast<uint32_t>(::std::strlen(mapKey)) + 1;

        WriteFile(fileHandle, &mapKeySize, sizeof(mapKeySize));

        WriteFile(fileHandle, mapKey, mapKeySize);

        const uint8_t type = 1;

        WriteFile(fileHandle, &type, sizeof(type));

        const auto mapValueSize = static_cast<uint32_t>(size);

        WriteFile(fileHandle, &mapValueSize, sizeof(mapValueSize));

        if (mapValueSize > 0)
        {
            WriteFile(fileHandle, value, mapValueSize);
        }

        const DWORD fileSize = ::GetFileSize(fileHandle, NULL);

        NN_ASSERT_NOT_EQUAL(fileSize, static_cast<DWORD>(-1));

        SetFilePointer(fileHandle, 0, FILE_BEGIN);

        const auto totalSize = static_cast<uint32_t>(fileSize);

        WriteFile(fileHandle, &totalSize, sizeof(totalSize));
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::settings::detail::ReloadKeyValueStoreForDebug());
}

namespace {

HANDLE CreateFile(const char* path,
                  DWORD accessMode,
                  DWORD shareMode,
                  DWORD creationMode) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(path);

    HANDLE fileHandle = INVALID_HANDLE_VALUE;

    for (int i = 0; i < TrialCount; ++i)
    {
        if (i > 0)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        fileHandle = ::CreateFileA(path,
                                   accessMode,
                                   shareMode,
                                   NULL,
                                   creationMode,
                                   FILE_ATTRIBUTE_NORMAL,
                                   NULL);

        if (fileHandle != INVALID_HANDLE_VALUE)
        {
            break;
        }
    }

    NN_ASSERT_NOT_EQUAL(fileHandle, INVALID_HANDLE_VALUE);

    return fileHandle;
}

void SetFilePointer(HANDLE fileHandle, LONG count, DWORD method) NN_NOEXCEPT
{
    DWORD pointer = INVALID_SET_FILE_POINTER;

    for (int i = 0; i < TrialCount; ++i)
    {
        if (i > 0)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        pointer = ::SetFilePointer(fileHandle, count, NULL, method);

        if (pointer != INVALID_SET_FILE_POINTER)
        {
            break;
        }
    }

    NN_ASSERT_NOT_EQUAL(pointer, INVALID_SET_FILE_POINTER);
}

void WriteFile(HANDLE fileHandle, LPCVOID buffer, DWORD size) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(buffer);

    BOOL isSuccess = FALSE;

    DWORD dataSize = 0;

    for (int i = 0; i < TrialCount; ++i)
    {
        if (i > 0)
        {
            ::nn::os::SleepThread(TrialInterval);
        }

        isSuccess = ::WriteFile(fileHandle, buffer, size, &dataSize, NULL);

        if (isSuccess == TRUE)
        {
            break;
        }
    }

    NN_ASSERT_EQUAL(isSuccess, TRUE);

    NN_ASSERT_EQUAL(dataSize, size);
}

} // namespace

}}} // namespace nnt::settings::fwdbg
