﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>

#include <nn/sdmmc/sdmmc_GcAsic.h>
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#include <nn/gc/gc.h>
#include <nn/gc/writer/gc_Writer.h>
#include <nn/dd.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include "Types.h"
#include "Util.h"

#include "sdmmcCommon.h"


#define NN_SC_DETAIL_RETURN_ERROR_NOT_SUPPORTED(message) {\
        NN_SC_DETAIL_ERR_LOG("%s does not support storage type %d.\n", message, m_StorageType);\
        return nn::sdmmc::ResultNotSupported();\
        }

inline nn::Result CheckSdmmcHiddenError(nn::sdmmc::Port port)
{
    nn::sdmmc::ErrorInfo errorInfo;
    char logBuffer[128];
    size_t logSize = 0;
    nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), port);
    bool isError = false;
    if (errorInfo.numActivationFailures > 0)
    {
        NN_SC_DETAIL_ERR_LOG("Port(%d) numActivationFailures: %u\n", port, errorInfo.numActivationFailures);
        isError = true;
    }
    if (errorInfo.numActivationErrorCorrections > 0)
    {
        NN_SC_DETAIL_ERR_LOG("Port(%d) numActivationErrorCorrections: %u\n", port, errorInfo.numActivationErrorCorrections);
        isError = true;
    }
    if (errorInfo.numReadWriteFailures > 0)
    {
        NN_SC_DETAIL_ERR_LOG("Port(%d) numReadWriteFailures: %u\n", port, errorInfo.numReadWriteFailures);
        isError = true;
    }
    if (errorInfo.numReadWriteErrorCorrections > 0)
    {
        NN_SC_DETAIL_ERR_LOG("Port(%d) numReadWriteErrorCorrections: %u\n", port, errorInfo.numReadWriteErrorCorrections);
        isError = true;
    }
    if (logSize > 0)
    {
        NN_SC_DETAIL_ERR_LOG("Port(%d) %s\n", port, logBuffer);
        isError = true;
    }

    if (isError)
    {
        return nn::sdmmc::ResultNotImplemented();   // 実行を止めるために、エラー Result を返す
    }
    else
    {
        return nn::ResultSuccess();
    }
}

class OperationData;

size_t GetDataBufferSize();

class StorageInterface
{
public:
    std::string storageName;
    u8* dataBuffer;
    size_t dataBufferSize;

protected:
    nn::dd::DeviceAddressSpaceType m_Das;
    nn::dd::DeviceVirtualAddress m_WorkBufferDeviceVirtualAddress;
    nn::dd::DeviceVirtualAddress m_DataBufferDeviceVirtualAddress;
    nn::dd::DeviceName m_DeviceName;

    static u8* m_DataBufferAddress;
    nn::sdmmc::Port m_MmcPort;

    StorageInterfaceType m_StorageType;

    u32 m_StorageSizeNumSectors;

public:
    StorageInterface();
    virtual ~StorageInterface(){}

    static void SetDataBufferAddress(u8* address)
    {
        m_DataBufferAddress = address;
    }
    static u8* GetDataBufferAddress()
    {
        return m_DataBufferAddress;
    }

    // 最終は純粋仮想関数にしたい
    virtual nn::Result Activate()
    {
        NN_SC_DETAIL_ERR_LOG("Operation 'Activate' does not support storage type %d.\n", m_StorageType);
        return nn::sdmmc::ResultNotSupported();
    }
    virtual void Deactivate()
    {
        NN_SC_DETAIL_ERR_LOG("Operation 'Deactivate' does not support storage type %d.\n", m_StorageType);
    }
    virtual nn::Result Write(uint32_t sectorIndex, uint32_t numSectors, u8* pDataBuffer, size_t dataBufferSize)
    {
        NN_RESULT_DO(nn::sdmmc::Write(m_MmcPort, sectorIndex, numSectors, pDataBuffer, dataBufferSize));
        NN_RESULT_DO(CheckSdmmcHiddenError(m_MmcPort));
        return nn::ResultSuccess();
    }
    virtual nn::Result Read(u8* pOutDataBuffer, size_t dataBufferSize, uint32_t sectorIndex, uint32_t numSectors)
    {
        NN_RESULT_DO(nn::sdmmc::Read(pOutDataBuffer, dataBufferSize, m_MmcPort, sectorIndex, numSectors));
        NN_RESULT_DO(CheckSdmmcHiddenError(m_MmcPort));
        return nn::ResultSuccess();
    }
    virtual nn::Result Insert(InsertDirection direction)
    {
        NN_SC_DETAIL_ERR_LOG("Operation 'Insert' does not support storage type %d.\n", m_StorageType);
        return nn::sdmmc::ResultNotSupported();
    }
    virtual nn::Result Erase()
    {
        NN_SC_DETAIL_ERR_LOG("Operation 'Erase' does not support storage type %d.\n", m_StorageType);
        return nn::sdmmc::ResultNotSupported();
    }
    virtual nn::Result Enable()
    {
        NN_SC_DETAIL_ERR_LOG("Operation 'Enable' does not support storage type %d.\n", m_StorageType);
        return nn::sdmmc::ResultNotSupported();
    }
    virtual nn::Result SelectPartition(nn::sdmmc::MmcPartition target)
    {
        NN_SC_DETAIL_ERR_LOG("Operation 'Partition' does not support storage type %d.\n", m_StorageType);
        return nn::sdmmc::ResultNotSupported();
    }
    virtual nn::Result Reg();
    virtual nn::Result WriteParameter(WriteParam param, uint32_t value)
    {
        NN_SC_DETAIL_ERR_LOG("Operation 'WriteParameter' does not support storage type %d.\n", m_StorageType);
        return nn::sdmmc::ResultNotSupported();
    }
    nn::Result Speed(nn::sdmmc::SpeedMode target);

    StorageInterfaceType GetStorageType()
    {
        return m_StorageType;
    }
    nn::sdmmc::Port GetMmcPort()
    {
        return m_MmcPort;
    }
    void MapDataBuffer();
    void UnmapDataBuffer();

protected:
    virtual nn::Result GetStorageSize();
};

class StorageInterfaceNand : public StorageInterface
{
protected:
    nn::sdmmc::MmcPartition m_CurrentPartition;

public:
    StorageInterfaceNand();
    ~StorageInterfaceNand();
    virtual nn::Result Activate();
    virtual void Deactivate();
    virtual nn::Result Erase();
    virtual nn::Result SelectPartition(nn::sdmmc::MmcPartition target);
    virtual nn::Result GetStorageSize();
    virtual nn::Result Reg();
};

class StorageInterfaceSd : public StorageInterface
{
public:
    StorageInterfaceSd();
    ~StorageInterfaceSd();
    virtual nn::Result Activate();
    virtual void Deactivate();
    virtual nn::Result Insert(InsertDirection direction);
    virtual nn::Result Reg();
};

void ReadGcCalibrationPartitionToSetInternalKeys();

class StorageInterfaceGc : public StorageInterface
{
protected:
    bool m_IsWriter; // TODO: TEMP:
    bool m_IsSecure; // TODO: TEMP:
    bool m_IsActivated; // TODO: TEMP:
    nn::gc::writer::MemorySize m_CardMemorySize;

public:
    StorageInterfaceGc();
    ~StorageInterfaceGc();
    virtual nn::Result Activate();
    virtual void Deactivate();
    virtual nn::Result Enable();
    virtual nn::Result Write(uint32_t sectorIndex, uint32_t numSectors, u8* pDataBuffer, size_t dataBufferSize);
    virtual nn::Result Read(u8* pOutDataBuffer, size_t dataBufferSize, uint32_t sectorIndex, uint32_t numSectors);
    virtual nn::Result Insert(InsertDirection direction);
    virtual nn::Result Erase();
    virtual nn::Result WriteParameter(WriteParam param, uint32_t value);
    virtual nn::Result GetStorageSize()
    {
        NN_SC_DETAIL_LOG("Warning: %s does not support storage type %d.\n", "GetStorageSize", m_StorageType);
        return nn::ResultSuccess();
    }
    virtual nn::Result Reg();
};
