﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>

#include "Types.h"

// ヘッダのデータ構造
struct DataHeader
{
    // データヘッダ自体の情報
    u16 dataHeaderVersion;
    u8 reserved[2];

    // オペレーションデータの登録情報
    u32 operationDataEntryStartAddress;
    u32 operationDataEntryCount;

    // 引数データの登録情報と実データ開始アドレス
    u32 argumentDataEntryStartAddress;
    u32 argumentDataEntryCount;
    u32 argumentDataBodyStartAddress;

    // バイナリデータの登録情報と実データ開始アドレス
    u32 binaryDataEntryStartAddress;
    u32 binaryDataEntryCount;
    u32 binaryDataBodyStartAddress;

    // ヘッダを含めたデータすべての合計サイズ
    u32 totalDataSize;
};

// オペレーションのエントリのデータ構造
struct OperationDataEntry
{
    // オペレーションの内容を表す実データ
    u16 operationId;
    // パディング
    u8 reserved[2];
    // 引数データの保管場所を示すインデックス
    u32 argIndex;
    // バイナリデータの保管場所を示すインデックス
    u32 dataIndex;
};

class ArgumentData
{
public:
    // 引数データのエントリのデータ構造
    struct ArgumentDataEntry
    {
        // アドレスは最終的に u8* でほしいので構造体を分離
        struct ArgumentDataEntryAddress
        {
            // 開始位置からのオフセット
            u32 offsAddress;
        };
        struct ArgumentDataEntryMain
        {
            // 引数の種類：StorageChecker ではオペレーションIDから対応する引数の種類を判断するので、
            // 引数データ有無以上の情報としては参照しない（argType = 0 ： 引数データなし）
            u8 argType;

            // その他共通設定項目（要パディング）
            u8 reserved[3];
        };
        ArgumentDataEntryAddress addressPart;
        ArgumentDataEntryMain itemPart;
    };

    // ** メンバ
    ArgumentDataEntry::ArgumentDataEntryMain item;
    u8* address;

    // ** メソッド
    ArgumentData()
    {
        address = NULL;
    }
    void SetData(ArgumentDataEntry* entry, u8* dataAddress)
    {
        item = entry->itemPart;
        address = dataAddress;
    }
};

class BinaryData
{
public:
    // バイナリデータのエントリのデータ構造
    struct BinaryDataEntry
    {
        // 開始位置からのオフセット
        u32 offsAddress;
    };

    // ** メンバ
    u8* address;
    u32 size;

    // ** メソッド
    BinaryData()
    {
        address = NULL;
        size = 0;
    }
};

// このクラスが元データの1行に対応する
class OperationData
{
    friend class OperationDataManager;

    // インデックス
    u32 index;

    // 双方向リスト構造
    OperationData* prev;
    OperationData* next;

    // 実行結果
    OpeartionResult result;

public:
    // オペレーションID
    Operation operationId;

    // NOTE: ArgumentData, BinaryData はポインタにした方がデータの重複が多い場合にメモリを節約できる可能性があるが、
    // おそらくポインタ分のメモリ増加に対する重複での削減効果が出るのがそこそこ後であり、メモリ管理も複雑になるのでうまみはなさそう
    // そんなことよりも読み取った後のオリジナルデータのエントリ領域のメモリを開放or再利用する方が節約になる（やらないけど）

    // 引数データ
    ArgumentData arg;

    // バイナリデータ
    BinaryData data;

    // ** メソッド
    OperationData()
    {
        index = 0;
        prev = NULL;
        next = NULL;
        result = OperationResult_Success;
        operationId = Operation_None;
    }

    u32 GetIndex()
    {
        return index;
    }
    OperationData* GetNext()
    {
        return next;
    }
    OperationData* GetPrevious()
    {
        return prev;
    }
    bool IsArgumentExist()
    {
        return arg.item.argType != 0;
    }
    bool IsDataExist()
    {
        return data.size != 0;
    }

    template <typename T>
    T& ParseArgument()
    {
        return *(reinterpret_cast<T*>(arg.address));
    }

    void SetResult(OpeartionResult resultValue)
    {
        result = resultValue;
    }
};


// 与えられたデータバッファをパースしてオペレーションデータのリストを作るクラス
class OperationDataManager
{
public:
    DataHeader dataHeader;
    OperationData* operationDataArray;
    u32 operationDataArrayCount;

public:
    // 元データはバッファごと実体を与えてやる必要がある
    OperationDataManager(u8* workBuffer, u32 workBufferSize);
    ~OperationDataManager();

    void ClearBuffer();
    void ParseDataStructure();

private:
    // 元データの実体へのポインタ
    u8* rawDataBuffer;
    u32 rawDataBufferSize;

    // ** メソッド
    u8* TranslateArgumentDataStartAddress(const u32 offsetAddress)
    {
        u8* address = rawDataBuffer + dataHeader.argumentDataBodyStartAddress + offsetAddress;
        NN_ABORT_UNLESS(address >= rawDataBuffer && address < (rawDataBuffer + rawDataBufferSize));
        return address;
    }
    u8* TranslateBinaryDataStartAddress(const u32 offsetAddress)
    {
        u8* address = rawDataBuffer + dataHeader.binaryDataBodyStartAddress + offsetAddress;
        NN_ABORT_UNLESS(address >= rawDataBuffer && address < (rawDataBuffer + rawDataBufferSize));
        return address;
    }
};

