﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/os.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include "SdmmcCommon.h"
#include <cinttypes>

//#define SELECT_AGING_MODE

#define CHECK_SEQUENTIAL_WRITE
#define CHECK_SEQUENTIAL_READ
#define CHECK_RANDOM_WRITE
#define CHECK_RANDOM_READ

namespace
{
    const uint32_t RandomAlignment      = 16 * 1024;          // ランダムアクセス時に守るアライメント
    const uint32_t NumCheckCountMax     = 128;                // 1 回のチェックでアクセスする回数
    const size_t   BufferSizeMax        = 8 * 1024 * 1024;    // 作業バッファの最大（チェックする最大チャンク以上にする）
    const uint32_t CheckStartingSector  = 0x400000;           // Write でシステムを壊さないためのオフセット
    const size_t DataBufferSize = nn::util::align_up(BufferSizeMax, nn::os::MemoryBlockUnitSize);

    #define  RETURN_IF_FAILURE(result, ...) \
             if (result.IsFailure()){ NN_LOG( __VA_ARGS__ ); return result; }

    #if 1 // MMC
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_Mmc0;
        #if NN_SDMMC_TEST_SMMU_ENABLE
            //const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc2a;
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
        #endif
        const size_t WorkBufferSize = nn::sdmmc::MmcWorkBufferSize;
    #else // SD カード
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_SdCard0;
        #if NN_SDMMC_TEST_SMMU_ENABLE
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
        #endif
        const size_t WorkBufferSize = nn::sdmmc::SdCardWorkBufferSize;
    #endif

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(WorkBufferSize)];
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBufferForHostController[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::sdmmc::WorkBufferSizeForHostController)];
    #endif

    typedef struct Measurements_t
    {
        int64_t speed;
        int64_t iops;
        int64_t min;
        int64_t max;
    } Measurements;

    nn::Result GetMemoryCapacity(uint32_t* pNumSectorsOfMemoryCapacity)
    {
        return nn::sdmmc::GetDeviceMemoryCapacity(pNumSectorsOfMemoryCapacity, TestPort);
    }

    nn::Result GetReadTick(uint8_t* pBuffer, nn::os::Tick* diffTick, uint32_t sectorIndex, size_t chunkSize)
    {
        uint32_t numSectors = chunkSize / nn::sdmmc::SectorSize;
        nn::os::Tick tick = nn::os::GetSystemTick();
        NN_RESULT_DO(nn::sdmmc::Read(pBuffer, BufferSizeMax, TestPort, sectorIndex, numSectors));
        *diffTick = nn::os::GetSystemTick() - tick;

        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
        if ((errorInfo.numReadWriteFailures != 0) || (errorInfo.numReadWriteErrorCorrections != 0))
        {
            NN_ABORT("Read had hidden error. %s\n", logBuffer);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result GetWriteTick(uint8_t* pBuffer, nn::os::Tick* diffTick, uint32_t sectorIndex, size_t chunkSize)
    {
        uint32_t numSectors = chunkSize / nn::sdmmc::SectorSize;
        nn::os::Tick tick = nn::os::GetSystemTick();
        NN_RESULT_DO(nn::sdmmc::Write(TestPort, sectorIndex, numSectors, pBuffer, BufferSizeMax));
        *diffTick = nn::os::GetSystemTick() - tick;

        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
        if ((errorInfo.numReadWriteFailures != 0) || (errorInfo.numReadWriteErrorCorrections != 0))
        {
            NN_ABORT("Write had hidden error. %s\n", logBuffer);
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result MeasureReadSpeed(Measurements* pOutMeasurements, size_t chunkSize, bool isRandomAccess)
    {
        nn::sdmmc::Initialize(TestPort);

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::dd::DeviceAddressSpaceType das;
            SetupDeviceAddressSpace(&das, DeviceName);
            nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

            nn::dd::DeviceAddressSpaceMapInfo info;
            nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
                reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
            deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
            nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
        #endif
        if (TestPort == nn::sdmmc::Port_Mmc0)
        {
            nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
        }
        else
        {
            nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
        }

        #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::dd::DeviceAddressSpaceMapInfo infoForHostController;
                uintptr_t hostControllerBufferAddress = reinterpret_cast<uintptr_t>(g_WorkBufferForHostController);
                size_t hostControllerBufferSize = sizeof(g_WorkBufferForHostController);
                nn::dd::DeviceVirtualAddress hostControllerBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &infoForHostController,
                    hostControllerBufferAddress, hostControllerBufferSize, deviceVirtualAddressOffset);
                deviceVirtualAddressOffset = hostControllerBufferDeviceVirtualAddress + hostControllerBufferSize;
                nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, hostControllerBufferAddress, hostControllerBufferSize, hostControllerBufferDeviceVirtualAddress);
            #endif

            nn::sdmmc::SetWorkBufferForHostController(TestPort, g_WorkBufferForHostController, sizeof(g_WorkBufferForHostController));
        #endif

        uintptr_t dataBufferAddress;
        nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
        NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
            nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        #endif

        result = nn::sdmmc::Activate(TestPort);
        RETURN_IF_FAILURE(result, "Error : Failed to Activate Sdmmc\n");

        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
        if ((errorInfo.numActivationFailures != 0) || (errorInfo.numActivationErrorCorrections != 0))
        {
            NN_ABORT("Activate had hidden error. %s\n", logBuffer);
        }

        // バス情報を表示
        LogBusStatus(TestPort);

        // デバイスサイズの取得（セクタ数）
        uint32_t numSectorsOfMemoryCapacity = 0;
        result = GetMemoryCapacity(&numSectorsOfMemoryCapacity);
        RETURN_IF_FAILURE(result, "Error : Failed to Get Memory Capacity\n");
        NN_LOG("Memory Capacity %u (sectors)\n", numSectorsOfMemoryCapacity);
        NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);

        uint8_t* pBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);
        nn::os::Tick totalTick(0);
        nn::os::Tick minTick(0);
        nn::os::Tick maxTick(0);
        for (uint32_t checkCount = 0; checkCount < NumCheckCountMax; checkCount++)
        {
            srand(checkCount);
            // システム領域を破壊しないようにある程度のオフセットを付ける（Read では破壊しないが Write に合わせる）
            // Sequential アクセス時も一定以下の Chunk サイズのときはアライメントを守ってアクセスする
            uint32_t sectorIndex = CheckStartingSector + (isRandomAccess ? (rand() % (numSectorsOfMemoryCapacity - CheckStartingSector)) & ~(RandomAlignment / nn::sdmmc::SectorSize - 1)
                                                          : ((chunkSize < RandomAlignment) ? RandomAlignment : chunkSize) * checkCount / nn::sdmmc::SectorSize);

            nn::os::Tick tick(0);
            result = GetReadTick(pBuffer, &tick, sectorIndex, chunkSize);
            RETURN_IF_FAILURE(result, "Error : SectorIndex = 0x%08X, ChunkSize = 0x%08zX\n", sectorIndex, chunkSize);

            totalTick += tick;

            //NN_LOG("%" PRId64 " (usec)\n", ConvertToTimeSpan(tick).GetMicroSeconds());

            minTick = (checkCount == 0 || tick < minTick) ? tick : minTick;
            maxTick = (checkCount == 0 || maxTick < tick) ? tick : maxTick;
        }

        nn::TimeSpan timeSpan = ConvertToTimeSpan(totalTick);
        pOutMeasurements->speed = ((static_cast<int64_t>(chunkSize) * NumCheckCountMax * 1000 * 1000 * 1000) / timeSpan.GetNanoSeconds()) / 1024; // X KByte/s
        pOutMeasurements->iops = (static_cast<int64_t>(NumCheckCountMax) * 1000 * 1000 * 1000) / timeSpan.GetNanoSeconds();
        pOutMeasurements->min = ConvertToTimeSpan(minTick).GetMicroSeconds();
        pOutMeasurements->max = ConvertToTimeSpan(maxTick).GetMicroSeconds();

        NN_LOG("%.1f (KB) %" PRId64 " (KB/sec) %" PRId64 " (IOPS) %" PRId64 " (usec) %" PRId64 " (usec)\n",
               chunkSize / 1024.0, pOutMeasurements->speed, pOutMeasurements->iops, pOutMeasurements->min, pOutMeasurements->max);

        nn::sdmmc::Deactivate(TestPort);

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
            UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        #endif
        nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

        #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, hostControllerBufferAddress, hostControllerBufferSize, hostControllerBufferDeviceVirtualAddress);
                UnmapDeviceAddressSpaceAligned(&infoForHostController);
            #endif
        #endif

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
            UnmapDeviceAddressSpaceAligned(&info);

            CleanDeviceAddressSpace(&das, DeviceName);
        #endif

        nn::sdmmc::Finalize(TestPort);

        return result;
    }

    nn::Result MeasureWriteSpeed(Measurements* pOutMeasurements, size_t chunkSize, bool isRandomAccess)
    {
        nn::sdmmc::Initialize(TestPort);

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::dd::DeviceAddressSpaceType das;
            SetupDeviceAddressSpace(&das, DeviceName);
            nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

            nn::dd::DeviceAddressSpaceMapInfo info;
            nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
                reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
            deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
            nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
        #endif
        if (TestPort == nn::sdmmc::Port_Mmc0)
        {
            nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
        }
        else
        {
            nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
        }

        #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::dd::DeviceAddressSpaceMapInfo infoForHostController;
                uintptr_t hostControllerBufferAddress = reinterpret_cast<uintptr_t>(g_WorkBufferForHostController);
                size_t hostControllerBufferSize = sizeof(g_WorkBufferForHostController);
                nn::dd::DeviceVirtualAddress hostControllerBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &infoForHostController,
                    hostControllerBufferAddress, hostControllerBufferSize, deviceVirtualAddressOffset);
                deviceVirtualAddressOffset = hostControllerBufferDeviceVirtualAddress + hostControllerBufferSize;
                nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, hostControllerBufferAddress, hostControllerBufferSize, hostControllerBufferDeviceVirtualAddress);
            #endif

            nn::sdmmc::SetWorkBufferForHostController(TestPort, g_WorkBufferForHostController, sizeof(g_WorkBufferForHostController));
        #endif

        uintptr_t dataBufferAddress;
        nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
        NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");
        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
            nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        #endif

        result = nn::sdmmc::Activate(TestPort);
        RETURN_IF_FAILURE(result, "Error : Failed to Activate Sdmmc\n");

        nn::sdmmc::ErrorInfo errorInfo;
        size_t logSize;
        char logBuffer[256];
        nn::sdmmc::GetAndClearErrorInfo(&errorInfo, &logSize, logBuffer, sizeof(logBuffer), TestPort);
        if ((errorInfo.numActivationFailures != 0) || (errorInfo.numActivationErrorCorrections != 0))
        {
            NN_ABORT("Activate had hidden error. %s\n", logBuffer);
        }

        // バス情報を表示
        LogBusStatus(TestPort);

        // デバイスサイズの取得（セクタ数）
        uint32_t numSectorsOfMemoryCapacity = 0;
        result = GetMemoryCapacity(&numSectorsOfMemoryCapacity);
        RETURN_IF_FAILURE(result, "Error : Failed to Get Memory Capacity\n");
        NN_LOG("Memory Capacity %u (sectors)\n", numSectorsOfMemoryCapacity);
        NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);

        uint8_t* pBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);

        for (uint32_t byte = 0; byte < BufferSizeMax; byte++)
        {
            pBuffer[byte] = (byte % 2 == 0) ? 0xAA : 0x55;
        }

        nn::os::Tick totalTick(0);
        nn::os::Tick minTick(0);
        nn::os::Tick maxTick(0);
        for (uint32_t checkCount = 0; checkCount < NumCheckCountMax; checkCount++)
        {
            srand(checkCount);
            // システム領域を破壊しないようにある程度のオフセットを付ける
            // Sequential アクセス時も一定以下の Chunk サイズのときはアライメントを守ってアクセスする
            uint32_t sectorIndex = CheckStartingSector + (isRandomAccess ? (rand() % (numSectorsOfMemoryCapacity - CheckStartingSector)) & ~(RandomAlignment / nn::sdmmc::SectorSize - 1)
                                    : ((chunkSize < RandomAlignment) ? RandomAlignment : chunkSize) * checkCount / nn::sdmmc::SectorSize);

            nn::os::Tick tick(0);
            result = GetWriteTick(pBuffer, &tick, sectorIndex, chunkSize);
            RETURN_IF_FAILURE(result, "Error : SectorIndex = 0x%08X, ChunkSize = 0x%08zX\n", sectorIndex, chunkSize);
            totalTick += tick;

            //NN_LOG("%" PRId64 " (usec)\n", ConvertToTimeSpan(tick).GetMicroSeconds());

            minTick = (checkCount == 0 || tick < minTick) ? tick : minTick;
            maxTick = (checkCount == 0 || maxTick < tick) ? tick : maxTick;
        }

        nn::TimeSpan timeSpan = ConvertToTimeSpan(totalTick);
        pOutMeasurements->speed = ((static_cast<int64_t>(chunkSize) * NumCheckCountMax * 1000 * 1000 * 1000) / timeSpan.GetNanoSeconds()) / 1024; // X KByte/s
        pOutMeasurements->iops = (static_cast<int64_t>(NumCheckCountMax) * 1000 * 1000 * 1000) / timeSpan.GetNanoSeconds();
        pOutMeasurements->min = ConvertToTimeSpan(minTick).GetMicroSeconds();
        pOutMeasurements->max = ConvertToTimeSpan(maxTick).GetMicroSeconds();

        NN_LOG("%.1f (KB) %" PRId64 " (KB/sec) %" PRId64 " (IOPS) %" PRId64 " (usec) %" PRId64 " (usec)\n",
               chunkSize / 1024.0, pOutMeasurements->speed, pOutMeasurements->iops, pOutMeasurements->min, pOutMeasurements->max);

        nn::sdmmc::Deactivate(TestPort);

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
            UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        #endif
        nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

        #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, hostControllerBufferAddress, hostControllerBufferSize, hostControllerBufferDeviceVirtualAddress);
                UnmapDeviceAddressSpaceAligned(&infoForHostController);
            #endif
        #endif

        #ifdef NN_SDMMC_TEST_SMMU_ENABLE
            nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
            UnmapDeviceAddressSpaceAligned(&info);

            CleanDeviceAddressSpace(&das, DeviceName);
        #endif

        nn::sdmmc::Finalize(TestPort);

        return result;
    }

    void LogMeasurements(const size_t* chunkSizeTable, uint32_t numChunkTable, const Measurements* measurements) NN_NOEXCEPT
    {
        NN_LOG("Chunk(KB) KB/sec IOPS min(usec) max(usec)\n");
        for (uint32_t index = 0; index < numChunkTable; index++)
        {
            NN_LOG("%.1f %" PRId64 " %" PRId64 " %" PRId64 " %" PRId64 "\n",
                   chunkSizeTable[index] / 1024.0,
                   measurements[index].speed, measurements[index].iops,
                   measurements[index].min, measurements[index].max);
        }
    }
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(512 * 1024 * 1024);
    if (result.IsFailure())
    {
        NN_LOG("Failed SetMemoryHeapSize\n");
    }
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    const size_t ChunkSizeTable[] =
    {
        512, (1 * 1024), (2 * 1024), (4 * 1024), (8 * 1024), (16 * 1024),
        (32 * 1024), (64 * 1024), (128 * 1024), (256 * 1024), (512 * 1024),
        (1 * 1024 * 1024), (2 * 1024 * 1024), (4 * 1024 * 1024), (8 * 1024 * 1024),
    };
    const uint32_t NumChunkTable = sizeof(ChunkSizeTable) / sizeof(ChunkSizeTable[0]);

    #ifdef CHECK_SEQUENTIAL_WRITE
        Measurements SequentialWriteMeasurements[NumChunkTable];
    #endif
    #ifdef CHECK_SEQUENTIAL_READ
        Measurements SequentialReadMeasurements[NumChunkTable];
    #endif
    #ifdef CHECK_RANDOM_WRITE
        Measurements RandomWriteMeasurements[NumChunkTable];
    #endif
    #ifdef CHECK_RANDOM_READ
        Measurements RandomReadMeasurements[NumChunkTable];
    #endif

    nn::Result result;

    #ifdef SELECT_AGING_MODE
        NN_LOG("/// SDMMC AGING CHECKER ///\n");
        uint32_t loop = 0;
        loop_bgn:
        NN_LOG("===== %d loop =====\n", loop++);
    #else
        NN_LOG("/// SDMMC SPEED CHECKER ///\n");
    #endif

    #ifdef CHECK_SEQUENTIAL_WRITE
        for (uint32_t index = 0; index < NumChunkTable; index++)
        {
            NN_LOG("Chunk Size: 0x%08zX (## Sequential Write ##)\n", ChunkSizeTable[index]);
            result = MeasureWriteSpeed(&SequentialWriteMeasurements[index], ChunkSizeTable[index], false);
            if (result.IsFailure())
            {
                goto end;
            }
        }
    #endif

    #ifdef CHECK_SEQUENTIAL_READ
        for (uint32_t index = 0; index < NumChunkTable; index++)
        {
            NN_LOG("Chunk Size: 0x%08zX (## Sequential Read ##)\n", ChunkSizeTable[index]);
            result = MeasureReadSpeed(&SequentialReadMeasurements[index], ChunkSizeTable[index], false);
            if (result.IsFailure())
            {
                goto end;
            }
        }
    #endif

    #ifdef CHECK_RANDOM_WRITE
        for (uint32_t index = 0; index < NumChunkTable; index++)
        {
            NN_LOG("Chunk Size: 0x%08zX (## Random Write ##)\n", ChunkSizeTable[index]);
            result = MeasureWriteSpeed(&RandomWriteMeasurements[index], ChunkSizeTable[index], true);
            if (result.IsFailure())
            {
                goto end;
            }
        }
    #endif

    #ifdef CHECK_RANDOM_READ
        for (uint32_t index = 0; index < NumChunkTable; index++)
        {
            NN_LOG("Chunk Size: 0x%08zX (## Random Read ##)\n", ChunkSizeTable[index]);
            result = MeasureReadSpeed(&RandomReadMeasurements[index], ChunkSizeTable[index], true);
            if (result.IsFailure())
            {
                goto end;
            }
        }
    #endif

    end:
    #ifdef CHECK_SEQUENTIAL_WRITE
        NN_LOG("--- Sequential Write ---\n");
        LogMeasurements(ChunkSizeTable, NumChunkTable, SequentialWriteMeasurements);
    #endif

    #ifdef CHECK_SEQUENTIAL_READ
        NN_LOG("--- Sequential Read ---\n");
        LogMeasurements(ChunkSizeTable, NumChunkTable, SequentialReadMeasurements);
    #endif

    #ifdef CHECK_RANDOM_WRITE
        NN_LOG("--- Random Write ---\n");
        LogMeasurements(ChunkSizeTable, NumChunkTable, RandomWriteMeasurements);
    #endif

    #ifdef CHECK_RANDOM_READ
        NN_LOG("--- Random Read ---\n");
        LogMeasurements(ChunkSizeTable, NumChunkTable, RandomReadMeasurements);
    #endif

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);

    NN_LOG("[[ %s ]]\n", result.IsFailure() ? "FAILED !!" : "PASSED !!");
    if (result.IsFailure())
    {
        NN_LOG("Module:%d Description:%d Inner:%d\n",
               result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }

    #ifdef SELECT_AGING_MODE
        goto loop_bgn;
    #endif
}

