﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/dd.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <cstring>
#include "SdmmcCommon.h"

// 注意：初期プロセスの FS の sMMU 設定を無効にした環境でしか動作しません。

namespace
{
    const nn::sdmmc::Port TestPort = nn::sdmmc::Port_SdCard0;
    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        #ifdef NN_BUILD_CONFIG_HARDWARE_JETSONTK1
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc3a;
        #else   // NN_BUILD_CONFIG_HARDWARE_JETSONTK2, NN_BUILD_CONFIG_HARDWARE_NX
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
        #endif
    #endif
    const size_t WorkBufferSize = nn::sdmmc::SdCardWorkBufferSize;

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)

    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(WorkBufferSize)];

    void LogSdCardCurrentConsumption(nn::sdmmc::SpeedMode speedMode)
    {
        uint16_t currentConsumption;
        nn::Result result = nn::sdmmc::GetSdCardCurrentConsumption(&currentConsumption, TestPort, speedMode);
        if (result.IsSuccess())
        {
            NN_LOG("%u\n", currentConsumption);
        }
        else if (nn::sdmmc::ResultSdCardNotSupportSwitchFunctionStatus::Includes(result))
        {
            NN_LOG("is unknown\n");
        }
        else if (nn::sdmmc::ResultSdCardNotSupportAccessMode::Includes(result))
        {
            NN_LOG("is not supported\n");
        }
        else
        {
            NN_LOG("is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        }
    }
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::sdmmc::Initialize(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType das;
        SetupDeviceAddressSpace(&das, DeviceName);
        nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

        nn::dd::DeviceAddressSpaceMapInfo info;
        nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
            reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
        deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
    #endif
    nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));

    NN_LOG("Activate SD Card\n");
    nn::Result result = nn::sdmmc::Activate(TestPort);
    if (result.IsSuccess())
    {
        LogBusStatus(TestPort);

        NN_LOG("Default Speed (mA): ");
        LogSdCardCurrentConsumption(nn::sdmmc::SpeedMode_SdCardDefaultSpeed);
        NN_LOG("High Speed (mA): ");
        LogSdCardCurrentConsumption(nn::sdmmc::SpeedMode_SdCardHighSpeed);
        NN_LOG("SDR50 (mA): ");
        LogSdCardCurrentConsumption(nn::sdmmc::SpeedMode_SdCardSdr50);
        NN_LOG("SDR104 (mA): ");
        LogSdCardCurrentConsumption(nn::sdmmc::SpeedMode_SdCardSdr104);
        NN_LOG("DDR50 (mA): ");
        LogSdCardCurrentConsumption(nn::sdmmc::SpeedMode_SdCardDdr50);

        nn::sdmmc::Deactivate(TestPort);
    }
    else
    {
        NN_LOG("nn::sdmmc::Activate is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&info);

        CleanDeviceAddressSpace(&das, DeviceName);
    #endif

    nn::sdmmc::Finalize(TestPort);

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);
}
