﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/os.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <nn/sdmmc/sdmmc_Mmc.h>
#include <nn/sdmmc/sdmmc_SdCard.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include "SdmmcCommon.h"

#define CHECK_READ
#define CHECK_WRITE

#define INFINITE_MODE
#define ENABLE_ACCESS_LOG

namespace
{
    const uint32_t RandomAlignment      = 16 * 1024;           // ランダムアクセス時に守るアライメント
    const uint32_t CheckChunkSize       = 1 * 1024 * 1024;    // チェックするチャンクサイズ
    const size_t   BufferSizeMax        = 8 * 1024 * 1024;    // 作業バッファの最大（チェックする最大チャンク以上にする）
    const uint32_t CheckStartingSector  = 0x400000;           // Write でシステムを壊さないためのオフセット
    const uint32_t SleepTimeSecond      = 0;                  // アクセス間隔
    const size_t DataBufferSize = nn::util::align_up(BufferSizeMax, nn::os::MemoryBlockUnitSize);

    #define  RETURN_IF_FAILURE(result, ...) \
             if (result.IsFailure()){ NN_LOG( __VA_ARGS__ ); return result; }

    #if 1 // MMC
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_Mmc0;
        #if NN_SDMMC_TEST_SMMU_ENABLE
            //const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc2a;
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
        #endif
        const size_t WorkBufferSize = nn::sdmmc::MmcWorkBufferSize;
    #else // SD カード
        const nn::sdmmc::Port TestPort = nn::sdmmc::Port_SdCard0;
        #if NN_SDMMC_TEST_SMMU_ENABLE
            const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc1a;
        #endif
        const size_t WorkBufferSize = nn::sdmmc::SdCardWorkBufferSize;
    #endif

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)
    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(WorkBufferSize)];

    nn::Result GetMemoryCapacity(uint32_t* pNumSectorsOfMemoryCapacity)
    {
        return nn::sdmmc::GetDeviceMemoryCapacity(pNumSectorsOfMemoryCapacity, TestPort);
    }

    #ifdef CHECK_READ
        nn::Result GetReadTick(uint8_t* pBuffer, nn::os::Tick* diffTick, uint32_t sectorIndex, size_t chunkSize)
        {
            uint32_t numSectors = chunkSize / nn::sdmmc::SectorSize;
            nn::os::Tick tick = nn::os::GetSystemTick();
            nn::Result result = nn::sdmmc::Read(pBuffer, BufferSizeMax, TestPort, sectorIndex, numSectors);
            *diffTick = nn::os::GetSystemTick() - tick;

            #ifdef ENABLE_ACCESS_LOG
                NN_LOG("addr = 0x%08X, data = 0x%02X%02X%02X%02X\n",
                       sectorIndex, pBuffer[0], pBuffer[1], pBuffer[2], pBuffer[3]);
            #endif

            return result;
        }

        nn::Result ReadAccess(size_t chunkSize, uint32_t sleepTime, bool isRandomAccess)
        {
            nn::sdmmc::Initialize(TestPort);

            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::dd::DeviceAddressSpaceType das;
                SetupDeviceAddressSpace(&das, DeviceName);
                nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

                nn::dd::DeviceAddressSpaceMapInfo info;
                nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
                    reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
                deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
                nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
            #endif
            if (TestPort == nn::sdmmc::Port_Mmc0)
            {
                nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
            }
            else
            {
                nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
            }

            uintptr_t dataBufferAddress;
            nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
            NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");
            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
                nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
            #endif

            result = nn::sdmmc::Activate(TestPort);
            RETURN_IF_FAILURE(result, "Error : Failed to Activate Sdmmc\n");

            // バス情報を表示
            LogBusStatus(TestPort);

            // デバイスサイズの取得（セクタ数）
            uint32_t numSectorsOfMemoryCapacity = 0;
            result = GetMemoryCapacity(&numSectorsOfMemoryCapacity);
            RETURN_IF_FAILURE(result, "Error : Failed to Get Memory Capacity\n");
            NN_LOG("Memory Capacity %u (sectors)\n", numSectorsOfMemoryCapacity);
            NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);

            uint8_t* pBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);
            uint32_t checkSectorsCountMax = numSectorsOfMemoryCapacity - CheckStartingSector;

            for (uint32_t checkCount = 0; ; checkCount++)
            {
                if (checkCount == checkSectorsCountMax / (chunkSize / nn::sdmmc::SectorSize))
                {
                    #ifdef INFINITE_MODE
                        checkCount = 0;
                    #else
                        break;
                    #endif
                }

                uint32_t offset;
                if (isRandomAccess == true)
                {
                    srand(checkCount);
                    offset = rand() % (numSectorsOfMemoryCapacity - CheckStartingSector);
                    offset &= ~(RandomAlignment / nn::sdmmc::SectorSize - 1);
                }
                else
                {
                    offset = (((chunkSize < RandomAlignment) ? RandomAlignment : chunkSize) / nn::sdmmc::SectorSize) * checkCount;
                }
                uint32_t sectorIndex = CheckStartingSector + offset;

                nn::os::Tick tick(0);
                result = GetReadTick(pBuffer, &tick, sectorIndex, chunkSize);
                RETURN_IF_FAILURE(result, "Error : SectorIndex = 0x%08X, ChunkSize = 0x%08zX\n", sectorIndex, chunkSize);

                if (sleepTime != 0)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(sleepTime));
                }
            }

            nn::sdmmc::Deactivate(TestPort);

            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
                UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
            #endif
            nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
                UnmapDeviceAddressSpaceAligned(&info);

                CleanDeviceAddressSpace(&das, DeviceName);
            #endif

            nn::sdmmc::Finalize(TestPort);

            return result;
        }
    #endif

    #ifdef CHECK_WRITE
        nn::Result GetWriteTick(uint8_t* pBuffer, nn::os::Tick* diffTick, uint32_t sectorIndex, size_t chunkSize)
        {
            uint32_t numSectors = chunkSize / nn::sdmmc::SectorSize;

            #ifdef ENABLE_ACCESS_LOG
                NN_LOG("addr = 0x%08X, data = 0x%02X%02X%02X%02X\n",
                       sectorIndex, pBuffer[0], pBuffer[1], pBuffer[2], pBuffer[3]);
            #endif

            nn::os::Tick tick = nn::os::GetSystemTick();
            nn::Result result = nn::sdmmc::Write(TestPort, sectorIndex, numSectors, pBuffer, BufferSizeMax);
            *diffTick = nn::os::GetSystemTick() - tick;

            return result;
        }

        nn::Result WriteAccess(size_t chunkSize, uint32_t sleepTime, bool isRandomAccess)
        {
            nn::sdmmc::Initialize(TestPort);

            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::dd::DeviceAddressSpaceType das;
                SetupDeviceAddressSpace(&das, DeviceName);
                nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

                nn::dd::DeviceAddressSpaceMapInfo info;
                nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
                    reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
                deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
                nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
            #endif
            if (TestPort == nn::sdmmc::Port_Mmc0)
            {
                nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
            }
            else
            {
                nn::sdmmc::SetSdCardWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));
            }

            uintptr_t dataBufferAddress;
            nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
            NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");
            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
                nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
            #endif

            result = nn::sdmmc::Activate(TestPort);
            RETURN_IF_FAILURE(result, "Error : Failed to Activate Sdmmc\n");

            // バス情報を表示
            LogBusStatus(TestPort);

            // デバイスサイズの取得（セクタ数）
            uint32_t numSectorsOfMemoryCapacity = 0;
            result = GetMemoryCapacity(&numSectorsOfMemoryCapacity);
            RETURN_IF_FAILURE(result, "Error : Failed to Get Memory Capacity\n");
            NN_LOG("Memory Capacity %u (sectors)\n", numSectorsOfMemoryCapacity);
            NN_LOG("Memory Capacity %u (MB)\n", numSectorsOfMemoryCapacity / 2 / 1024);

            uint8_t* pBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);
            uint32_t checkSectorsCountMax = numSectorsOfMemoryCapacity - CheckStartingSector;

            for (uint32_t checkCount = 0; ; checkCount++)
            {
                if (checkCount == checkSectorsCountMax / (chunkSize / nn::sdmmc::SectorSize))
                {
                    #ifdef INFINITE_MODE
                        checkCount = 0;
                    #else
                        break;
                    #endif
                }

                uint32_t offset;
                if (isRandomAccess == true)
                {
                    srand(checkCount);
                    offset = rand() % (numSectorsOfMemoryCapacity - CheckStartingSector);
                    offset &= ~(RandomAlignment / nn::sdmmc::SectorSize - 1);
                }
                else
                {
                    offset = (((chunkSize < RandomAlignment) ? RandomAlignment : chunkSize) / nn::sdmmc::SectorSize) * checkCount;
                }
                uint32_t sectorIndex = CheckStartingSector + offset;

                // ランダムパターンの生成
                srand(checkCount);
                for (uint32_t byte = 0; byte < chunkSize; byte++)
                {
                    pBuffer[byte] = (rand() & 0xFF);
                }

                nn::os::Tick tick(0);
                result = GetWriteTick(pBuffer, &tick, sectorIndex, chunkSize);
                RETURN_IF_FAILURE(result, "Error : SectorIndex = 0x%08X, ChunkSize = 0x%08zX\n", sectorIndex, chunkSize);

                if (sleepTime != 0)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(sleepTime));
                }
            }

            nn::sdmmc::Deactivate(TestPort);

            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
                UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
            #endif
            nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

            #ifdef NN_SDMMC_TEST_SMMU_ENABLE
                nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
                UnmapDeviceAddressSpaceAligned(&info);

                CleanDeviceAddressSpace(&das, DeviceName);
            #endif

            nn::sdmmc::Finalize(TestPort);

            return result;
        }
    #endif
}

extern "C" void nninitStartup()
{
    nn::Result result = nn::os::SetMemoryHeapSize(512 * 1024 * 1024);
    if (result.IsFailure())
    {
        NN_LOG("Failed SetMemoryHeapSize\n");
    }
}

extern "C" void nnMain()
{
    NN_LOG("/// SDMMC EMI CHECKER ///\n");

    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::Result result;
#ifdef CHECK_READ
    result = ReadAccess(CheckChunkSize, SleepTimeSecond, false);
    if (result.IsFailure())
    {
        NN_LOG("Module:%d Description:%d Inner:%d\n",
               result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }
#endif
#ifdef CHECK_WRITE
    result = WriteAccess(CheckChunkSize, SleepTimeSecond, false);
    if (result.IsFailure())
    {
        NN_LOG("Module:%d Description:%d Inner:%d\n",
               result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }
#endif

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);

    if (result.IsFailure())
    {
        NN_LOG("[[ FAILED !! ]]\n");
    }
    else
    {
        NN_LOG("[[ PASSED !! ]]\n");
    }
}

