﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <cinttypes>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>

#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_ErrorInfoPrivate.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

namespace
{
    const size_t BufferSize = 8 * 1024 * 1024;
    uint8_t g_SendBuffer[BufferSize];
    uint8_t g_ReceiveBuffer[BufferSize];

    const char* MountPath = "sdcard";
    const char* RootPath = "sdcard:/";
    const char* SmallFilePath = "sdcard:/s.bin";
    const char* LargeFilePath = "sdcard:/L.bin";

    int64_t g_ReadSpeedKbPerSec = 0;

    void LogData(uint8_t* pData, size_t size, size_t logAddress) NN_NOEXCEPT
    {
        size_t address = logAddress;
        uint8_t* pCurrentData =  reinterpret_cast<uint8_t*>(pData);
        uint8_t* pDataEnd = pCurrentData + size;
        while (true)
        {
            NN_LOG("%08X: ", address);
            for (uint32_t i = 0; i < 0x10; i++)
            {
                NN_LOG("%02X ", *pCurrentData);
                pCurrentData++;
                if (pCurrentData >= pDataEnd)
                {
                    NN_LOG("\n");
                    return;
                }
            }
            NN_LOG("\n");
            address += 0x10;
        }
    }

    void LogDifferentData(uint8_t* pActualData, uint8_t* pExpectedData, size_t size) NN_NOEXCEPT
    {
        const size_t UnitSize = 1024;
        size_t offset = 0;
        size_t restSize = size;
        while (restSize > 0)
        {
            size_t compareSize = (restSize < UnitSize) ? restSize : UnitSize;
            if (std::memcmp(pExpectedData + offset, pActualData + offset, compareSize) != 0)
            {
                NN_LOG("expect:\n");
                LogData(pExpectedData + offset, compareSize, offset);
                NN_LOG("actual:\n");
                LogData(pActualData + offset, compareSize, offset);
                break;
            }
            offset += compareSize;
            restSize -= compareSize;
        }
    }

    bool CheckAccess(bool isSmall) NN_NOEXCEPT
    {
        const char* filePath = isSmall ? SmallFilePath : LargeFilePath;
        const size_t DataSize = isSmall ? 4 : BufferSize;

        int64_t firstfileSize = isSmall ? 0 : DataSize;
        nn::Result result = nn::fs::CreateFile(filePath, firstfileSize);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::CreateFile(%s) is failure(Module:%d, Description:%d)\n", filePath, result.GetModule(), result.GetDescription());
            return false;
        }

        nn::fs::FileHandle fileHandle;
        int writeOpenMode = isSmall ? (nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend) : nn::fs::OpenMode_Write;
        result = nn::fs::OpenFile(&fileHandle, filePath, writeOpenMode);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::OpenFile(%s) is failure(Module:%d, Description:%d)\n", filePath, result.GetModule(), result.GetDescription());
            return false;
        }
        srand(0);
        for (size_t i = 0; i < DataSize; i++)
        {
            g_SendBuffer[i] = static_cast<uint8_t>(rand());
        }
        result = nn::fs::WriteFile(fileHandle, 0, g_SendBuffer, DataSize, nn::fs::WriteOption());
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::WriteFile(%s) is failure(Module:%d, Description:%d)\n", filePath, result.GetModule(), result.GetDescription());
            nn::fs::CloseFile(fileHandle);
            return false;
        }
        result = nn::fs::FlushFile(fileHandle);
        nn::fs::CloseFile(fileHandle);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::FlushFile(%s) is failure(Module:%d, Description:%d)\n", filePath, result.GetModule(), result.GetDescription());
            return false;
        }

        result = nn::fs::OpenFile(&fileHandle,filePath, nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::OpenFile(%s) is failure(Module:%d, Description:%d)\n", filePath, result.GetModule(), result.GetDescription());
            return false;
        }
        std::memset(g_ReceiveBuffer, 0xA5, DataSize);
        nn::os::Tick tick = nn::os::GetSystemTick();
        result = nn::fs::ReadFile(fileHandle, 0, g_ReceiveBuffer, DataSize);
        nn::TimeSpan timeSpan = ConvertToTimeSpan(nn::os::GetSystemTick() - tick);
        if (!isSmall)
        {
            g_ReadSpeedKbPerSec = ((static_cast<int64_t>(DataSize) * 1000) / timeSpan.GetMilliSeconds()) / 1024;
        }
        nn::fs::CloseFile(fileHandle);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::ReadFile(%s) is failure(Module:%d, Description:%d)\n", filePath, result.GetModule(), result.GetDescription());
            return false;
        }

        if (std::memcmp(g_ReceiveBuffer, g_SendBuffer, DataSize) != 0)
        {
            NN_LOG("Verify Error %s\n", filePath);
            LogDifferentData(g_ReceiveBuffer, g_SendBuffer, DataSize);
            return false;
        }

        result = nn::fs::DeleteFile(filePath);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::DeleteFile(%s) is failure(Module:%d, Description:%d)\n", filePath, result.GetModule(), result.GetDescription());
            return false;
        }

        return true;
    }

    const char *GetSdCardSpeed() NN_NOEXCEPT
    {
        // SD カード挿入後にアクセスしておく必要がある
        nn::fs::SdCardSpeedMode speedMode;
        nn::Result result = nn::fs::GetSdCardSpeedMode(&speedMode);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::GetSdCardSpeedMode() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return nullptr;
        }
        switch (speedMode)
        {
        case nn::fs::SdCardSpeedMode_Identification:
            return "Identification";
        case nn::fs::SdCardSpeedMode_DefaultSpeed:
            return "Default";
        case nn::fs::SdCardSpeedMode_HighSpeed:
            return "High";
        case nn::fs::SdCardSpeedMode_Sdr12:
            return "SDR12";
        case nn::fs::SdCardSpeedMode_Sdr25:
            return "SDR25";
        case nn::fs::SdCardSpeedMode_Sdr50:
            return "SDR50";
        case nn::fs::SdCardSpeedMode_Sdr104:
            return "SDR104";
        case nn::fs::SdCardSpeedMode_Ddr50:
            return "DDR50";
        case nn::fs::SdCardSpeedMode_Unknown:
            return "Unknown";
        default:
            return "Unexpected";
        }
    }

    void DoTestSdCard() NN_NOEXCEPT
    {
        nn::os::Tick tick = nn::os::GetSystemTick();
        nn::Result result = nn::fs::FormatSdCard();
        nn::TimeSpan timeSpan = ConvertToTimeSpan(nn::os::GetSystemTick() - tick);
        if (result.IsSuccess())
        {
            result = nn::fs::MountSdCardForDebug(MountPath);
            if (result.IsSuccess())
            {
                bool isSuccess = CheckAccess(true);
                if (isSuccess)
                {
                    isSuccess = CheckAccess(false);
                    if (isSuccess)
                    {
                        int64_t totalSize = 0;
                        result = nn::fs::GetTotalSpaceSize(&totalSize, RootPath);
                        if (result.IsSuccess())
                        {
                            const char *speedMode = GetSdCardSpeed();
                            if (speedMode != nullptr)
                            {
                                NN_LOG("SpeedMode Speed[KB/s] TotalSize[B] Format[s]\n");
                                NN_LOG("%s %" PRId64 " %" PRId64 " %" PRId64 "\n", speedMode, g_ReadSpeedKbPerSec, totalSize, timeSpan.GetSeconds());
                            }
                        }
                        else
                        {
                            NN_LOG("nn::fs::GetTotalSpaceSize() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
                        }
                    }
                }
                nn::fs::Unmount(MountPath);
            }
            else
            {
                NN_LOG("nn::fs::MountSdCardForDebug() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            }
        }
        else
        {
            NN_LOG("nn::fs::FormatSdCard() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        }

        nn::fs::StorageErrorInfo storageErrorInfo;
        char logBuffer[256];
        size_t logSize = 0;
        result = nn::fs::GetAndClearSdCardErrorInfo(&storageErrorInfo, &logSize, logBuffer, sizeof(logBuffer));
        if (result.IsSuccess())
        {
            if (storageErrorInfo.numActivationFailures != 0)
            {
                NN_LOG("The number of activation failures: %u\n", storageErrorInfo.numActivationFailures);
            }
            if (storageErrorInfo.numActivationErrorCorrections != 0)
            {
                NN_LOG("The number of activation error corrections: %u\n", storageErrorInfo.numActivationErrorCorrections);
            }
            if (storageErrorInfo.numReadWriteFailures != 0)
            {
                NN_LOG("The number of read/write failuress: %u\n", storageErrorInfo.numReadWriteFailures);
            }
            if (storageErrorInfo.numReadWriteErrorCorrections != 0)
            {
                NN_LOG("The number of read/write error corrections: %u\n", storageErrorInfo.numReadWriteErrorCorrections);
            }
            if (logSize > 0)
            {
                NN_LOG("Log: %s\n", logBuffer);
            }
        }
        else
        {
            NN_LOG("nn::sdmmc::GetAndClearSdCardErrorInfo() is failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        }
    }
}

extern "C" void nnMain()
{
    nn::fs::SetEnabledAutoAbort(false);

    std::unique_ptr<nn::fs::IEventNotifier> sdCardDetectionEventNotifier;
    nn::Result result = nn::fs::OpenSdCardDetectionEventNotifier(&sdCardDetectionEventNotifier);
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::OpenSdCardDetectionEventNotifier() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }

    nn::os::SystemEventType sdCardDetectionEvent;
    result = sdCardDetectionEventNotifier->BindEvent(&sdCardDetectionEvent, nn::os::EventClearMode_ManualClear);
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::IEventNotifier->BindEvent() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }

    for (int i = 0; ; i++)
    {
        if (nn::fs::IsSdCardInserted())
        {
            NN_LOG("No.%03d\n", i);
            NN_LOG("Please wait...\n");
            DoTestSdCard();
            NN_LOG("Please remove SD Card...\n");
            while (true)
            {
                nn::os::WaitSystemEvent(&sdCardDetectionEvent);
                nn::os::ClearSystemEvent(&sdCardDetectionEvent);
                if (!nn::fs::IsSdCardInserted())
                {
                    break;
                }
                else
                {
                    NN_LOG("SD Card is still inserted.\n");
                }
            }
        }

        NN_LOG("Please insert SD Card...\n");
        while (true)
        {
            nn::os::WaitSystemEvent(&sdCardDetectionEvent);
            nn::os::ClearSystemEvent(&sdCardDetectionEvent);
            if (nn::fs::IsSdCardInserted())
            {
                break;
            }
            else
            {
                NN_LOG("SD Card is not inserted.\n");
            }
        }
    }
}
