﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>

namespace
{
    const size_t BufferSize = 4 * 1024;
    uint8_t g_SendBuffer[BufferSize];
    uint8_t g_ReceiveBuffer[BufferSize];

    bool CheckAccess() NN_NOEXCEPT
    {
        nn::Result result = nn::fs::CreateFile("sdcard:/data.bin", 0);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::CreateFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return false;
        }

        nn::fs::FileHandle fileHandle;
        result = nn::fs::OpenFile(&fileHandle, "sdcard:/data.bin", nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::OpenFile(OpenMode_AllowAppend) is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return false;
        }
        srand(0);
        for (size_t i = 0; i < BufferSize; i++)
        {
            g_SendBuffer[i] = static_cast<uint8_t>(rand());
        }
        result = nn::fs::WriteFile(fileHandle, 0, g_SendBuffer, BufferSize, nn::fs::WriteOption());
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::WriteFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            nn::fs::CloseFile(fileHandle);
            return false;
        }
        result = nn::fs::FlushFile(fileHandle);
        nn::fs::CloseFile(fileHandle);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::FlushFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return false;
        }

        result = nn::fs::OpenFile(&fileHandle, "sdcard:/data.bin", nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::OpenFile(OpenMode_Read) is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return false;
        }
        std::memset(g_ReceiveBuffer, 0xA5, BufferSize);
        result = nn::fs::ReadFile(fileHandle, 0, g_ReceiveBuffer, BufferSize);
        nn::fs::CloseFile(fileHandle);
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::ReadFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return false;
        }

        result = nn::fs::DeleteFile("sdcard:/data.bin");
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::DeleteFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return false;
        }

        if (std::memcmp(g_ReceiveBuffer, g_SendBuffer, BufferSize) != 0)
        {
            NN_LOG("Verify Error\n");
            return false;
        }

        return true;
    }
}

extern "C" void nnMain()
{
    std::unique_ptr<nn::fs::IEventNotifier> sdCardDetectionEventNotifier;
    nn::Result result = nn::fs::OpenSdCardDetectionEventNotifier(&sdCardDetectionEventNotifier);
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::OpenSdCardDetectionEventNotifier() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        NN_LOG("Test Failure\n");
        return;
    }

    nn::os::SystemEventType sdCardDetectionEvent;
    result = sdCardDetectionEventNotifier->BindEvent(&sdCardDetectionEvent, nn::os::EventClearMode_ManualClear);
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::IEventNotifier->BindEvent() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        NN_LOG("Test Failure\n");
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::DestroySystemEvent(&sdCardDetectionEvent);
    };

    bool isSdCardInserted = nn::fs::IsSdCardInserted();
    if (isSdCardInserted)
    {
        NN_LOG("SD Card is already inserted.\n");
    }
    else
    {
        NN_LOG("Please insert SD Card that suppports SDR104 and is formatted...\n");
        while (true)
        {
            nn::os::WaitSystemEvent(&sdCardDetectionEvent);
            nn::os::ClearSystemEvent(&sdCardDetectionEvent);
            if (nn::fs::IsSdCardInserted())
            {
                break;
            }
            else
            {
                NN_LOG("SD Card is not inserted.\n");
            }
        }
    }

    NN_LOG("Please wait...\n");

    // Initialize SD Card
    result = nn::fs::MountSdCardForDebug("sdcard");
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::MountSdCardForDebug() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        NN_LOG("Test Failure\n");
        return;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("sdcard");
    };

    // Check speed after nn::fs::MountSdCardXxxx()
    nn::fs::SdCardSpeedMode speedMode;
    result = nn::fs::GetSdCardSpeedMode(&speedMode);
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::GetSdCardSpeedMode() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        NN_LOG("Test Failure\n");
        return;
    }
    if (speedMode != nn::fs::SdCardSpeedMode_Sdr104)
    {
        NN_LOG("SD Card Speed Mode: %d\n", speedMode);
        NN_LOG("Test Failure\n");
        return;
    }

    bool ret = CheckAccess();
    if (!ret)
    {
        NN_LOG("Test Failure\n");
        return;
    }

    NN_LOG("Please remove SD Card...\n");
    while (true)
    {
        nn::os::WaitSystemEvent(&sdCardDetectionEvent);
        nn::os::ClearSystemEvent(&sdCardDetectionEvent);
        if (!nn::fs::IsSdCardInserted())
        {
            break;
        }
        else
        {
            NN_LOG("SD Card is still inserted.\n");
        }
    }

    NN_LOG("Test Success\n");
}
