﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/nn_Log.h>
#include <cinttypes>

namespace
{
    struct MmcCidInfo
    {
        uint8_t mid;
        uint64_t pnm;
        uint8_t prv;
    };

    #if 1   // SDEV, EDEV
        const MmcCidInfo MmcCidInfoList[] = {\
            { 0x11, 0x303634473330, 0x00 }, // Toshiba 64GB THGBMHG9C4LBAIR
        };
    #endif
    #if 0   // Prod
        const MmcCidInfo MmcCidInfoList[] = {\
            { 0x11, 0x303332473332, 0x00 }, // Toshiba 32GB THGBMHG8C2LBAIL
            { 0x15, 0x000000000000, 0x00 }, // Samsung 32GB (TBD)
        };
    #endif

    bool IsExpectedMmcCid(const uint8_t* pMmcCid)
    {
        uint8_t mid = nn::fs::GetMidFromMmcCid(pMmcCid);
        uint64_t pnm = nn::fs::GetPnmFromMmcCid(pMmcCid);
        uint8_t prv = nn::fs::GetPrvFromMmcCid(pMmcCid);

        for (int i = 0; i < (sizeof (MmcCidInfoList) / sizeof (MmcCidInfo)); i++)
        {
            if ((mid == MmcCidInfoList[i].mid) && (pnm == MmcCidInfoList[i].pnm) && (prv == MmcCidInfoList[i].prv))
            {
                return true;    // Hit!
            }
        }

        NN_LOG("MID: 0x%02x\n", mid);
        NN_LOG("PNM: 0x%" PRIx64 "\n", pnm);
        NN_LOG("PRV: 0x%02x\n", prv);
        return false;   // Unexpected CID
    }
}

extern "C" void nnMain()
{
    nn::fs::MmcSpeedMode speedMode;
    nn::Result result = nn::fs::GetMmcSpeedMode(&speedMode);
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::GetMmcSpeedMode() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        NN_LOG("Test Failure\n");
        return;
    }
    if (speedMode != nn::fs::MmcSpeedMode_Hs400)
    {
        NN_LOG("MMC Speed Mode: %d\n", speedMode);
        NN_LOG("Test Failure\n");
        return;
    }

    uint8_t cid[nn::fs::MmcCidSize];
    result = nn::fs::GetMmcCid(cid, sizeof(cid));
    if (result.IsFailure())
    {
        NN_LOG("nn::fs::GetMmcCid() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        NN_LOG("Test Failure\n");
        return;
    }
    bool isExpected = IsExpectedMmcCid(cid);
    if (!isExpected)
    {
        NN_LOG("Test Failure\n");
        return;
    }

    NN_LOG("Test Success\n");
}
