﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>

namespace
{
    void TestMmcConnection() NN_NOEXCEPT
    {
        // 本体保存メモリーの制御ポートは Firmware 起動時に Activate されるため、マウントなどは不要

        nn::fs::SdmmcSpeedMode sdmmcSpeedMode = nn::fs::SdmmcSpeedMode_Unknown;
        nn::fs::SdmmcBusWidth sdmmcBusWidth = nn::fs::SdmmcBusWidth_Unknown;
        nn::Result result = nn::fs::GetSdmmcConnectionStatus(&sdmmcSpeedMode, &sdmmcBusWidth, nn::fs::SdmmcPort_Mmc0);
        if (result.IsFailure())
        {
            // 本体保存メモリーの制御ポートの通信状態が異常
            NN_LOG("MMC Connection is abnormal(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return;
        }

        bool isNormal = true;
        if (sdmmcSpeedMode != nn::fs::SdmmcSpeedMode_MmcHs400)
        {
            // 本体保存メモリーの制御ポートの Speed Mode が HS400 ではない場合は異常
            NN_LOG("MMC Port is NOT HS400(nn::fs::SdmmcSpeedMode:%d)\n", sdmmcSpeedMode);
            isNormal = false;
        }
        if (sdmmcBusWidth != nn::fs::SdmmcBusWidth_8Bit)
        {
            // 本体保存メモリーの制御ポートのバス幅が 8 bit ではない場合は異常
            NN_LOG("MMC Port is NOT 8 bit(nn::fs::SdmmcBusWidth:%d)\n", sdmmcBusWidth);
            isNormal = false;
        }
        if (isNormal)
        {
            NN_LOG("MMC Connection is normal\n");
        }
    }

    void TestSdCardConnection() NN_NOEXCEPT
    {
        // SD カードの制御ポートは SD カード挿入状態でマウントしなければ Actiavte されない
        nn::Result result = nn::fs::MountSdCardForDebug("sdcard");
        if (result.IsFailure())
        {
            NN_LOG("nn::fs::MountSdCardForDebug() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return;
        }
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("sdcard");
        };

        nn::fs::SdmmcSpeedMode sdmmcSpeedMode = nn::fs::SdmmcSpeedMode_Unknown;
        nn::fs::SdmmcBusWidth sdmmcBusWidth = nn::fs::SdmmcBusWidth_Unknown;
        result = nn::fs::GetSdmmcConnectionStatus(&sdmmcSpeedMode, &sdmmcBusWidth, nn::fs::SdmmcPort_SdCard0);
        if (result.IsFailure())
        {
            // SD カードの制御ポートの通信状態が異常
            NN_LOG("SD Card Connection is abnormal(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return;
        }

        bool isNormal = true;
        switch (sdmmcSpeedMode)
        {
        case nn::fs::SdmmcSpeedMode_SdCardDefaultSpeed:
            NN_LOG("SD Card Port is Default Speed\n");  // 挿入した SD カードが High Speed 非対応ならば正常
            break;
        case nn::fs::SdmmcSpeedMode_SdCardHighSpeed:
            NN_LOG("SD Card Port is High Speed\n");     // 挿入した SD カードが UHS-I 非対応ならば正常
            break;
        case nn::fs::SdmmcSpeedMode_SdCardSdr50:
            NN_LOG("SD Card Port is SDR50\n");          // 挿入した SD カードが UHS-I 対応で SDR104 非対応ならば正常
            break;
        case nn::fs::SdmmcSpeedMode_SdCardSdr104:
            NN_LOG("SD Card Port is SDR104\n");         // 挿入した SD カードが SDR104 対応ならば正常
            break;
        default:
            NN_LOG("SD Card Port is Unexpected Speed Mode(nn::fs::SdmmcSpeedMode:%d)\n", sdmmcSpeedMode);   // 異常
            isNormal = false;
            break;
        }
        if (sdmmcBusWidth != nn::fs::SdmmcBusWidth_4Bit)
        {
            // SD カードの制御ポートのバス幅が 4 bit ではない場合は異常
            NN_LOG("SD Card Port is NOT 4 bit(nn::fs::SdmmcBusWidth:%d)\n", sdmmcBusWidth);
            isNormal = false;
        }
        if (isNormal)
        {
            NN_LOG("SD Card Connection is normal if speed mode is expected\n");
        }
    }

    void TestGcAsicConnection() NN_NOEXCEPT
    {
        // GC ASIC の制御ポートは Firmware 起動時に Activate されるため、マウントなどは不要、ゲームカード挿入も不要。
        // ただし、本体起動直後は Activate されていない期間があるため、このテストアプリ自体をゲームカードから起動する方が確実。

        nn::fs::SdmmcSpeedMode sdmmcSpeedMode = nn::fs::SdmmcSpeedMode_Unknown;
        nn::fs::SdmmcBusWidth sdmmcBusWidth = nn::fs::SdmmcBusWidth_Unknown;
        nn::Result result = nn::fs::GetSdmmcConnectionStatus(&sdmmcSpeedMode, &sdmmcBusWidth, nn::fs::SdmmcPort_GcAsic0);
        if (result.IsFailure())
        {
            // GC ASIC の制御ポートの通信状態が異常
            NN_LOG("GC ASIC Connection is abnormal(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return;
        }

        bool isNormal = true;
        if (sdmmcSpeedMode != nn::fs::SdmmcSpeedMode_GcAsicSpeed)
        {
            // GC ASIC の制御ポートの Speed Mode が GC ASIC 用ではない場合は異常
            NN_LOG("GC ASIC Port is NOT HS400(nn::fs::SdmmcSpeedMode:%d)\n", sdmmcSpeedMode);
            isNormal = false;
        }
        if (sdmmcBusWidth != nn::fs::SdmmcBusWidth_8Bit)
        {
            // GC ASIC の制御ポートのバス幅が 8 bit ではない場合は異常
            NN_LOG("GC ASIC Port is NOT 8 bit(nn::fs::SdmmcBusWidth:%d)\n", sdmmcBusWidth);
            isNormal = false;
        }
        if (isNormal)
        {
            NN_LOG("GC ASIC Connection is normal\n");
        }
    }
}

extern "C" void nnMain()
{
    TestMmcConnection();
    TestSdCardConnection();
    TestGcAsicConnection();
    NN_LOG("Done.\n");
}
