﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sdmmc/sdmmc_Mmc.h>
#ifdef NN_SDMMC_TEST_SMMU_ENABLE
    #include <nn/sdmmc/sdmmc_DeviceVirtualAddress.h>
#endif
#include <nn/os/os_MemoryHeap.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/dd.h>
#include <nn/fs/fs_SdmmcControl.h>
#include <cstring>
#include "SdmmcCommon.h"

// 注意：初期プロセスの FS の sMMU 設定を無効にした環境でしか動作しません。

namespace
{
    const nn::sdmmc::Port TestPort = nn::sdmmc::Port_Mmc0;
    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        const nn::dd::DeviceName DeviceName = nn::dd::DeviceName::DeviceName_Sdmmc4a;
    #endif

    //const nn::sdmmc::MmcPartition TestMmcPartition = nn::sdmmc::MmcPartition_UserData;
    const nn::sdmmc::MmcPartition TestMmcPartition = nn::sdmmc::MmcPartition_BootPartition1;
    //const nn::sdmmc::MmcPartition TestMmcPartition = nn::sdmmc::MmcPartition_BootPartition2;

    #define SDMMC_DETAIL_CEILING(value, unit)   ((((value) + (unit) - 1) / (unit)) * (unit))
    #define SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(value)    SDMMC_DETAIL_CEILING((value), nn::dd::DeviceAddressSpaceMemoryRegionAlignment)
    NN_DD_ALIGNAS_DEVICE_ADDRESS_SPACE_MEMORY uint8_t g_WorkBuffer[SDMMC_DETAIL_CEILING_FOR_DEVICE_ADDRESS_SPACE(nn::sdmmc::MmcWorkBufferSize)];

    //const size_t ReadSize = 4 * 1024 * 1024;
    const size_t ReadSize = 4 * 1024;
    const uint32_t NumReadSectors = ReadSize / nn::sdmmc::SectorSize;
    const size_t DataBufferSize = nn::util::align_up(ReadSize, nn::os::MemoryBlockUnitSize);
    void* g_pDataBuffer;

    void LogData(void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pData);
        if (dataSize == 0)
        {
            return;
        }
        uint32_t address = 0;
        uint8_t* pCurrentData =  reinterpret_cast<uint8_t*>(pData);
        uint8_t* pDataEnd = pCurrentData + dataSize;
        while (true)
        {
            NN_LOG("%08X: ", address);
            for (uint32_t i = 0; i < 0x10; i++)
            {
                NN_LOG("%02X ", *pCurrentData);
                pCurrentData++;
                if (pCurrentData >= pDataEnd)
                {
                    NN_LOG("\n");
                    return;
                }
            }
            NN_LOG("\n");
            address += 0x10;
        }
    }
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");
}

extern "C" void nnMain()
{
    // メモリヒープから malloc で使用するメモリ領域を確保し、設定する
    uintptr_t heapPtrForMalloc;
    const size_t HeapSizeForMalloc = 256 * 1024 * 1024;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&heapPtrForMalloc, HeapSizeForMalloc));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(heapPtrForMalloc), HeapSizeForMalloc);

    // fs プロセスの SDMMC 制御を停止する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::SuspendSdmmcControl());

    // クロック制御は pcv 経由で行う
    nn::sdmmc::SwitchToPcvClockResetControl();

    nn::sdmmc::Initialize(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceAddressSpaceType das;
        SetupDeviceAddressSpace(&das, DeviceName);
        nn::dd::DeviceVirtualAddress deviceVirtualAddressOffset = 0;

        nn::dd::DeviceAddressSpaceMapInfo info;
        nn::dd::DeviceVirtualAddress workBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, &info,
            reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), deviceVirtualAddressOffset);
        deviceVirtualAddressOffset = workBufferDeviceVirtualAddress + sizeof(g_WorkBuffer);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
    #endif
    nn::sdmmc::SetMmcWorkBuffer(TestPort, g_WorkBuffer, sizeof(g_WorkBuffer));

    uintptr_t dataBufferAddress;
    nn::Result result = nn::os::AllocateMemoryBlock(&dataBufferAddress, DataBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot allocate memory.");
    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::dd::DeviceVirtualAddress dataBufferDeviceVirtualAddress = MapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, deviceVirtualAddressOffset);
        nn::sdmmc::RegisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif
    g_pDataBuffer = reinterpret_cast<uint8_t*>(dataBufferAddress);

    result = nn::sdmmc::Activate(TestPort);
    if (result.IsSuccess())
    {
        // バス状態のダンプ
        LogBusStatus(TestPort);

        // 容量取得
        uint32_t numSectors = 0;
        result = nn::sdmmc::GetDeviceMemoryCapacity(&numSectors, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("Memory Capacity: %u (MB)\n", numSectors / 2 / 1024);
        result = nn::sdmmc::GetMmcBootPartitionCapacity(&numSectors, TestPort);
        NN_ABORT_UNLESS(result.IsSuccess());
        NN_LOG("MMC Boot Partition Capacity: %u (KB)\n", numSectors / 2);
        uint32_t readNumSectors = (numSectors < NumReadSectors) ? numSectors : NumReadSectors;

        // TestMmcPartition を選択
        result = nn::sdmmc::SelectMmcPartition(TestPort, TestMmcPartition);
        NN_ABORT_UNLESS(result.IsSuccess());
        switch (TestMmcPartition)
        {
        case nn::sdmmc::MmcPartition_UserData:
            NN_LOG("MMC User Data:\n");
            break;
        case nn::sdmmc::MmcPartition_BootPartition1:
            NN_LOG("Boot Partition 1:\n");
            break;
        case nn::sdmmc::MmcPartition_BootPartition2:
            NN_LOG("Boot Partition 2:\n");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // TestMmcPartition を先頭からリード
        result = nn::sdmmc::Read(g_pDataBuffer, DataBufferSize, TestPort, 0, readNumSectors);
        NN_ABORT_UNLESS(result.IsSuccess());
        LogData(g_pDataBuffer, readNumSectors * nn::sdmmc::SectorSize);

        // User Data を選択（安全のため、元に戻す）
        if (TestMmcPartition != nn::sdmmc::MmcPartition_UserData)
        {
            result = nn::sdmmc::SelectMmcPartition(TestPort, nn::sdmmc::MmcPartition_UserData);
            NN_ABORT_UNLESS(result.IsSuccess());
        }
    }
    else
    {
        NN_LOG("nn::sdmmc::Activate is failure. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
    }
    nn::sdmmc::Deactivate(TestPort);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&das, dataBufferAddress, DataBufferSize, dataBufferDeviceVirtualAddress);
    #endif
    nn::os::FreeMemoryBlock(dataBufferAddress, DataBufferSize);

    #ifdef NN_SDMMC_TEST_SMMU_ENABLE
        nn::sdmmc::UnregisterDeviceVirtualAddress(TestPort, reinterpret_cast<uintptr_t>(g_WorkBuffer), sizeof(g_WorkBuffer), workBufferDeviceVirtualAddress);
        UnmapDeviceAddressSpaceAligned(&info);

        CleanDeviceAddressSpace(&das, DeviceName);
    #endif

    nn::sdmmc::Finalize(TestPort);

    // fs プロセスの SDMMC 制御を再開する
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ResumeSdmmcControl());

    nn::os::FreeMemoryBlock(heapPtrForMalloc, HeapSizeForMalloc);
}
