﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new> // for placement new

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/util/util_TypedStorage.h>

#include "lsm6ds3.h"
#include "lsm6ds3_Debug.h"
#include "lsm6ds3_Register.h"
#include "lsm6ds3_Util.h"

namespace {

const nnt::sasbus::lsm6ds3::AccelerometerFsr DefaultAccelerometerFsr = nnt::sasbus::lsm6ds3::AccelerometerFsr::AccelerometerFsr_2G;
const nnt::sasbus::lsm6ds3::AccelerometerOdr DefaultAccelerometerOdr = nnt::sasbus::lsm6ds3::AccelerometerOdr::AccelerometerOdr_1660Hz;

const nnt::sasbus::lsm6ds3::GyroscopeFsr DefaultGyroscopeFsr = nnt::sasbus::lsm6ds3::GyroscopeFsr::GyroscopeFsr_245dps;
const nnt::sasbus::lsm6ds3::GyroscopeOdr DefaultGyroscopeOdr = nnt::sasbus::lsm6ds3::GyroscopeOdr::GyroscopeOdr_208Hz;

const uint8_t SasbusReceiveMask = 0x80; // [Ref1] 6.2.1 spi Read

}

namespace nnt {
namespace sasbus {
namespace lsm6ds3 {

void Initialize(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    // デバイスのリセット
    ResetDevice(pSession);

    // Initialize 時点でセンサーは ON
    StartAccelerometer(pSession);
    StartGyroscope(pSession);

    uint8_t ctrl3C = 0x00;
    nn::sasbus::Receive(&ctrl3C, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl3C) | SasbusReceiveMask);
    detail::SetBitOn8(&ctrl3C, detail::Register::Ctrl3CIfInc::Mask);    // 多バイトアクセスを ON に
    detail::SetBitOn8(&ctrl3C, detail::Register::Ctrl3CBdu::Mask);      // Block Data Update を ON に
    nn::sasbus::Send(pSession, &ctrl3C, 1, static_cast<uint8_t>(detail::Address::Ctrl3C));

    // DRDY_MASK を ON に
    uint8_t ctrl4C = 0x00;
    nn::sasbus::Receive(&ctrl4C, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl4C) | SasbusReceiveMask);
    detail::SetBitOn8(&ctrl4C, detail::Register::Ctrl4CDrdyMask::Mask);
    nn::sasbus::Send(pSession, &ctrl4C, 1, static_cast<uint8_t>(detail::Address::Ctrl4C));

    SetNoiseReductionFilter(pSession, false);
}

void Finalize(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    // Finalize 時点でセンサーは OFF
    StopAccelerometer(pSession);
    StopGyroscope(pSession);
}

void ResetDevice(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    uint8_t  ctrl3C = 0x00;

    nn::sasbus::Receive(&ctrl3C, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl3C) | SasbusReceiveMask);

    detail::SetBitOn8(&ctrl3C, detail::Register::Ctrl3CSwReset::Mask);
    nn::sasbus::Send(pSession, &ctrl3C, 1, static_cast<uint8_t>(detail::Address::Ctrl3C));
}

void GetState(nn::sasbus::Session* pSession, SixAxisSensorCountState* pOutState) NN_NOEXCEPT
{
    const size_t  DataBytes = 12;
    uint8_t       buffer[DataBytes];
    std::memset(buffer, 0, sizeof(buffer));

    nn::sasbus::Receive(buffer, pSession, DataBytes, static_cast<uint8_t>(detail:: Address::OutXLG) | SasbusReceiveMask);

    pOutState->angularVelocity.x = (buffer[1] << 8) | buffer[0];
    pOutState->angularVelocity.y = (buffer[3] << 8) | buffer[2];
    pOutState->angularVelocity.z = (buffer[5] << 8) | buffer[4];

    pOutState->acceleration.x = (buffer[7] << 8) | buffer[6];
    pOutState->acceleration.y = (buffer[9] << 8) | buffer[8];
    pOutState->acceleration.z = (buffer[11] << 8) | buffer[10];

    detail::DebugDumpSensorCountState(pOutState);
}

void GetStateForPeriodicReceiveMode(nn::sasbus::Session* pSession, SixAxisSensorCountState* pOutState) NN_NOEXCEPT
{
    const size_t                                 DataBytes = 12;
    nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType       buffer[DataBytes];
    std::memset(buffer, 0, sizeof(buffer));

    nn::sasbus::GetPeriodicReceiveModeData<nn::sasbus::Lsm6ds3PeriodicReceiveModeDataType, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataCountMax>(buffer, pSession, 1);

    pOutState->angularVelocity.x = (buffer->data[1] << 8) | buffer->data[0];
    pOutState->angularVelocity.y = (buffer->data[3] << 8) | buffer->data[2];
    pOutState->angularVelocity.z = (buffer->data[5] << 8) | buffer->data[4];

    pOutState->acceleration.x = (buffer->data[7] << 8) | buffer->data[6];
    pOutState->acceleration.y = (buffer->data[9] << 8) | buffer->data[8];
    pOutState->acceleration.z = (buffer->data[11] << 8) | buffer->data[10];

    detail::DebugDumpSensorCountState(pOutState);
}

void StartPeriodicReceiveMode(nn::sasbus::Session* pSession, nn::TimeSpan interval, char* receiveBuffer, size_t receiveBufferLength) NN_NOEXCEPT
{
    nn::sasbus::StartPeriodicReceiveMode(pSession, nn::sasbus::Lsm6ds3PeriodicReceiveModeDataSize,
                                         static_cast<uint8_t>(detail::Address::OutXLG) | SasbusReceiveMask, interval, receiveBuffer, receiveBufferLength);
}

void StopPeriodicReceiveMode(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    nn::sasbus::StopPeriodicReceiveMode(pSession);
}

void SetNoiseReductionFilter(nn::sasbus::Session* pSession, bool enable) NN_NOEXCEPT
{
    // Acc のアンチエイリアシングフィルタ係数を設定
    uint8_t ctrl1Xl = 0x00;
    detail::AccAntiAliasingFilterBandwidthRegValue filterRegValue =
        enable ? detail::AccAntiAliasingFilterBandwidthRegValue::AccAntiAliasingFilterBandwidthRegValue_100Hz :
        detail::AccAntiAliasingFilterBandwidthRegValue::AccAntiAliasingFilterBandwidthRegValue_400Hz;

    nn::sasbus::Receive(&ctrl1Xl, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl1Xl) | SasbusReceiveMask);
    detail::Modify8(&ctrl1Xl,
        static_cast<uint8_t>(filterRegValue) << detail::Register::Ctrl1XlBw::Pos,
        detail::Register::Ctrl1XlBw::Mask);
    nn::sasbus::Send(pSession, &ctrl1Xl, 1, static_cast<uint8_t>(detail::Address::Ctrl1Xl));

    // Acc の LPF 設定
    uint8_t ctrl8Xl = 0x00;
    nn::sasbus::Receive(&ctrl8Xl, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl8Xl) | SasbusReceiveMask);
    detail::Modify8(&ctrl8Xl,
        enable << detail::Register::Ctrl8XlHpSlopeXlEn::Pos,
        detail::Register::Ctrl8XlHpSlopeXlEn::Mask);
    detail::Modify8(&ctrl8Xl,
        enable << detail::Register::Ctrl8XlLpf2XlEn::Pos,
        detail::Register::Ctrl8XlLpf2XlEn::Mask);
    nn::sasbus::Send(pSession, &ctrl8Xl, 1, static_cast<uint8_t>(detail::Address::Ctrl8Xl));

    // フィルタ設定
    uint8_t ctrl10C = 0x00;
    nn::sasbus::Receive(&ctrl10C, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl10C) | SasbusReceiveMask);
    detail::Modify8(&ctrl10C,
        enable << detail::Register::Ctrl10CFuncEn::Pos,
        detail::Register::Ctrl10CFuncEn::Mask);
    nn::sasbus::Send(pSession, &ctrl10C, 1, static_cast<uint8_t>(detail::Address::Ctrl10C));
}

// Acceleration
void StartAccelerometer(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    // Accelerometer を Activate する
    uint8_t ctrl9Xl = 0x00;
    nn::sasbus::Receive(&ctrl9Xl, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl9Xl) | SasbusReceiveMask);
    detail::SetBitOn8(&ctrl9Xl, detail::Register::Ctrl9XlXen::Mask);
    detail::SetBitOn8(&ctrl9Xl, detail::Register::Ctrl9XlYen::Mask);
    detail::SetBitOn8(&ctrl9Xl, detail::Register::Ctrl9XlZen::Mask);
    nn::sasbus::Send(pSession, &ctrl9Xl, 1, static_cast<uint8_t>(detail::Address::Ctrl9Xl));

    // ODR,FSR を設定する
    SetAccelerometerFsr(pSession, DefaultAccelerometerFsr);
    SetAccelerometerOdr(pSession, DefaultAccelerometerOdr);
}

void StopAccelerometer(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    // Accelerometer を Deactivate する
    uint8_t ctrl9Xl = 0x00;
    nn::sasbus::Receive(&ctrl9Xl, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl9Xl) | SasbusReceiveMask);
    detail::SetBitOff8(&ctrl9Xl, detail::Register::Ctrl9XlXen::Mask);
    detail::SetBitOff8(&ctrl9Xl, detail::Register::Ctrl9XlYen::Mask);
    detail::SetBitOff8(&ctrl9Xl, detail::Register::Ctrl9XlZen::Mask);
    nn::sasbus::Send(pSession, &ctrl9Xl, 1, static_cast<uint8_t>(detail::Address::Ctrl9Xl));

    // ODR を設定する
    SetAccelerometerOdr(pSession, AccelerometerOdr::AccelerometerOdr_0Hz);
}

void GetAccelerometerFsr(nn::sasbus::Session* pSession, AccelerometerFsr* pOutAccelerometerFsr) NN_NOEXCEPT
{
    uint8_t ctrl1Xl = 0x00;
    nn::sasbus::Receive(&ctrl1Xl, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl1Xl) | SasbusReceiveMask);

    detail::AccFsrRegValue reg = static_cast<detail::AccFsrRegValue>((ctrl1Xl & detail::Register::Ctrl1XlFs::Mask) >> detail::Register::Ctrl1XlFs::Pos);
    *pOutAccelerometerFsr = ConvertToAccelerometerFsr(reg);
}

void SetAccelerometerFsr(nn::sasbus::Session* pSession, AccelerometerFsr accelerometerFsr) NN_NOEXCEPT
{

    uint8_t ctrl1Xl = 0x00;
    nn::sasbus::Receive(&ctrl1Xl, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl1Xl) | SasbusReceiveMask);
    detail::Modify8(&ctrl1Xl,
        static_cast<uint8_t>(detail::ConvertToAccFsrRegValue(accelerometerFsr)) << detail::Register::Ctrl1XlFs::Pos,
        detail::Register::Ctrl1XlFs::Mask);
    nn::sasbus::Send(pSession, &ctrl1Xl, 1, static_cast<uint8_t>(detail::Address::Ctrl1Xl));
}

void GetAccelerometerOdr(nn::sasbus::Session* pSession, AccelerometerOdr* pOutAccelerometerOdr) NN_NOEXCEPT
{
    uint8_t ctrl1Xl = 0x00;
    nn::sasbus::Receive(&ctrl1Xl, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl1Xl) | SasbusReceiveMask);
    detail::AccOdrRegValue reg = static_cast<detail::AccOdrRegValue>((ctrl1Xl & detail::Register::Ctrl1XlOdr::Mask) >> detail::Register::Ctrl1XlOdr::Pos);
    *pOutAccelerometerOdr = ConvertToAccelerometerOdr(reg);
}

void SetAccelerometerOdr(nn::sasbus::Session* pSession, AccelerometerOdr accelerometerOdr) NN_NOEXCEPT
{
    uint8_t ctrl1Xl = 0x00;
    nn::sasbus::Receive(&ctrl1Xl, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl1Xl) | SasbusReceiveMask);
    detail::Modify8(&ctrl1Xl,
        static_cast<uint8_t>(detail::ConvertToAccOdrRegValue(accelerometerOdr)) << detail::Register::Ctrl1XlOdr::Pos,
        detail::Register::Ctrl1XlOdr::Mask);
    nn::sasbus::Send(pSession, &ctrl1Xl, 1, static_cast<uint8_t>(detail::Address::Ctrl1Xl));
}

// Gyroscope
void StartGyroscope(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    // Gyroscope を Activate する
    uint8_t ctrl10C = 0x00;
    nn::sasbus::Receive(&ctrl10C, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl10C) | SasbusReceiveMask);
    detail::SetBitOn8(&ctrl10C, detail::Register::Ctrl10CXen::Mask);
    detail::SetBitOn8(&ctrl10C, detail::Register::Ctrl10CYen::Mask);
    detail::SetBitOn8(&ctrl10C, detail::Register::Ctrl10CZen::Mask);
    nn::sasbus::Send(pSession, &ctrl10C, 1, static_cast<uint8_t>(detail::Address::Ctrl10C));

    // ODR,FSR を設定する
    SetGyroscopeOdr(pSession, DefaultGyroscopeOdr);
    SetGyroscopeFsr(pSession, DefaultGyroscopeFsr);
}

void StopGyroscope(nn::sasbus::Session* pSession) NN_NOEXCEPT
{
    // Gyroscope を Deactivate する
    uint8_t ctrl10C = 0x00;
    nn::sasbus::Receive(&ctrl10C, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl10C) | SasbusReceiveMask);
    detail::SetBitOff8(&ctrl10C, detail::Register::Ctrl10CXen::Mask);
    detail::SetBitOff8(&ctrl10C, detail::Register::Ctrl10CYen::Mask);
    detail::SetBitOff8(&ctrl10C, detail::Register::Ctrl10CZen::Mask);
    nn::sasbus::Send(pSession, &ctrl10C, 1, static_cast<uint8_t>(detail::Address::Ctrl10C));

    // ODR を設定する
    SetGyroscopeOdr(pSession, GyroscopeOdr::GyroscopeOdr_0Hz);
}

void GetGyroscopeFsr(nn::sasbus::Session* pSession, GyroscopeFsr* pOutGyroscopeFsr) NN_NOEXCEPT
{
    uint8_t ctrl2G = 0x00;
    nn::sasbus::Receive(&ctrl2G, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl2G) | SasbusReceiveMask);

    detail::GyroFsrRegValue reg = static_cast<detail::GyroFsrRegValue>((ctrl2G & detail::Register::Ctrl2GFs::Mask) >> detail::Register::Ctrl2GFs::Pos);
    *pOutGyroscopeFsr = ConvertToGyroscopeFsr(reg);
}

void SetGyroscopeFsr(nn::sasbus::Session* pSession, GyroscopeFsr gyroscopeFsr) NN_NOEXCEPT
{
    uint8_t ctrl2G = 0x00;
    nn::sasbus::Receive(&ctrl2G, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl2G) | SasbusReceiveMask);
    detail::Modify8(&ctrl2G,
        static_cast<uint8_t>(detail::ConvertToGyroFsrRegValue(gyroscopeFsr)) << detail::Register::Ctrl2GFs::Pos,
        detail::Register::Ctrl2GFs::Mask);
    nn::sasbus::Send(pSession, &ctrl2G, 1, static_cast<uint8_t>(detail::Address::Ctrl2G));
}

void GetGyroscopeOdr(nn::sasbus::Session* pSession, GyroscopeOdr* pOutGyroscopeOdr) NN_NOEXCEPT
{
    uint8_t ctrl2G = 0x00;
    nn::sasbus::Receive(&ctrl2G, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl2G) | SasbusReceiveMask);
    detail::GyroOdrRegValue reg = static_cast<detail::GyroOdrRegValue>((ctrl2G & detail::Register::Ctrl2GOdr::Mask) >> detail::Register::Ctrl2GOdr::Pos);
    *pOutGyroscopeOdr = ConvertToGyroscopeOdr(reg);
}

void SetGyroscopeOdr(nn::sasbus::Session* pSession, GyroscopeOdr gyroscopeOdr) NN_NOEXCEPT
{
    uint8_t ctrl2G = 0x00;
    nn::sasbus::Receive(&ctrl2G, pSession, 1, static_cast<uint8_t>(detail::Address::Ctrl2G) | SasbusReceiveMask);
    detail::Modify8(&ctrl2G,
        static_cast<uint8_t>(detail::ConvertToGyroOdrRegValue(gyroscopeOdr)) << detail::Register::Ctrl2GOdr::Pos,
        detail::Register::Ctrl2GOdr::Mask);
    nn::sasbus::Send(pSession, &ctrl2G, 1, static_cast<uint8_t>(detail::Address::Ctrl2G));
}

} // lsm6ds3
} // sasbus
} // nnt
