﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <tuple>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nnt.h>

#include "fileutils.h"
#include "textutils.h"

#include "executils.h"

using namespace std;

#define ENABLE_ALL_CIPHER_TEST
#define ENABLE_GCM_CIPHER_TEST


//  External declaration of testMain used by existing NSS test code
//  so we can call it from the test framework.
extern "C" int blapitestMain(int argc, char **argv);
extern "C" int pk11gcmtestMain(int argc, char **argv);

//  Pick up NSPR's snprintf to avoid conditionalizing our code
extern "C" unsigned long PR_snprintf(char *out, unsigned long outlen, const char *fmt, ...);

#define MOUNT_NAME                      "appsave"
#define TXT_DELIM                       " \t"
#define NSS_QA_DIR                      "/tests"
#define NSS_CIPHER_DIR                  "/test_results/cipher"
#define NSS_CIPHER_TXT                  MOUNT_NAME ":" NSS_QA_DIR "/cipher/cipher.txt"
#define NSS_CIPHER_TEST_DIR             "/cmd/bltest"
#define NSS_GCM_TXT                     MOUNT_NAME ":" NSS_QA_DIR "/cipher/gcm.txt"
#define NSS_GCM_TEST_DIR                "/cmd/pk11gcmtest/tests/"

#define SIGLOHOST_OPTION                "-sighost"

static const char                       *MOUNT_POINT = MOUNT_NAME;
static const char                       *HOST_MOUNT_PATH = "C:\\WINDOWS\\TEMP";
const char                              *hostRootPath;

namespace nnt
{

namespace nss
{


typedef enum {
    MAX_LINE_LENGTH     = 256,
    INOUT_OFF_STR_LEN   = 3,
} SizeOrLengthConstants;


#ifdef ENABLE_ALL_CIPHER_TEST

typedef enum {
    CIPHER_TEST_ARGV_PROG_NAME_INDEX      = 0,
    CIPHER_TEST_ARGV_SELF_TEST_INDEX      = 1,
    CIPHER_TEST_ARGV_MODE_INDEX           = 2,
    CIPHER_TEST_ARGV_PARM1_INDEX          = 3,
    CIPHER_TEST_ARGV_PARM2_INDEX          = 4,
    CIPHER_TEST_ARGV_DIR_FLAG_INDEX       = 5,
    CIPHER_TEST_ARGV_DIR_INDEX            = 6,
    CIPHER_TEST_ARGV_IN_OFFSET_INDEX      = 7,
    CIPHER_TEST_ARGV_IN_OFFSET_VAL_INDEX  = 8,
    CIPHER_TEST_ARGV_OUT_OFFSET_INDEX     = 9,
    CIPHER_TEST_ARGV_OUT_OFFSET_VAL_INDEX = 10,

    NUM_INOUT_OFFSET                      = 8,
} CipherTestArgvConstants;

typedef enum {
    CIPHER_TEST_FIELD_RESULT_INDEX        = 0,
    CIPHER_TEST_FIELD_PARM_INDEX          = 1,
    CIPHER_TEST_FIELD_NAME_INDEX          = 2,
    CIPHER_TEST_FIELD_COUNT               = CIPHER_TEST_FIELD_NAME_INDEX + 1,
} CipherTestFieldConstants;


TEST(DataOnHostTest, AllCiphers)
{
    nn::fs::FileHandle          fh;
    int64_t                     len;
    size_t                      offset;
    char                        lineBuf[MAX_LINE_LENGTH];
    int                         testArgc;
    char                        inOffBuf[INOUT_OFF_STR_LEN];
    char                        outOffBuf[INOUT_OFF_STR_LEN];
    char                        *testArgv[MAX_ARGS];

    // Params for 'testArgv' because they cann't be 'const*'
    char                        paramProgName[]  = "bltest";
    char                        paramSelfTest[]  = "-T";
    char                        paramMode[]      = "-m";
    char                        paramDirFlag[]   = "-d";
    char                        paramDir[]       = NSS_CIPHER_TEST_DIR;
    char                        paramInOffset[]  = "-1";
    char                        paramOutOffset[] = "-2";

    //  Open the ciphers.txt file for input processing
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&fh,
                                               NSS_CIPHER_TXT,
                                               nn::fs::OpenMode_Read));

    //  Prep the test args array and count.  NULL values are filled
    //  in the looper for each iteration of the test.
    testArgc = CIPHER_TEST_ARGV_OUT_OFFSET_VAL_INDEX + 1;

    testArgv[CIPHER_TEST_ARGV_PROG_NAME_INDEX]      = paramProgName;
    testArgv[CIPHER_TEST_ARGV_SELF_TEST_INDEX]      = paramSelfTest;
    testArgv[CIPHER_TEST_ARGV_MODE_INDEX]           = paramMode;
    testArgv[CIPHER_TEST_ARGV_PARM1_INDEX]          = NULL;
    testArgv[CIPHER_TEST_ARGV_PARM2_INDEX]          = NULL;
    testArgv[CIPHER_TEST_ARGV_DIR_FLAG_INDEX]       = paramDirFlag;
    testArgv[CIPHER_TEST_ARGV_DIR_INDEX]            = paramDir;
    testArgv[CIPHER_TEST_ARGV_IN_OFFSET_INDEX]      = paramInOffset;
    testArgv[CIPHER_TEST_ARGV_IN_OFFSET_VAL_INDEX]  = inOffBuf;
    testArgv[CIPHER_TEST_ARGV_OUT_OFFSET_INDEX]     = paramOutOffset;
    testArgv[CIPHER_TEST_ARGV_OUT_OFFSET_VAL_INDEX] = outOffBuf;

    //  Read each line and parse out parameters (see cipher.sh)
    len = 0;
    offset = 0;
    for (len = helperutils::ReadLine(fh, offset, lineBuf, MAX_LINE_LENGTH - 1);
         len > 0;
         len = helperutils::ReadLine(fh, offset, lineBuf, MAX_LINE_LENGTH - 1))
    {
        char                    *fields[CIPHER_TEST_FIELD_COUNT];
        char                    *parm2;
        int                     expResultVal;

        lineBuf[len] = '\0';
        offset += len;

        if (helperutils::ExtractFields(lineBuf,
                                       CIPHER_TEST_FIELD_COUNT,
                                       fields,
                                       TXT_DELIM) != 0) {
            continue;
        }

        //  The raw parameters needs a little manipulation.  The input file
        //  jams them all together to pull extracted together, but there
        //  are really 2 parameters in there.
        parm2 = strstr(fields[CIPHER_TEST_FIELD_PARM_INDEX], "_-");
        if (parm2 == NULL)
        {
            //  malformed, skip it.
            continue;
        }

        //  Terminate the previous parameter so both can be passed
        *parm2 = 0;
        parm2++;

        //  Now that we have all the arguments, setup argv
        testArgv[CIPHER_TEST_ARGV_PARM1_INDEX] =
            fields[CIPHER_TEST_FIELD_PARM_INDEX];
        testArgv[CIPHER_TEST_ARGV_PARM2_INDEX] = parm2;
        expResultVal               =
            atoi(fields[CIPHER_TEST_FIELD_RESULT_INDEX]);

        for (int inOff = 0; inOff < NUM_INOUT_OFFSET; inOff++)
        {
            for (int outOff = 0; outOff < NUM_INOUT_OFFSET; outOff++)
            {
                int             testResult;

                PR_snprintf(inOffBuf, INOUT_OFF_STR_LEN, "%d", inOff);
                PR_snprintf(outOffBuf, INOUT_OFF_STR_LEN, "%d", outOff);

                ::helperutils::dumpCallInfo("bltestapiMain", testArgc, testArgv);

                testResult = blapitestMain(testArgc, testArgv);
                EXPECT_EQ(expResultVal, testResult);
            }
        }
    }

    nn::fs::CloseFile(fh);
}
#endif    //  ENABLE_ALL_CIPHER_TEST


#ifdef ENABLE_GCM_CIPHER_TEST

typedef enum {
    GCM_TEST_ARGV_PROG_NAME_INDEX    = 0,
    GCM_TEST_ARGV_AES_OPT_INDEX      = 1,
    GCM_TEST_ARGV_KAT_OPT_INDEX      = 2,
    GCM_TEST_ARGV_GCM_OPT_INDEX      = 3,
    GCM_TEST_ARGV_INPUT_FILE_INDEX   = 4,
} GcmTestArgvConstants;

typedef enum {
    GCM_TEST_FIELD_RESULT_INDEX      = 0,
    GCM_TEST_FIELD_INPUT_INDEX       = 1,
    GCM_TEST_FIELD_NAME_INDEX        = 2,
    GCM_TEST_FIELD_COUNT             = GCM_TEST_FIELD_NAME_INDEX + 1,
} GcmTestFieldConstants;

TEST(DataOnHostTest, GcmTest)
{
    nn::fs::FileHandle          fh;
    int64_t                     len;
    size_t                      offset;
    char                        lineBuf[MAX_LINE_LENGTH];
    int                         testArgc;
    char                        scratch[MAX_LINE_LENGTH];
    char                        *testArgv[MAX_ARGS];

    // Params for 'testArgv' because they cann't be 'const*'
    char                        paramProgName[] = "pk11gcmtest";
    char                        paramAesOpt[]   = "aes";
    char                        paramKatOpt[]   = "kat";
    char                        paramGcmOpt[]   = "gcm";


    //  Open the gcm.txt file for input processing
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&fh,
                                               NSS_GCM_TXT,
                                               nn::fs::OpenMode_Read));

    //  Prep the test args array and count.  NULL values are filled
    //  in the looper for each iteration of the test.
    //  pk11gcmtest aes kat gcm $GCM_TESTDIR/tests/$INPUT_FILE
    testArgc = GCM_TEST_ARGV_INPUT_FILE_INDEX + 1;

    testArgv[GCM_TEST_ARGV_PROG_NAME_INDEX]    = paramProgName;
    testArgv[GCM_TEST_ARGV_AES_OPT_INDEX]      = paramAesOpt;
    testArgv[GCM_TEST_ARGV_KAT_OPT_INDEX]      = paramKatOpt;
    testArgv[GCM_TEST_ARGV_GCM_OPT_INDEX]      = paramGcmOpt;
    testArgv[GCM_TEST_ARGV_INPUT_FILE_INDEX]   = scratch;
    memset(scratch, 0, MAX_LINE_LENGTH);

    //  Read each line and parse out parameters (see cipher.sh)
    len = 0;
    offset = 0;
    for (len = helperutils::ReadLine(fh, offset, lineBuf, MAX_LINE_LENGTH - 1);
         len > 0;
         len = helperutils::ReadLine(fh, offset, lineBuf, MAX_LINE_LENGTH - 1))
    {
        char                    *fields[GCM_TEST_FIELD_COUNT];
        int                     expResultVal;
        int                     testResult;

        lineBuf[len] = '\0';
        offset += len;

        if (helperutils::ExtractFields(lineBuf,
                                       GCM_TEST_FIELD_COUNT,
                                       fields,
                                       TXT_DELIM) != 0) {
            continue;
        }

        PR_snprintf(scratch,
                    MAX_LINE_LENGTH,
                    "%s%s",
                    NSS_GCM_TEST_DIR,
                    fields[GCM_TEST_FIELD_INPUT_INDEX]);
        expResultVal               =
            atoi(fields[GCM_TEST_FIELD_RESULT_INDEX]);

        ::helperutils::dumpCallInfo("pk11gcmtestMain", testArgc, testArgv);

        testResult = pk11gcmtestMain(testArgc, testArgv);
        EXPECT_EQ(expResultVal, testResult);
    }

    nn::fs::CloseFile(fh);
}
#endif    //  ENABLE_GCM_CIPHER_TEST

}    //  nss
}    //  nnt


extern "C" void nnMain()
{
    int                         argc = nnt::GetHostArgc();
    char                        **argv = nnt::GetHostArgv();

    //  Init the test harness, get it ready to go
    ::testing::InitGoogleTest(&argc, argv);

    for (int i = 0; i < argc; i++)
    {

        if (strcmp(SIGLOHOST_OPTION, argv[i]) == 0)
        {
            //  Found the host option, grab it
            i++;
            HOST_MOUNT_PATH = argv[i];
            break;
        }
    }

    //  Mount the provided host path as the System mount location
    hostRootPath = nnsdkSetupHostSystemFs(MOUNT_POINT, HOST_MOUNT_PATH, 1);
    EXPECT_TRUE(NULL != ::hostRootPath);

    const int exitCode = RUN_ALL_TESTS();

    //  Testing is done, unmount the System mount location
    nnsdkShutdownHostFs();
    hostRootPath = NULL;

    nnt::Exit(exitCode);
}

extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 24 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    NN_ASSERT( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );

    NN_LOG("nninitStartup: loaded at %p\n", (void *)nninitStartup);
}
