﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>

#include <nnt/nntest.h>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/nifm.h>
#include <nn/socket.h>
#include <nn/ssl.h>

#include <Common/testCommonUtil.h>

#include "Executer.h"

namespace
{
static const int         ServerPortNumber      = 5555;
static const int         ServerThreadStackSize = 1024 * 32;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

nn::Result Initialize(SslTestCommonUtil* pInCommonUtil)
{
    nn::Result result;

    do
    {
        // ----------------------------------------------------------------------------------------
        // Basic initialization
        // ----------------------------------------------------------------------------------------
        result = pInCommonUtil->SetupNetwork();
        if (result.IsFailure())
        {
            NN_LOG("[ERROR] Failed to initialize the network interface.\n");
            break;
        }

        result = nn::socket::Initialize(
            g_SocketMemoryPoolBuffer,
            nn::socket::DefaultSocketMemoryPoolSize,
            nn::socket::MinSocketAllocatorSize,
            nn::socket::DefaultConcurrencyLimit);
        if (result.IsFailure())
        {
            NN_LOG("[ERROR] Failed to initialize the socket library.\n");
            break;
        }

        result = nn::ssl::Initialize();
        if (result.IsFailure())
        {
            NN_LOG("[ERROR] Failed to initialize the SSL library.\n");
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    return result;
}

void Finalize(SslTestCommonUtil* pInCommonUtil)
{
    nn::ssl::Finalize();
    nn::socket::Finalize();
    pInCommonUtil->FinalizeNetwork();
}

void RunWithArguments()
{
    NN_LOG(" Found command line argument, will run one time job based on it.\n");

    static const int StringSize = 1024 * 4;

    char**   argv        = nn::os::GetHostArgv();
    char*    pArgsString = new char[StringSize];
    uint32_t curOffset   = 0;

    if (pArgsString == nullptr)
    {
        NN_LOG(" [ERROR] Failed to allocate memory for arguments.\n");
        return;
    }

    // Create a string from args
    for (int i = 1; i < nn::os::GetHostArgc(); i++)
    {
        char* pCurPtr = argv[i];
        while (*pCurPtr != '\0')
        {
            pArgsString[curOffset++] = *pCurPtr;
            pCurPtr++;
        }

        pArgsString[curOffset++] = 0x20;
    }
    pArgsString[curOffset] = '\0';

    // Run test
    Executer executer;
    executer.ParseCommand(pArgsString, strnlen(pArgsString, StringSize));
    executer.RunTest();

    if (pArgsString != nullptr)
    {
        delete[] pArgsString;
    }
}

} // Un-named namespace


extern "C" void nnMain()
{
    nn::Result        result;
    SslTestCommonUtil commonUtil;
    ExecuterServer*   pServer = nullptr;

    do
    {
        if (Initialize(&commonUtil).IsFailure())
        {
            break;
        }

        NN_LOG("----------------------------------\n");
        NN_LOG(" Starting stress util\n");
        NN_LOG("----------------------------------\n");

        if (nn::os::GetHostArgc() > 1)
        {
            RunWithArguments();
        }
        else
        {
            pServer = new ExecuterServer(ServerThreadStackSize);
            if (pServer == nullptr)
            {
                NN_LOG(" [ERROR] Failed to allocate memory for the server.\n");
                break;
            }

            result = pServer->Initialize(ServerPortNumber);
            if (result.IsFailure())
            {
                break;
            }
        }
    } while (NN_STATIC_CONDITION(false));

    if (pServer != nullptr && result.IsSuccess())
    {
        pServer->Wait();
        pServer->Finalize();
    }

    if (pServer != nullptr)
    {
        delete pServer;
    }

    Finalize(&commonUtil);

    NN_LOG("----------------------------------\n");
    NN_LOG(" Terminating stress util\n");
    NN_LOG("----------------------------------\n");
}

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}
