﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/ssl.h>

// ------------------------------------------------------------------------------------------------
// Executer
// ------------------------------------------------------------------------------------------------
class Executer
{
private:
    // ------------------------------------------------------------------------
    // Constatns
    // ------------------------------------------------------------------------
    static const int HostNameLength            = 128;
    static const int ArgOptionMessageLength    = 128;
    static const int TestCaseDescriptionLength = 128;

    // ------------------------------------------------------------------------
    // Enums/Structures
    // ------------------------------------------------------------------------
    enum InternalResult
    {
        InternalResult_Failure = 0,
        InternalResult_Success,
        InternalResult_Unsupported,
        InternalResult_NoCommand,
        InternalResult_Terminated,
    };

    enum ArgOptionType
    {
        ArgOptionType_Command = 0x00,
        ArgOptionType_Message,
        ArgOptionType_Max
    };

    // Make sure to change this along with Executer::ArgOptions
    enum ArgOption
    {
        ArgOption_Help = 0x0,
        ArgOption_Exit,
        ArgOption_Verbose,
        ArgOption_Host,
        ArgOption_Port,
        ArgOption_RunCount,
        ArgOption_RunTestCase,
        ArgOption_VerifyOption,
        ArgOption_SessionCacheMode,
        ArgOption_Multi,
        ArgOption_EnablePeriodicHeapDump,
        ArgOption_DisablePeriodicHeapDump,
        ArgOption_EnablePeriodicSessionCacheDump,
        ArgOption_DisablePeriodicSessionCacheDump,
        ArgOption_DumpHeapStats,
        ArgOption_Max
    };

    // Make sure to change this along with Executer::TestCaseDescription
    enum TestCase
    {
        TestCase_Invalid = 0x00,
        TestCase_RunContextConnection,
        TestCase_RunContextConnectionWithHandshake,
        TestCase_RunServerPkiImport,
        TestCase_RunClientPkiImport,
        TestCase_HandshakeWithSingleHost,
        TestCase_HandshakeWithMultieHosts,
        TestCase_Abuse,
    };

    typedef struct ConfigParams
    {
        bool                                  isVerbose;
        char                                  pHostName[HostNameLength];
        uint16_t                              portNumber;
        uint32_t                              runCount;
        int                                   testCaseNumber;
        nn::ssl::Connection::VerifyOption     verifyOption;
        nn::ssl::Connection::SessionCacheMode sessionCacheMode;
        uint32_t                              multiCount;
    } ConfigParams;

    // ------------------------------------------------------------------------
    // Members
    // ------------------------------------------------------------------------
    ConfigParams m_ConfigParams;
    uint64_t     m_CommandCount;

    // ------------------------------------------------------------------------
    // APIs
    // ------------------------------------------------------------------------
    void           InitializeConfigParams();
    InternalResult FindCommand(char* pOutCommand, const char* pInCommand, int& curOffset);
    void           PrintHelp();


    Executer::InternalResult ConfigureVerifyOption(
        nn::ssl::Connection::VerifyOption* pOutVerifyOption,
        const char* pInConfigString);
    Executer::InternalResult ConfigureSessionCacheMode(
        nn::ssl::Connection::SessionCacheMode* pOutSessionCacheMode,
        const char* pInConfigString);

public:

    // ------------------------------------------------------------------------
    // Constatns
    // ------------------------------------------------------------------------
    static const char ArgOptions[ArgOption_Max + 1][ArgOptionType_Max][ArgOptionMessageLength];
    static const char TestCaseDescription[][TestCaseDescriptionLength];

    // ------------------------------------------------------------------------
    // APIs
    // ------------------------------------------------------------------------
    bool ParseCommand(const char* pInCommand, int commandLength);
    bool RunTest();
};

// ------------------------------------------------------------------------------------------------
// ExecuterServerServer
// ------------------------------------------------------------------------------------------------
class ExecuterServer
{
private:
    static const int DefaultThreadStackSize = 1024 * 8;
    static const int ReadBufferSize         = 1024;

    char*              m_pThreadStack;
    uint64_t           m_ThreadStackSize;
    nn::os::ThreadType m_ThreadTid;
    uint16_t           m_Port;
    nn::os::Event      m_WaitToBeDone;

    static void MainLoop(void* arg);

public:
    ExecuterServer();
    NN_IMPLICIT ExecuterServer(uint64_t ThreadStackSize);
    ~ExecuterServer();

    nn::Result Initialize(uint16_t port);
    void       Finalize();
    void       Wait();
    void       SignalReadyToBeFinalized();
    uint16_t   GetPortNumber();
};
