﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/ssl.h>
#include <nn/ssl/ssl_Api.debug.h>
#include <nn/socket.h>

#include "ExecuterTests/ExecuterTests.h"
#include "Executer.h"

// ------------------------------------------------------------------------------------------------
// Constants
// ------------------------------------------------------------------------------------------------
const char Executer::ArgOptions[Executer::ArgOption_Max + 1][Executer::ArgOptionType_Max][Executer::ArgOptionMessageLength] = {
    {"--help",                       "This command."},
    {"--exit",                       "Exit the program."},
    {"--verbose",                    "Enable verbose prints"},
    {"--host",                       "Specify the host name (no need to pass https:// in a string."},
    {"--port",                       "Specify the port number (443 will be used when not specified)."},
    {"--count",                      "Specify the number of times to run the test case."},
    {"--run",                        "Specify the test case number to run."},
    {"--verify",                     "Specify the verify option (none, all, peer, name, date). all will be used when not specified"},
    {"--cache",                      "Specify the session cache mode (none, id, ticket). id will be used when not specified"},
    {"--multi",                      "Specify the number, meaning of this parameter varies (refer the description of each test case)"},
    {"--enable_heap_dump",           "Enable periodic heap stats dump"},
    {"--disable_heap_dump",          "Disable periodic heap stats dump"},
    {"--enable_session_cache_dump",  "Enable periodic session cache info dump"},
    {"--disable_session_cache_dump", "Disable periodic session cache info dump"},
    {"--dump_heap_stats",            "Dump the stats of the SSL process heap"}
};

const char Executer::TestCaseDescription[][TestCaseDescriptionLength] = {
    {"Create/Destroy SSL connections/contexts without the SSL handshake (optional: --count)"},
    {"Create/Destroy SSL connections/contexts with the SSL handshake (required: --host)(optional: --port, --count)"},
    {"Import server PKIs, --multi option could specify the number of certs imported in one test run (optional: --count, --multi)"},
    {"Import client PKI (optional: --count)"},
    {"Establish the SSL connection (required: --host)(optional: --port, --count, --verify, --cache)"},
    {"Establish SSL connections with pre-defined multiple hosts (optional: --count, --verify, --cache)"},
    {"Run abuse test case"}
};

// ------------------------------------------------------------------------------------------------
// Private methods
// ------------------------------------------------------------------------------------------------
void Executer::PrintHelp()
{
    NN_LOG(" Usage -------------------------------------------------\n");
    for(int i = 0; i < ArgOption_Max; i++)
    {
        NN_LOG(" %s - %s\n",
            ArgOptions[i][ArgOptionType_Command],
            ArgOptions[i][ArgOptionType_Message]);
    }

    NN_LOG("  [Test Cases]\n");
    for(int i = 0; i < sizeof(TestCaseDescription) / sizeof(TestCaseDescription[0]); i++)
    {
        NN_LOG("  [%d] - %s\n", i + 1, TestCaseDescription[i]);
    }

    NN_LOG("\n");
}

void Executer::InitializeConfigParams()
{
    memset(&m_ConfigParams, 0x00, sizeof(m_ConfigParams));

    m_ConfigParams.testCaseNumber    = -1;
    m_ConfigParams.isVerbose         = false;
    m_ConfigParams.verifyOption      = nn::ssl::Connection::VerifyOption::VerifyOption_All;
    m_ConfigParams.sessionCacheMode  = nn::ssl::Connection::SessionCacheMode_SessionId;
    m_ConfigParams.multiCount        = 1;

    m_CommandCount = 0;
}

Executer::InternalResult Executer::FindCommand(char* pOutCommand, const char* pInCommand, int& curOffset)
{
    int            count = 0;
    InternalResult ret   = InternalResult_Success;

    while (pInCommand[curOffset] == 0x20)
    {
        curOffset++; // skip space
    }

    while (pInCommand[curOffset] != 0x20)
    {
        if ((pInCommand[curOffset] == 0x00) || // NULL
            (pInCommand[curOffset] == 0x0A) || // LF
            (pInCommand[curOffset] == 0x0D))   // CR
        {
            ret = InternalResult_Terminated;
            break;
        }

        pOutCommand[count++] = pInCommand[curOffset++];
    }

    if (count == 0 && ret!= InternalResult_Terminated)
    {
        ret = InternalResult_NoCommand;
    }

    return ret;
}

Executer::InternalResult Executer::ConfigureVerifyOption(
    nn::ssl::Connection::VerifyOption* pOutVerifyOption,
    const char* pInConfigString)
{
    Executer::InternalResult result = InternalResult_Failure;

    *pOutVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;

    do
    {
        if (strcmp(pInConfigString, "none") == 0)
        {
            break; // already initialized above
        }
        else if (strcmp(pInConfigString, "all") == 0)
        {
            *pOutVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_All;
        }
        else if (strcmp(pInConfigString, "peer") == 0)
        {
            *pOutVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa;
        }
        else if (strcmp(pInConfigString, "name") == 0)
        {
            *pOutVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_HostName;
        }
        else if (strcmp(pInConfigString, "date") == 0)
        {
            *pOutVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_DateCheck;
        }
        else
        {
            NN_LOG(" [ERROR] Unsupported string (%s)\n", pInConfigString);
            result = InternalResult_Unsupported;
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    if (result != InternalResult_Unsupported)
    {
        result = InternalResult_Success;
    }

    return result;
}

Executer::InternalResult Executer::ConfigureSessionCacheMode(
    nn::ssl::Connection::SessionCacheMode* pOutSessionCacheMode,
    const char* pInConfigString)
{
    Executer::InternalResult result = InternalResult_Failure;

    *pOutSessionCacheMode = nn::ssl::Connection::SessionCacheMode_None;

    do
    {
        if (strcmp(pInConfigString, "none") == 0)
        {
            break; // already initialized above
        }
        else if (strcmp(pInConfigString, "id") == 0)
        {
            *pOutSessionCacheMode = nn::ssl::Connection::SessionCacheMode_SessionId;
        }
        else if (strcmp(pInConfigString, "ticket") == 0)
        {
            *pOutSessionCacheMode = nn::ssl::Connection::SessionCacheMode_SessionTicket;
        }
        else
        {
            NN_LOG(" [ERROR] Unsupported string (%s)\n", pInConfigString);
            result = InternalResult_Unsupported;
            break;
        }
    } while (NN_STATIC_CONDITION(false));

    if (result != InternalResult_Unsupported)
    {
        result = InternalResult_Success;
    }

    return result;
}

// ------------------------------------------------------------------------------------------------
// Public methods
// ------------------------------------------------------------------------------------------------
bool Executer::ParseCommand(const char* pInCommand, int commandLength)
{
    bool isFailed    = false;
    bool isDone      = false;
    bool isParseDone = false;
    int  curOffset   = 0;

    InitializeConfigParams();

    while (curOffset < commandLength)
    {
        // Parse the current arg command
        int  commandIndex = 0;
        char curArg[128] = {0};

        if (FindCommand(curArg, pInCommand, curOffset) == InternalResult_Terminated)
        {
            isParseDone = true;
        }

        // Search the command
        while (commandIndex < ArgOption_Max)
        {
            const char* pCurCommandCandidate = Executer::ArgOptions[commandIndex][ArgOptionType_Command];

            int maxLen = (strlen(pCurCommandCandidate) > sizeof(curArg))?sizeof(curArg):strlen(pCurCommandCandidate);
            if (strncmp(curArg, Executer::ArgOptions[commandIndex][ArgOptionType_Command], maxLen) == 0)
            {
                break;
            }

            commandIndex++;
        }

        // Found the command
        switch (commandIndex)
        {
        case ArgOption_Help:
            PrintHelp();
            break;
        case ArgOption_Exit:
            NN_LOG(" Received exit command.\n");
            isDone = true;
            break;
        case ArgOption_Verbose:
            m_ConfigParams.isVerbose = true;
            NN_LOG(" [%s] Enabled\n", Executer::ArgOptions[commandIndex][ArgOptionType_Command]);
            break;
        case ArgOption_Host:
            memset(&curArg, 0x00, sizeof(curArg));
            if (FindCommand(curArg, pInCommand, curOffset) == InternalResult_NoCommand)
            {
                NN_LOG(" [ERROR] Host name was not found for --host: %s\n", curArg);
                isFailed = true;
                break;
            }
            strncpy(m_ConfigParams.pHostName, curArg, strnlen(curArg, sizeof(curArg)));
            NN_LOG(" [%s] Set (%s)\n", Executer::ArgOptions[commandIndex][ArgOptionType_Command], m_ConfigParams.pHostName);
            break;
        case ArgOption_Port:
            memset(&curArg, 0x00, sizeof(curArg));
            if (FindCommand(curArg, pInCommand, curOffset) == InternalResult_NoCommand)
            {
                NN_LOG(" [ERROR] Port number was not found for --port: %s\n", curArg);
                isFailed = true;
                break;
            }
            m_ConfigParams.portNumber = static_cast<uint16_t>(atoi(curArg));
            NN_LOG(" [%s] Set (%d)\n", Executer::ArgOptions[commandIndex][ArgOptionType_Command], m_ConfigParams.portNumber);
            break;
        case ArgOption_RunCount:
            memset(&curArg, 0x00, sizeof(curArg));
            if (FindCommand(curArg, pInCommand, curOffset) == InternalResult_NoCommand)
            {
                NN_LOG(" [ERROR] Count number was not found for --port: %s\n", curArg);
                isFailed = true;
                break;
            }
            m_ConfigParams.runCount = static_cast<uint32_t>(atoi(curArg));
            NN_LOG(" [%s] Set (%d)\n", Executer::ArgOptions[commandIndex][ArgOptionType_Command], m_ConfigParams.runCount);
            break;
        case ArgOption_RunTestCase:
            memset(&curArg, 0x00, sizeof(curArg));
            if (FindCommand(curArg, pInCommand, curOffset) == InternalResult_NoCommand)
            {
                NN_LOG(" [ERROR] Test case number was not found for --port: %s\n", curArg);
                isFailed = true;
                break;
            }
            m_ConfigParams.testCaseNumber = atoi(curArg);
            NN_LOG(" [%s] Set (%d)\n", Executer::ArgOptions[commandIndex][ArgOptionType_Command], m_ConfigParams.testCaseNumber);
            break;
        case ArgOption_VerifyOption:
            {
                memset(&curArg, 0x00, sizeof(curArg));
                if (FindCommand(curArg, pInCommand, curOffset) == InternalResult_NoCommand)
                {
                    NN_LOG(" [ERROR] option string was not found for --verify: %s\n", curArg);
                    isFailed = true;
                    break;
                }

                nn::ssl::Connection::VerifyOption tmpVerifyOption;
                if (Executer::ConfigureVerifyOption(&tmpVerifyOption, curArg) != InternalResult_Success)
                {
                    isFailed = true;
                    break;
                }
                m_ConfigParams.verifyOption = tmpVerifyOption;
                NN_LOG(" [%s] Set (0x%x)\n", Executer::ArgOptions[commandIndex][ArgOptionType_Command], m_ConfigParams.verifyOption);
            }
            break;
        case ArgOption_SessionCacheMode:
            {
                memset(&curArg, 0x00, sizeof(curArg));
                if (FindCommand(curArg, pInCommand, curOffset) == InternalResult_NoCommand)
                {
                    NN_LOG(" [ERROR] option string was not found for --cache: %s\n", curArg);
                    isFailed = true;
                    break;
                }

                nn::ssl::Connection::SessionCacheMode tmpSessionCacheMode;
                if (Executer::ConfigureSessionCacheMode(&tmpSessionCacheMode, curArg) != InternalResult_Success)
                {
                    isFailed = true;
                    break;
                }
                m_ConfigParams.sessionCacheMode = tmpSessionCacheMode;
                NN_LOG(" [%s] Set (0x%x)\n", Executer::ArgOptions[commandIndex][ArgOptionType_Command], m_ConfigParams.sessionCacheMode);
            }
            break;
        case ArgOption_Multi:
            memset(&curArg, 0x00, sizeof(curArg));
            if (FindCommand(curArg, pInCommand, curOffset) == InternalResult_NoCommand)
            {
                NN_LOG(" [ERROR] Count number was not found for --port: %s\n", curArg);
                isFailed = true;
                break;
            }
            m_ConfigParams.multiCount = static_cast<uint32_t>(atoi(curArg));
            NN_LOG(" [%s] Set (%d)\n", Executer::ArgOptions[commandIndex][ArgOptionType_Command], m_ConfigParams.runCount);
            break;
        case ArgOption_EnablePeriodicHeapDump:
        case ArgOption_EnablePeriodicSessionCacheDump:
            {
                nn::ssl::Debug::PeriodicDumpConfig config;
                config.enable = true;
                if (commandIndex == ArgOption_EnablePeriodicHeapDump)
                {
                    config.type = nn::ssl::Debug::PeriodicDumpType_HeapStats;
                }
                else
                {
                    config.type = nn::ssl::Debug::PeriodicDumpType_SessionCacheInfo;
                }

                nn::ssl::Debug::Input inputData;
                inputData.pBuffer = reinterpret_cast<const char*>(&config);
                inputData.bufferSize = sizeof(config);

                if (nn::ssl::Debug::Ioctl(nullptr, &inputData, nn::ssl::Debug::IoctlCommand_ConfigurePeriodicDump).IsFailure())
                {
                    NN_LOG(" [ERROR] Failed to enable %s periodic dump\n",
                        (commandIndex == ArgOption_EnablePeriodicHeapDump)?"heap":"session cache");
                }
                else
                {
                    NN_LOG(" Enabled %s periodic dump\n",
                        (commandIndex == ArgOption_EnablePeriodicHeapDump)?"heap":"session cache");
                }
            }
            break;
        case ArgOption_DisablePeriodicHeapDump:
        case ArgOption_DisablePeriodicSessionCacheDump:
            {
                nn::ssl::Debug::PeriodicDumpConfig config;
                config.enable = false;
                if (commandIndex == ArgOption_DisablePeriodicHeapDump)
                {
                    config.type = nn::ssl::Debug::PeriodicDumpType_HeapStats;
                }
                else
                {
                    config.type = nn::ssl::Debug::PeriodicDumpType_SessionCacheInfo;
                }

                nn::ssl::Debug::Input inputData;
                inputData.pBuffer = reinterpret_cast<const char*>(&config);
                inputData.bufferSize = sizeof(config);

                if (nn::ssl::Debug::Ioctl(nullptr, &inputData, nn::ssl::Debug::IoctlCommand_ConfigurePeriodicDump).IsFailure())
                {
                    NN_LOG(" [ERROR] Failed to enable %s periodic dump\n",
                        (commandIndex == ArgOption_DisablePeriodicHeapDump)?"heap":"session cache");
                }
                else
                {
                    NN_LOG(" Disabled %s periodic dump\n",
                        (commandIndex == ArgOption_DisablePeriodicHeapDump)?"heap":"session cache");
                }
            }
            break;
        case ArgOption_DumpHeapStats:
            {
                nn::ssl::Debug::HeapStats stats;
                nn::ssl::Debug::Output    outputData;

                outputData.pBuffer    = reinterpret_cast<char*>(&stats);
                outputData.bufferSize = sizeof(stats);

                if (nn::ssl::Debug::Ioctl(&outputData, nullptr, nn::ssl::Debug::IoctlCommand_GetHeapStats).IsFailure())
                {
                    NN_LOG(" [ERROR] Failed to get heap stats\n");
                }
                else
                {
                    NN_LOG(" The SSL heap stats\n  cur:%d(%dKB)\n  max:%d(%dKB)\n  min:%d(%dKB)\n",
                        stats.curSize, stats.curSize / 1024,
                        stats.maxSize, stats.maxSize / 1024,
                        stats.minSize, stats.minSize / 1024);
                }
            }
            break;
        default:
            if (isParseDone != true || (isParseDone == true && m_CommandCount == 0))
            {
                NN_LOG(" [ERROR] Received unkown command: %s\n", curArg);
            }
            break;
        }

        if (isFailed == true || isDone == true || isParseDone == true)
        {
            break;
        }
        m_CommandCount++;
    }

    return isDone;
} // NOLINT(impl/function_size)

bool Executer::RunTest()
{
    if (m_CommandCount == 0)
    {
        return true; // nothing to run
    }

    if (m_ConfigParams.testCaseNumber < 0)
    {
        NN_LOG(" [ERROR] No test case number is set.\n");
        return false;
    }

    bool isSucceeded = false;

    NN_LOG(" ****************\n");
    NN_LOG(" Run test case:%d (%s)\n",
        m_ConfigParams.testCaseNumber,
        TestCaseDescription[m_ConfigParams.testCaseNumber - 1]);
    NN_LOG(" ****************\n");

    switch (m_ConfigParams.testCaseNumber)
    {
    case TestCase_RunContextConnection:
        {
            ExecuterTestsRunContextConnection* pTest = new ExecuterTestsRunContextConnection(m_ConfigParams.runCount);
            if (pTest == nullptr)
            {
                break;
            }
            if (m_ConfigParams.isVerbose == true)
            {
                pTest->EnableVerbose();
            }
            pTest->Run();
            isSucceeded = pTest->IsSucceeded();
        }
        break;
    case TestCase_RunContextConnectionWithHandshake:
        {
            ExecuterTestsRunContextConnection* pTest = new ExecuterTestsRunContextConnection(
                m_ConfigParams.pHostName,
                m_ConfigParams.portNumber,
                m_ConfigParams.runCount);
            if (pTest == nullptr)
            {
                break;
            }
            if (m_ConfigParams.isVerbose == true)
            {
                pTest->EnableVerbose();
            }
            pTest->Run();
            isSucceeded = pTest->IsSucceeded();
        }
        break;
    case TestCase_RunServerPkiImport:
        {
            ExecuterTestsRunServerPkiImport* pTest = new ExecuterTestsRunServerPkiImport(
                m_ConfigParams.runCount,
                m_ConfigParams.multiCount);
            if (pTest == nullptr)
            {
                break;
            }
            if (m_ConfigParams.isVerbose == true)
            {
                pTest->EnableVerbose();
            }
            pTest->Run();
            isSucceeded = pTest->IsSucceeded();
        }
        break;
    case TestCase_RunClientPkiImport:
        {
            ExecuterTestsRunClientPkiImport* pTest = new ExecuterTestsRunClientPkiImport(
                m_ConfigParams.runCount);
            if (pTest == nullptr)
            {
                break;
            }
            if (m_ConfigParams.isVerbose == true)
            {
                pTest->EnableVerbose();
            }
            pTest->Run();
            isSucceeded = pTest->IsSucceeded();
        }
        break;
    case TestCase_HandshakeWithSingleHost:
        {
            ExecuterTestsRunWithSingleHost* pTest = new ExecuterTestsRunWithSingleHost(
                m_ConfigParams.pHostName,
                m_ConfigParams.verifyOption,
                m_ConfigParams.sessionCacheMode,
                m_ConfigParams.portNumber,
                m_ConfigParams.runCount);
            if (pTest == nullptr)
            {
                break;
            }
            if (m_ConfigParams.isVerbose == true)
            {
                pTest->EnableVerbose();
            }
            pTest->Run();
            isSucceeded = pTest->IsSucceeded();
        }
        break;
    case TestCase_HandshakeWithMultieHosts:
        {
            ExecuterTestsRunWithMultiHosts* pTest = new ExecuterTestsRunWithMultiHosts(
                m_ConfigParams.verifyOption,
                m_ConfigParams.sessionCacheMode,
                m_ConfigParams.runCount);
            if (pTest == nullptr)
            {
                break;
            }
            if (m_ConfigParams.isVerbose == true)
            {
                pTest->EnableVerbose();
            }
            pTest->Run();
            isSucceeded = pTest->IsSucceeded();
        }
        break;
    case TestCase_Abuse:
        {
            ExecuterTestsRunAbuse* pTest = new ExecuterTestsRunAbuse(m_ConfigParams.runCount);
            if (pTest == nullptr)
            {
                break;
            }
            if (m_ConfigParams.isVerbose == true)
            {
                pTest->EnableVerbose();
            }
            pTest->Run();
            isSucceeded = pTest->IsSucceeded();
        }
        break;
    default:
        {
            NN_LOG(" [ERROR] No test case was found for the specified number (%d).\n",
                m_ConfigParams.testCaseNumber);
        }
        break;
    }

    return isSucceeded;
} // NOLINT(impl/function_size)

// ------------------------------------------------------------------------------------------------
// ExecuterServerServer
// ------------------------------------------------------------------------------------------------
void ExecuterServer::MainLoop(void* arg)
{
    ExecuterServer* pServer      = reinterpret_cast<ExecuterServer*>(arg);
    int             serverSocket = -1;

    do
    {
        nn::socket::SockAddrIn serverAddr     = {0};
        serverAddr.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
        serverAddr.sin_port        = nn::socket::InetHtons(pServer->GetPortNumber());
        serverAddr.sin_family      = nn::socket::Family::Af_Inet;

        serverSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        if (serverSocket < 0)
        {
            NN_LOG(" [ERROR] Failed to create a server socket.\n");
            break;
        }

        if (nn::socket::Bind(serverSocket, reinterpret_cast<nn::socket::SockAddr *>(&serverAddr), sizeof(serverAddr)) < 0)
        {
            NN_LOG(" [ERROR] Failed to create a server socket.\n");
            break;
        }

        if (nn::socket::Listen(serverSocket, 1) < 0)
        {
            NN_LOG(" [ERROR] Failed to create a server socket.\n");
            break;
        }

        // ----------------------------------------------------------------------------------------
        // Main loop
        // ----------------------------------------------------------------------------------------

        char* pDataBuffer = new char[ReadBufferSize];
        if (pDataBuffer == nullptr)
        {
            NN_LOG(" [ERROR] Failed to allocate a buffer for read data.\n");
            break;
        }

        do
        {
            bool        isDone            = false;
            int         clientSocket      = -1;
            nn::socket::SockAddrIn clientAddr        = { 0 };
            nn::socket::SockLenT   clientAddressSize = sizeof(clientAddr);

            NN_LOG(" Waiting for connection on port %d....\n", pServer->GetPortNumber());
            if( (clientSocket = nn::socket::Accept(serverSocket, reinterpret_cast<nn::socket::SockAddr *>(&clientAddr), &clientAddressSize)) < 0 )
            {
                NN_LOG(" [ERROR] Failed to accept a client.\n");
                break;
            }
            NN_LOG(" Accepted connection from %s\n", nn::socket::InetNtoa(clientAddr.sin_addr));

            // ----------------------------------------------------------------------------------------
            // Wait for receiving commands
            // ----------------------------------------------------------------------------------------
            do
            {
                ssize_t receivedBytes   = 0;
                memset(pDataBuffer, 0x00, ReadBufferSize);
                receivedBytes = nn::socket::Recv(clientSocket, pDataBuffer, ReadBufferSize - 1, nn::socket::MsgFlag::Msg_None);
                if (receivedBytes < 0)
                {
                    NN_LOG(" [ERROR] Failed to receive data\n");
                    isDone = true;
                    break;
                }
                else if (receivedBytes == 0)
                {
                    NN_LOG(" Connection was closed by the client.\n");
                    break;
                }
                else
                {
                    Executer executer;
                    if (executer.ParseCommand(pDataBuffer, static_cast<int>(receivedBytes)) == true)
                    {
                        isDone = true;
                        break;
                    }
                    executer.RunTest();
                }
            } while (NN_STATIC_CONDITION(true));

            nn::socket::Close(clientSocket);
            if (isDone == true)
            {
                break;
            }
        } while (NN_STATIC_CONDITION(true));

        if (pDataBuffer != nullptr)
        {
            delete[] pDataBuffer;
        }
    } while (NN_STATIC_CONDITION(false));

    if (serverSocket >= 0)
    {
        nn::socket::Close(serverSocket);
    }

    pServer->SignalReadyToBeFinalized();
}

ExecuterServer::ExecuterServer() :
    m_ThreadStackSize(DefaultThreadStackSize),
    m_Port(0),
    m_WaitToBeDone(nn::os::EventClearMode_ManualClear)
{
    m_pThreadStack = new char[DefaultThreadStackSize + nn::os::StackRegionAlignment - 1];
}

ExecuterServer::ExecuterServer(uint64_t threadStackSize) :
    m_Port(0),
    m_WaitToBeDone(nn::os::EventClearMode_ManualClear)
{
    m_ThreadStackSize = threadStackSize;
    m_pThreadStack = new char[threadStackSize + nn::os::StackRegionAlignment - 1];
}

ExecuterServer::~ExecuterServer()
{
    if (m_pThreadStack)
    {
        delete[] m_pThreadStack;
    }
}
nn::Result ExecuterServer::Initialize(uint16_t port)
{
    nn::Result result;

    if (m_pThreadStack == nullptr)
    {
        NN_LOG(" [ERROR] Failed to allocate memory for the thread stack.\n");
        return nn::ssl::ResultErrorLower();
    }

    char* pAligned = reinterpret_cast<char *>(
        reinterpret_cast<uint64_t>(m_pThreadStack + (nn::os::StackRegionAlignment - 1)) &
        ~(nn::os::StackRegionAlignment - 1));
    m_Port = port;

    result = nn::os::CreateThread(
        &m_ThreadTid,
        MainLoop,
        this,
        pAligned,
        m_ThreadStackSize,
        nn::os::LowestThreadPriority);
    if (result.IsFailure())
    {
        return result;
    }

    nn::os::StartThread(&m_ThreadTid);

    return result;
}

void ExecuterServer::Finalize()
{
    nn::os::WaitThread(&m_ThreadTid);
    nn::os::DestroyThread(&m_ThreadTid);

    delete[] m_pThreadStack;
    m_pThreadStack = nullptr;
}

void ExecuterServer::Wait()
{
    m_WaitToBeDone.Wait();
}

void ExecuterServer::SignalReadyToBeFinalized()
{
    m_WaitToBeDone.Signal();
}

uint16_t ExecuterServer::GetPortNumber()
{
    return m_Port;
}
