﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/nntest.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <Common/testCommonUtil.h>

#ifndef NN_BUILD_CONFIG_OS_WIN32
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_TimeZoneApi.h>
#endif

#include "PkiInfo.h"
#include "Common.h"


// ------------------------------------------------------------------------------------------------
// Note
// ------------------------------------------------------------------------------------------------
/*

[APIs in nn::ssl]
--------------------+-------------+----------------------------------------------------------------
  namespace/class   |     From    | ID  | API
--------------------+-------------+----------------------------------------------------------------
nn::ssl::Context    | ----------- | --- |
                    | Pre-Install | 200 | nn::Result ImportClientPki(
                    | ----------- | --- |     CertStoreId* pOutCertId,
                    | ----------- | --- |     const char* pInP12Data,
                    | ----------- | --- |     const char* pInPwData,
                    | ----------- | --- |     uint32_t  p12DataSize,
                    | ----------- | --- |     uint32_t  pwDataSize) NN_NOEXCEPT;
                    | Pre-Install | 201 | nn::Result RegisterInternalPki(CertStoreId* pOutCertId, InternalPki pkiType) NN_NOEXCEPT;
--------------------+-------------+----------------------------------------------------------------

*/

namespace
{
// ------------------------------------------------------------------------------------------------
// Grobal parameters
// ------------------------------------------------------------------------------------------------
const char     TestNintendoHost[]   = "dauth-dd1.ndas.srv.nintendo.net";
const uint16_t HttpsPortNumber      = 443;

SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

TestParam g_testResult[] = {
    {false, 200, "Pre-Install", "nn::ssl::Context::ImportClientPki"},
#ifdef NN_BUILD_CONFIG_OS_WIN
    {true, 201, "Pre-Install", "nn::ssl::Context::RegisterInternalPki"},
#else
    {false, 201, "Pre-Install", "nn::ssl::Context::RegisterInternalPki"},
#endif
};

const int TestIdCount = sizeof(g_testResult) / sizeof(g_testResult[0]);


} // Un-named namespace

//-------------------------------------------------------------------------------------------------
// InitTest - Initialization of the test
//-------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    ASSERT_TRUE(g_CommonUtil.SetupNetwork().IsSuccess());
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());

    nn::Result result = nn::ssl::Initialize();
    ASSERT_TRUE(result.IsSuccess());

#ifndef NN_BUILD_CONFIG_OS_WIN32
    nn::time::PosixTime posixTime;
    nn::time::CalendarTime calendarTime;

    result = nn::time::Initialize();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    result = nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    result = nn::time::ToCalendarTime(&calendarTime, nullptr, posixTime);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    NN_LOG(
        "\n Date/Time:  %04d-%02d-%02d %02d:%02d:%02d\n\n",
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);

    nn::time::Finalize();
#endif
}

TEST(HandshakeWithInternet, Success)
{
    nn::Result           result;
    nn::Result           verifyError;
    nn::ssl::CertStoreId certStoreId;
    int                  socketFd;
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;

    // ------------------------------------------------------------------------
    // 7th connection
    // ------------------------------------------------------------------------
    socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestNintendoHost, 0);
    ASSERT_TRUE(socketFd >= 0);
    pSslContext    = new nn::ssl::Context();
    pSslConnection = new nn::ssl::Connection();
    SetupSsl(pSslContext, pSslConnection, socketFd, TestNintendoHost);

    // Testing ID - 200 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslContext->ImportClientPki(
        &certStoreId,
        (const char *)PsuedoDeviceCert,
        (const char *)PsuedoDeviceCertPassword,
        PsuedoDeviceCertLen,
        PsuedoDeviceCertPasswordLen);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 200, result.IsSuccess());

    result = pSslConnection->DoHandshake();
    CHECK_HANDSHAKE_RESULT(result.IsSuccess(), result, pSslConnection);
    result = pSslConnection->FlushSessionCache();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    CleanupSsl(pSslContext, pSslConnection);

#ifndef NN_BUILD_CONFIG_OS_WIN
    // ------------------------------------------------------------------------
    // 8th connection
    // ------------------------------------------------------------------------
    socketFd = g_CommonUtil.CreateTcpSocket(true, HttpsPortNumber, TestNintendoHost, 0);
    ASSERT_TRUE(socketFd >= 0);
    pSslContext    = new nn::ssl::Context();
    pSslConnection = new nn::ssl::Connection();
    SetupSsl(pSslContext, pSslConnection, socketFd, TestNintendoHost);

    // Testing ID - 201 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
    result = pSslContext->RegisterInternalPki(
        &certStoreId,
        nn::ssl::Context::InternalPki_DeviceClientCertDefault);
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);
    SetTestDoneFlag(g_testResult, TestIdCount, 201, result.IsSuccess());

    result = pSslConnection->DoHandshake();
    CHECK_HANDSHAKE_RESULT(result.IsSuccess(), result, pSslConnection);
    CleanupSsl(pSslContext, pSslConnection);
#endif // NN_BUILD_CONFIG_OS_WIN
}

//-------------------------------------------------------------------------------------------------
// FinalizeTest - Finalization of the test
//-------------------------------------------------------------------------------------------------
TEST(FinalizeTest, Success)
{
    nn::Result result = nn::ssl::Finalize();
    EXPECT_TRUE_AND_LOG(result.IsSuccess(), result);

    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();

    DumpTestResult(g_testResult, TestIdCount);
}

//-------------------------------------------------------------------------------------------------
// nninitStartup
//-------------------------------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}
