﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <cstdlib>
#include <cstring>
#include <nn/os.h>
#include <nn/fs.h>
#include "util_TestFsManager.h"
#include "test_HeapConfig.h"
#include "util_TestResult.h"

namespace {
    const char* dirName = "rom:/";
    const int MaxFileName = 1023;
    char g_FileName[MaxFileName + 1];
    void* g_MountRomCacheBuffer = nullptr;
    size_t g_MountRomCacheBufferSize = 0;

    void* Allocator(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }

    char* MakeFileName(const char* name)
    {
        if (std::strlen(dirName) + std::strlen(name) > MaxFileName)
        {
            return NULL;
        }
        std::strcpy(g_FileName, dirName);
        return std::strcat(g_FileName, name);
    }
}

TestFsManager::TestFsManager() NN_NOEXCEPT
{
    nn::fs::SetAllocator(Allocator, Deallocate);

    auto result = nn::fs::QueryMountRomCacheSize(&g_MountRomCacheBufferSize);
    NN_ASSERT_RESULT_SUCCESS(result);
    g_MountRomCacheBuffer = Allocator(g_MountRomCacheBufferSize);
    NN_ASSERT_NOT_NULL(g_MountRomCacheBuffer);
    result = nn::fs::MountRom("rom", g_MountRomCacheBuffer, g_MountRomCacheBufferSize);
    NN_ASSERT_RESULT_SUCCESS(result);
}

TestFsManager::~TestFsManager() NN_NOEXCEPT
{
    nn::fs::Unmount("rom");
    Deallocate(g_MountRomCacheBuffer, g_MountRomCacheBufferSize);
    g_MountRomCacheBuffer = nullptr;
    g_MountRomCacheBufferSize = 0;
}

size_t TestFsManager::ReadAll(void* pOut, size_t bufferSize, const char* name)
{
    char* romName = MakeFileName(name);

    nn::fs::FileHandle file;
    auto result = nn::fs::OpenFile(&file, romName, nn::fs::OpenMode_Read);
    NN_ASSERT_RESULT_SUCCESS(result);

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, pOut, bufferSize);
    NN_ASSERT_RESULT_SUCCESS(result);
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);

    nn::fs::CloseFile(file);

    return readSize;
}

