﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ro.h>
#include <nn/ro/ro_Types.h>

#include "util_Common.h"
#include "bin_Definitions.h"

#include <nn/nn_Log.h>

namespace {
    const char* BinVersionScriptLocal1Name = "nro/BinVersionScriptLocal1.nro";
    const char* BinVersionScriptGlobal1Name = "nro/BinVersionScriptGlobal1.nro";
    const char* BinVersionScriptLocal2Name = "nro/BinVersionScriptLocal2.nro";
    const char* BinVersionScriptGlobal2Name = "nro/BinVersionScriptGlobal2.nro";

    typedef int (*ReturnInt)();
    typedef TestClassInt& (*ReturnClass)();
    typedef int* (*ReturnPtr)();
    const char* GetGlobalIntName = "_Z12GetGlobalIntv";
    const char* GetGlobalClassName = "_Z14GetGlobalClassv";
    const char* GetGlobalPtrName = "_Z12GetGlobalPtrv";
    const char* GlobalIntName = "g_GlobalInt";

    const char* GetGlobalIntFromDll1Name = "_Z20GetGlobalIntFromDll1v";
    const char* GetGlobalClassFromDll1Name = "_Z22GetGlobalClassFromDll1v";
    const char* GetGlobalPtrFromDll1Name = "_Z20GetGlobalPtrFromDll1v";

    const char* GetGlobalIntFromDll2Name = "_Z20GetGlobalIntFromDll2v";
    const char* GetGlobalClassFromDll2Name = "_Z22GetGlobalClassFromDll2v";
    const char* GetGlobalPtrFromDll2Name = "_Z20GetGlobalPtrFromDll2v";

    class VersionScriptTest : public ::testing::TestWithParam<nn::ro::BindFlag>
    {
    protected:
        virtual void SetUp()
        {
            m_Allocator = &TestAllocator::GetInstance();
            m_NroLocal1.SetUp(BinVersionScriptLocal1Name, m_Allocator->GetAllocator());
            m_NroGlobal1.SetUp(BinVersionScriptGlobal1Name, m_Allocator->GetAllocator());
            m_NroLocal2.SetUp(BinVersionScriptLocal2Name, m_Allocator->GetAllocator());
            m_NroGlobal2.SetUp(BinVersionScriptGlobal2Name, m_Allocator->GetAllocator());
            nn::ro::Initialize();
        }

        virtual void TearDown()
        {
            nn::ro::Finalize();
        }

        TestAllocator* m_Allocator;
        TestNro m_NroLocal1;
        TestNro m_NroLocal2;
        TestNro m_NroGlobal1;
        TestNro m_NroGlobal2;
    };

} // namespace

INSTANTIATE_TEST_CASE_P(ManualDll, VersionScriptTest, ::testing::Values(nn::ro::BindFlag_Lazy, nn::ro::BindFlag_Now));

TEST_P(VersionScriptTest, Local)
{
    auto result = m_NroLocal1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    /*
        1. LookupSymbol でLocal のシンボルを見つけられない
        2. LookupModuleSymbol でそのモジュールのLocal シンボルにアクセスできない
        3. モジュール内で正しくシンボル解決されている
     */

    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 2
    result = m_NroLocal1.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 3
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnInt getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 2
    result = m_NroLocal1.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 3
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnClass getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass = getGlobalClass();
    ASSERT_EQ(testClass.Get(), 1);
    ASSERT_EQ(testClass.Calc(), 2);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 2
    result = m_NroLocal1.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 3
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr = reinterpret_cast<ReturnPtr>(addr);

    ASSERT_TRUE(getGlobalPtr() != nullptr);

    m_NroLocal1.Unload();
}

TEST_P(VersionScriptTest, LocalLocal)
{
    auto result = m_NroLocal1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_NroLocal2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    /*
        1. LookupSymbol でLocal のシンボルを見つけられない
        2. LookupModuleSymbol でそのモジュールのLocal シンボルにアクセスできない
        3. モジュール内で正しくシンボル解決されている
     */

    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 2
    result = m_NroLocal1.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    result = m_NroLocal2.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 3
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnInt getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);

    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 2);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 2
    result = m_NroLocal1.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    result = m_NroLocal2.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 3
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnClass getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass = getGlobalClass();
    ASSERT_EQ(testClass.Get(), 1);
    ASSERT_EQ(testClass.Calc(), 2);

    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    testClass = getGlobalClass();
    ASSERT_EQ(testClass.Get(), 2);
    ASSERT_EQ(testClass.Calc(), 3);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 2
    result = m_NroLocal1.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    result = m_NroLocal2.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 3
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr1 = reinterpret_cast<ReturnPtr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr2 = reinterpret_cast<ReturnPtr>(addr);

    ASSERT_NE(getGlobalPtr1(), getGlobalPtr2());

    m_NroLocal1.Unload();
    m_NroLocal2.Unload();
}

TEST_P(VersionScriptTest, Global)
{
    auto result = m_NroGlobal1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    /*
        1. 関数の実行結果がGlobal 関数の結果になっている
        2. LookupSymbol の結果が Global 関数を指している
        3. LookupModuleSymbol でそのモジュールのGlobal シンボルにアクセスできる
        4. モジュール内で正しくシンボル解決されている
     */


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnInt getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);

    // 2, 3
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalInt));

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnClass getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass = getGlobalClass();
    ASSERT_EQ(testClass.Get(), 1);
    ASSERT_EQ(testClass.Calc(), 2);

    // 2, 3
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalClass));

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass1 = getGlobalClass();
    ASSERT_EQ(testClass1.Get(), 1);
    ASSERT_EQ(testClass1.Calc(), 2);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr = reinterpret_cast<ReturnPtr>(addr);

    result = nn::ro::LookupSymbol(&addr, GlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr()));

    // 2, 3
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr));

    result = m_NroGlobal1.FindSymbol(&addr, GlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr()));

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr1 = reinterpret_cast<ReturnPtr>(addr);
    ASSERT_EQ(getGlobalPtr1(), getGlobalPtr());


    m_NroGlobal1.Unload();
}

TEST_P(VersionScriptTest, GlobalGlobal)
{
    auto result = m_NroGlobal1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_NroGlobal2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    /*
       先にロードされたシンボルを参照することの確認
        1. 関数の実行結果が先にロードされた関数の結果になっている
        2. LookupSymbol の結果とロード順が正しい
        3. LookupModuleSymbol でそのモジュールのシンボルにアクセスできる
        4. モジュール内で正しくシンボル解決されている
     */

    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnInt getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalInt));

    // 3
    result = m_NroGlobal2.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_NE(addr, reinterpret_cast<uintptr_t>(getGlobalInt));

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);

    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnClass getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass = getGlobalClass();
    ASSERT_EQ(testClass.Get(), 1);
    ASSERT_EQ(testClass.Calc(), 2);

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalClass));

    // 3
    result = m_NroGlobal2.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_NE(addr, reinterpret_cast<uintptr_t>(getGlobalClass));

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass1 = getGlobalClass();
    ASSERT_EQ(testClass1.Get(), 1);
    ASSERT_EQ(testClass1.Calc(), 2);

    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass2 = getGlobalClass();
    ASSERT_EQ(testClass2.Get(), 1);
    ASSERT_EQ(testClass2.Calc(), 2);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr = reinterpret_cast<ReturnPtr>(addr);

    result = nn::ro::LookupSymbol(&addr, GlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr()));

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr));

    // 3
    result = m_NroGlobal2.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_NE(addr, reinterpret_cast<uintptr_t>(getGlobalPtr));

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr1 = reinterpret_cast<ReturnPtr>(addr);
    ASSERT_EQ(getGlobalPtr1(), getGlobalPtr());

    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr2 = reinterpret_cast<ReturnPtr>(addr);

    ASSERT_EQ(getGlobalPtr1(), getGlobalPtr2());

    m_NroGlobal2.Unload();
    m_NroGlobal1.Unload();
}

TEST_P(VersionScriptTest, GlobalLocal)
{
    auto result = m_NroGlobal1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_NroLocal2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    /*
       Local 内は、Local の関数を参照する
        1. 関数の実行結果がGlobal 関数の結果になっている
        2. LookupSymbol の結果が Global 関数を指している
        3. LookupModuleSymbol でそのモジュールのLocal シンボルにアクセスできない
        4. モジュール内で正しくシンボル解決されている
     */


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnInt getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalInt));

    // 3
    result = m_NroLocal2.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);

    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 2);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnClass getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass = getGlobalClass();
    ASSERT_EQ(testClass.Get(), 1);
    ASSERT_EQ(testClass.Calc(), 2);

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalClass));

    // 3
    result = m_NroLocal2.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass1 = getGlobalClass();
    ASSERT_EQ(testClass1.Get(), 1);
    ASSERT_EQ(testClass1.Calc(), 2);

    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass2 = getGlobalClass();
    ASSERT_EQ(testClass2.Get(), 2);
    ASSERT_EQ(testClass2.Calc(), 3);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr = reinterpret_cast<ReturnPtr>(addr);

    result = nn::ro::LookupSymbol(&addr, GlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr()));

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr));

    // 3
    result = m_NroLocal2.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr1 = reinterpret_cast<ReturnPtr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr2 = reinterpret_cast<ReturnPtr>(addr);

    ASSERT_NE(getGlobalPtr1(), getGlobalPtr2());

    m_NroLocal2.Unload();
    m_NroGlobal1.Unload();
}

TEST_P(VersionScriptTest, LocalGlobal)
{
    auto result = m_NroLocal2.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    result = m_NroGlobal1.Load(GetParam());
    ASSERT_RESULT_SUCCESS(result);

    uintptr_t addr;

    /*
       Local 内は、Local の関数を参照する
        1. 関数の実行結果がGlobal 関数の結果になっている
        2. LookupSymbol の結果が Global 関数を指している
        3. LookupModuleSymbol でそのモジュールのLocal シンボルにアクセスできない
        4. モジュール内で正しくシンボル解決されている
     */


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnInt getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalInt));

    // 3
    result = m_NroLocal2.FindSymbol(&addr, GetGlobalIntName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 1);

    result = nn::ro::LookupSymbol(&addr, GetGlobalIntFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalInt = reinterpret_cast<ReturnInt>(addr);
    ASSERT_EQ(getGlobalInt(), 2);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnClass getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass = getGlobalClass();
    ASSERT_EQ(testClass.Get(), 1);
    ASSERT_EQ(testClass.Calc(), 2);

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalClass));

    // 3
    result = m_NroLocal2.FindSymbol(&addr, GetGlobalClassName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass1 = getGlobalClass();
    ASSERT_EQ(testClass1.Get(), 1);
    ASSERT_EQ(testClass1.Calc(), 2);

    result = nn::ro::LookupSymbol(&addr, GetGlobalClassFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    getGlobalClass = reinterpret_cast<ReturnClass>(addr);
    TestClassInt& testClass2 = getGlobalClass();
    ASSERT_EQ(testClass2.Get(), 2);
    ASSERT_EQ(testClass2.Calc(), 3);


    // 1
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr = reinterpret_cast<ReturnPtr>(addr);

    result = nn::ro::LookupSymbol(&addr, GlobalIntName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr()));

    // 2
    result = m_NroGlobal1.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(addr, reinterpret_cast<uintptr_t>(getGlobalPtr));

    // 3
    result = m_NroLocal2.FindSymbol(&addr, GetGlobalPtrName);
    ASSERT_RESULT_FAILURE_VALUE(result, nn::ro::ResultNotFound());

    // 4
    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll1Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr1 = reinterpret_cast<ReturnPtr>(addr);

    result = nn::ro::LookupSymbol(&addr, GetGlobalPtrFromDll2Name);
    ASSERT_RESULT_SUCCESS(result);
    ReturnPtr getGlobalPtr2 = reinterpret_cast<ReturnPtr>(addr);

    ASSERT_NE(getGlobalPtr1(), getGlobalPtr2());

    m_NroLocal2.Unload();
    m_NroGlobal1.Unload();
}

