﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

#include "RidCompatibilityVerifier_SystemUpdateTest.h"

namespace
{
    //! ダウンロード状況のラベル
    struct SystemUpdateStatePair
    {
        nn::rid::SystemUpdateProgress::State state;
        std::string stateStr;
    } SystemUpdateStateLabelList[] = {
        { nn::rid::SystemUpdateProgress::State::DoNothing, "DoNothing" },
        { nn::rid::SystemUpdateProgress::State::Checking, "Checking" },
        { nn::rid::SystemUpdateProgress::State::Downloading, "Downloading" },
        { nn::rid::SystemUpdateProgress::State::Applying, "Applying" },
        { nn::rid::SystemUpdateProgress::State::Completed, "Completed" },
        { nn::rid::SystemUpdateProgress::State::Failed, "Failed" },
        { nn::rid::SystemUpdateProgress::State::NeedNoUpdate, "NeedNoUpdate" },
    };
}


RidTestSystemUpdater::RidTestSystemUpdater() NN_NOEXCEPT
    : m_IsStartUpdate(false), m_IsSystemUpdateDone(false)
{
}

RidTestSystemUpdater::~RidTestSystemUpdater() NN_NOEXCEPT
{
    if (m_IsStartUpdate)
    {
        //! 実行スレッドの破棄
        nn::os::WaitThread(&m_UpdateExecutionThread);
        nn::os::DestroyThread(&m_UpdateExecutionThread);
    }
}

void RidTestSystemUpdater::Start() NN_NOEXCEPT
{
    //! 本体更新スレッドの生成
    static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_UpdateExecutionThread, [](void * p)
    {
        auto self = reinterpret_cast<RidTestSystemUpdater*>(p);
        NN_LOG("Call SystemUpdater.Execute() Start\n");
        self->m_SystemUpdateResult = self->m_SystemUpdater.Execute();
        NN_LOG("Call SystemUpdater.Execute() End\n");

        self->m_IsSystemUpdateDone = true;
    }
    , this, s_Stack, sizeof(s_Stack), nn::os::DefaultThreadPriority));

    // 本体更新開始
    nn::os::StartThread(&m_UpdateExecutionThread);

    // 本体更新処理が着実に開始されるまで待つ
    while (NN_STATIC_CONDITION(true))
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        auto progress = m_SystemUpdater.GetProgress();
        if (progress.state != nn::rid::SystemUpdateProgress::State::DoNothing)
        {
            break;
        }
    }

    m_IsStartUpdate = true;
}

std::string RidTestSystemUpdater::GetState() NN_NOEXCEPT
{
    auto progress = m_SystemUpdater.GetProgress();

    for (auto& usl : SystemUpdateStateLabelList)
    {
        if (usl.state == progress.state)
        {
            return usl.stateStr;
        }
    }
    // ありえないはずだが該当する状態が無い場合は空文字を返しておく
    return "";
}

void RidTestSystemUpdater::GetProgress(int64_t& outLoaded, int64_t& outTotal) NN_NOEXCEPT
{
    auto progress = m_SystemUpdater.GetProgress();
    if (progress.state == nn::rid::SystemUpdateProgress::State::Downloading)
    {
        outLoaded = progress.loaded;
        outTotal = progress.total;
    }
}
