﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/psc.h>
#include <nn/psm/driver/detail/psm_Constants.h>
#include <nn/psm/driver/psm.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_SystemProcess.h>

#include <nnt/nntest.h>

#include "testPsm_ChargerDriverStub.h"
#include "testPsm_Common.h"

namespace nnt { namespace psm {

namespace {

// テスト内容(初期は OTG は対象外とする)
//
// 1. Status 設定(AC 4 種類)
// 2. ActiveNotice 通知
// 3. VDD50 確認
// 4. FastChargeCurrentLimit 確認
// 4. InputCurrentLimit 確認

// voltage HE15V 1.2A
// voltage HE13V - LT15V 1.5A
// voltage LT13V 2.0A

// OTG の時に設定するべきもの。
//        status.m_Data.Set<nn::usb::pd::Status::PowerRole>(nn::usb::pd::StatusPowerRole_Sink);
//        status.m_Data.Set<nn::usb::pd::Status::Request>(nn::usb::pd::StatusRequest_PowerOutput);
//        notice.Set<nn::usb::pd::Notice::RequestNotice>(true);

// OTG から戻すときに設定するべきもの。
//        status.m_Data.Set<nn::usb::pd::Status::PowerRole>(nn::usb::pd::StatusPowerRole_Source);
//        status.m_Data.Set<nn::usb::pd::Status::Request>(nn::usb::pd::StatusRequest_PowerInput);
//        notice.Set<nn::usb::pd::Notice::RequestNotice>(true);

// エラーチェック用。
//        status.m_Data.Set<nn::usb::pd::Status::Error>(nn::usb::pd::StatusError_OverVoltage);
//        status.m_Data.Set<nn::usb::pd::Status::Error>(nn::usb::pd::StatusError_CradleUsbHubUndetected);

struct PsmPolicyTestSetting
{
    // power supply for controllers
    bool controllerAcquired;

    // boost / normal
    bool enableFastBatteryCharging;

    // condition of power supply
    UsbPowerState usbPowerState;

    // expected outputs
    Vdd50Condition vdd50Condition;
    int fastChargeCurrentLimit;
    int inputCurrentLimit;
    int inputVoltageLimitMilliVolt;

    // expected outputs
    Vdd50Condition vdd50ConditionWithSleep;
    int fastChargeCurrentLimitWithSleep;
    int inputCurrentLimitWithSleep;
};

const PsmPolicyTestSetting PsmPolicyTestSettings[] =
{
    { true,  true,  UsbPowerState::He15V,       Vdd50Condition_Vdd50B, 2048, 1200, 3880, Vdd50Condition_Vdd50B, 2048, 1200 },
    { true,  true,  UsbPowerState::He13V,       Vdd50Condition_Vdd50B, 2048, 1500, 3880, Vdd50Condition_Vdd50B, 2048, 1500 },
    { true,  true,  UsbPowerState::Lt13V,       Vdd50Condition_Vdd50B, 2048, 2000, 3880, Vdd50Condition_Vdd50B, 2048, 2000 },
    { true,  true,  UsbPowerState::Dcp1500Ma,   Vdd50Condition_Vdd50A, 2048, 1500, 4360, Vdd50Condition_Vdd50A, 2048, 1500 },
    { true,  true,  UsbPowerState::Cdp1500Ma,   Vdd50Condition_Vdd50A, 2048, 1500, 4360, Vdd50Condition_Vdd50A, 2048, 1500 },
    { true,  true,  UsbPowerState::Sdp500Ma,    Vdd50Condition_Vdd50A, 2048, 500,  4360, Vdd50Condition_Vdd50A, 2048, 500 },
    { true,  true,  UsbPowerState::Apple500Ma,  Vdd50Condition_Vdd50A, 2048, 500,  4360, Vdd50Condition_Vdd50A, 2048, 500 },
    { true,  true,  UsbPowerState::Apple1000Ma, Vdd50Condition_Vdd50A, 2048, 900,  4360, Vdd50Condition_Vdd50A, 2048, 900 },
    { true,  true,  UsbPowerState::Apple2000Ma, Vdd50Condition_Vdd50A, 2048, 2000, 4360, Vdd50Condition_Vdd50A, 2048, 2000 },
    { true,  true,  UsbPowerState::Inactive,    Vdd50Condition_Vdd50A, 2048, 500,  4360, Vdd50Condition_Vdd50A, 2048, 500 },

    { true,  false, UsbPowerState::He15V,       Vdd50Condition_Vdd50B, 512,  1200, 3880, Vdd50Condition_Vdd50B, 512,  1200 },
    { true,  false, UsbPowerState::He13V,       Vdd50Condition_Vdd50B, 512,  1500, 3880, Vdd50Condition_Vdd50B, 512,  1500 },
    { true,  false, UsbPowerState::Lt13V,       Vdd50Condition_Vdd50B, 512,  2000, 3880, Vdd50Condition_Vdd50B, 512,  2000 },
    { true,  false, UsbPowerState::Dcp1500Ma,   Vdd50Condition_Vdd50A, 512,  1500, 4360, Vdd50Condition_Vdd50A, 512,  1500 },
    { true,  false, UsbPowerState::Cdp1500Ma,   Vdd50Condition_Vdd50A, 512,  1500, 4360, Vdd50Condition_Vdd50A, 512,  1500 },
    { true,  false, UsbPowerState::Sdp500Ma,    Vdd50Condition_Vdd50A, 512,  500,  4360, Vdd50Condition_Vdd50A, 512,  500 },
    { true,  false, UsbPowerState::Apple500Ma,  Vdd50Condition_Vdd50A, 512,  500,  4360, Vdd50Condition_Vdd50A, 512,  500 },
    { true,  false, UsbPowerState::Apple1000Ma, Vdd50Condition_Vdd50A, 512,  900,  4360, Vdd50Condition_Vdd50A, 512,  900 },
    { true,  false, UsbPowerState::Apple2000Ma, Vdd50Condition_Vdd50A, 512,  2000, 4360, Vdd50Condition_Vdd50A, 512,  2000 },
    { true,  false, UsbPowerState::Inactive,    Vdd50Condition_Vdd50A, 512,  500,  4360, Vdd50Condition_Vdd50A, 512,  500 },

    { false, true,  UsbPowerState::He15V,       Vdd50Condition_Vdd50B, 2048, 1200, 3880, Vdd50Condition_None,   2048, 1200 },
    { false, true,  UsbPowerState::He13V,       Vdd50Condition_Vdd50B, 2048, 1500, 3880, Vdd50Condition_None,   2048, 1500 },
    { false, true,  UsbPowerState::Lt13V,       Vdd50Condition_Vdd50B, 2048, 2000, 3880, Vdd50Condition_None,   2048, 2000 },
    { false, true,  UsbPowerState::Dcp1500Ma,   Vdd50Condition_Vdd50A, 2048, 1500, 4360, Vdd50Condition_None,   2048, 1500 },
    { false, true,  UsbPowerState::Cdp1500Ma,   Vdd50Condition_Vdd50A, 2048, 1500, 4360, Vdd50Condition_None,   2048, 1500 },
    { false, true,  UsbPowerState::Sdp500Ma,    Vdd50Condition_Vdd50A, 2048, 500,  4360, Vdd50Condition_None,   2048, 500 },
    { false, true,  UsbPowerState::Apple500Ma,  Vdd50Condition_Vdd50A, 2048, 500,  4360, Vdd50Condition_None,   2048, 500 },
    { false, true,  UsbPowerState::Apple1000Ma, Vdd50Condition_Vdd50A, 2048, 900,  4360, Vdd50Condition_None,   2048, 900 },
    { false, true,  UsbPowerState::Apple2000Ma, Vdd50Condition_Vdd50A, 2048, 2000, 4360, Vdd50Condition_None,   2048, 2000 },
    { false, true,  UsbPowerState::Inactive,    Vdd50Condition_Vdd50A, 2048, 500,  4360, Vdd50Condition_None,   2048, 500 },

    { false, false, UsbPowerState::He15V,       Vdd50Condition_Vdd50B, 512,  1200, 3880, Vdd50Condition_None,   512,  1200 },
    { false, false, UsbPowerState::He13V,       Vdd50Condition_Vdd50B, 512,  1500, 3880, Vdd50Condition_None,   512,  1500 },
    { false, false, UsbPowerState::Lt13V,       Vdd50Condition_Vdd50B, 512,  2000, 3880, Vdd50Condition_None,   512,  2000 },
    { false, false, UsbPowerState::Dcp1500Ma,   Vdd50Condition_Vdd50A, 512,  1500, 4360, Vdd50Condition_None,   512,  1500 },
    { false, false, UsbPowerState::Cdp1500Ma,   Vdd50Condition_Vdd50A, 512,  1500, 4360, Vdd50Condition_None,   512,  1500 },
    { false, false, UsbPowerState::Sdp500Ma,    Vdd50Condition_Vdd50A, 512,  500,  4360, Vdd50Condition_None,   512,  500 },
    { false, false, UsbPowerState::Apple500Ma,  Vdd50Condition_Vdd50A, 512,  500,  4360, Vdd50Condition_None,   512,  500 },
    { false, false, UsbPowerState::Apple1000Ma, Vdd50Condition_Vdd50A, 512,  900,  4360, Vdd50Condition_None,   512,  900 },
    { false, false, UsbPowerState::Apple2000Ma, Vdd50Condition_Vdd50A, 512,  2000, 4360, Vdd50Condition_None,   512,  2000 },
    { false, false, UsbPowerState::Inactive,    Vdd50Condition_Vdd50A, 512,  500,  4360, Vdd50Condition_None,   512,  500 },
};

class PsmPolicyTest : public ::testing::TestWithParam<PsmPolicyTestSetting>
{
public:
    static void SetUpTestCase()
    {
        ::nnt::psm::Initialize();
    }

    static void TearDownTestCase()
    {
        ::nnt::psm::Finalize();
    }
};

INSTANTIATE_TEST_CASE_P(PsmPolicyTestName, PsmPolicyTest, ::testing::ValuesIn(PsmPolicyTestSettings));

// 通常時とスリープ時の VDD50 の設定と充電電流、入力電流のチェック。
void CheckChargerAndVdd50(const PsmPolicyTestSetting& policyTestSetting,
    int fastChargeCurrentLimit,
    bool enoughPowerChargeEmulation) NN_NOEXCEPT
{
    nnt::psm::ChangeUsbPowerState(policyTestSetting.usbPowerState);

    if ( policyTestSetting.controllerAcquired )
    {
        nn::psm::AcquireControllerPowerSupply();
    }
    else
    {
        nn::psm::ReleaseControllerPowerSupply();
    }

    if ( policyTestSetting.enableFastBatteryCharging )
    {
        nn::psm::EnableFastBatteryCharging();
    }
    else
    {
        nn::psm::DisableFastBatteryCharging();
    }

    if ( enoughPowerChargeEmulation )
    {
        nnt::psm::CheckVdd50Condition((policyTestSetting.vdd50Condition == Vdd50Condition_None) ? Vdd50Condition_None : Vdd50Condition_Vdd50A);

        EXPECT_EQ(1200, nnt::psm::driver::detail::GetInputCurrentLimitMilliAmpere());
    }
    else
    {
        nnt::psm::CheckVdd50Condition(policyTestSetting.vdd50Condition);

        EXPECT_EQ(policyTestSetting.inputCurrentLimit, nnt::psm::driver::detail::GetInputCurrentLimitMilliAmpere());
    }

    EXPECT_EQ(policyTestSetting.inputVoltageLimitMilliVolt, nnt::psm::driver::detail::GetInputVoltageLimitMilliVolt());

    int expectedFastChargeCurrentLimit = std::min(policyTestSetting.fastChargeCurrentLimit, fastChargeCurrentLimit);
    EXPECT_EQ(expectedFastChargeCurrentLimit, nnt::psm::driver::detail::GetChargeCurrentLimitMilliAmpere());

    // MinimuｍAwake へ遷移させる。
    nnt::psm::ChangePmState(nn::psc::PmState_MinimumAwake);

    // SleepReady へ遷移させる。
    nnt::psm::ChangePmState(nn::psc::PmState_SleepReady);

    // EssentialServicesSleepReady へ遷移させる。
    nnt::psm::ChangePmState(nn::psc::PmState_EssentialServicesSleepReady);

    if ( enoughPowerChargeEmulation )
    {
        nnt::psm::CheckVdd50Condition((policyTestSetting.vdd50ConditionWithSleep == Vdd50Condition_None) ? Vdd50Condition_None : Vdd50Condition_Vdd50A);

        EXPECT_EQ(1200, nnt::psm::driver::detail::GetInputCurrentLimitMilliAmpere());
    }
    else
    {
        nnt::psm::CheckVdd50Condition(policyTestSetting.vdd50ConditionWithSleep);

        EXPECT_EQ(policyTestSetting.inputCurrentLimitWithSleep, nnt::psm::driver::detail::GetInputCurrentLimitMilliAmpere());
    }

    EXPECT_EQ(policyTestSetting.inputVoltageLimitMilliVolt, nnt::psm::driver::detail::GetInputVoltageLimitMilliVolt());

    int expectedFastChargeCurrentLimitWithSleep = std::min(policyTestSetting.fastChargeCurrentLimitWithSleep, fastChargeCurrentLimit);
    EXPECT_EQ(expectedFastChargeCurrentLimitWithSleep, nnt::psm::driver::detail::GetChargeCurrentLimitMilliAmpere());

    // EssentialServicesAwake へ遷移させる。
    nnt::psm::ChangePmState(nn::psc::PmState_EssentialServicesAwake);

    // SleepReady へ遷移させる。
    nnt::psm::ChangePmState(nn::psc::PmState_SleepReady);

    // MinimuｍAwake へ遷移させる。
    nnt::psm::ChangePmState(nn::psc::PmState_MinimumAwake);

    // FullAwake へ遷移させる。
    nnt::psm::ChangePmState(nn::psc::PmState_FullAwake);
}

} // namespace

// 常温、電池電圧十分時のテスト。
TEST_P(PsmPolicyTest, CheckChargeSettings)
{
    const bool enoughPowerChargeEmulation = false;

    ChangeEnoughPowerChargeEmulation(enoughPowerChargeEmulation);
    ChangeBatteryTemperature(nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdLow);
    ChangeBatteryVoltageMilliVolt(3320);

    CheckChargerAndVdd50(GetParam(), 2048, enoughPowerChargeEmulation);
}

// 低温、電池電圧十分時のテスト。充電電流の上限 768 を指定する。
TEST_P(PsmPolicyTest, CheckChargeSettingsLowTemperature)
{
    const bool enoughPowerChargeEmulation = false;

    ChangeEnoughPowerChargeEmulation(enoughPowerChargeEmulation);
    ChangeBatteryTemperature(nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdLow - 1.0);
    ChangeBatteryVoltageMilliVolt(3320);

    CheckChargerAndVdd50(GetParam(), 768, enoughPowerChargeEmulation);
}

// 常温、電池電圧不十分時のテスト。充電電流の上限 512 を指定する。
TEST_P(PsmPolicyTest, CheckChargeSettingsLowVoltage)
{
    const bool enoughPowerChargeEmulation = false;

    ChangeEnoughPowerChargeEmulation(enoughPowerChargeEmulation);
    ChangeBatteryTemperature(nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdLow);
    ChangeBatteryVoltageMilliVolt(3319);

    CheckChargerAndVdd50(GetParam(), 512, enoughPowerChargeEmulation);
}

// 常温、電池電圧十分時のテスト。強制ブーストモード用の EnoughPowerChargeEmulation を有効にする。
TEST_P(PsmPolicyTest, CheckChargeSettingsWithEnoughPowerChargeEmulation)
{
    const bool enoughPowerChargeEmulation = true;

    ChangeEnoughPowerChargeEmulation(enoughPowerChargeEmulation);
    ChangeBatteryTemperature(nn::psm::driver::detail::BatteryTemperatureCelsiusThresholdLow);
    ChangeBatteryVoltageMilliVolt(3320);

    CheckChargerAndVdd50(GetParam(), 2048, enoughPowerChargeEmulation);
}

}} // namespace nnt::psm
