﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/gpio/gpio.h>
#include <nn/result/result_HandlingUtility.h>

#include "psm_ChargerDriver.h"
#include "testPsm_ChargerDriverStub.h"

namespace {

const uint8_t FaultTypeBoostFault = 0x01;

struct ChargerStatus
{
    int  inputCurrentLimit;
    int  inputVoltageLimitMilliVolt;
    int  fastChargeCurrentLimit;
    int  outputCurrentLimit;
    int  chargeVoltageLimit;
    bool hiZCondition;
    bool isWatchdogTimerEnabled;
    ::nn::TimeSpan watchdogTimerPeriod;
    ::nn::psm::driver::detail::ChargerDriver::FastChargeCurrentLimitOption chargeCurrentLimitOption;

    uint8_t fault;

    ::nn::psm::driver::detail::ChargerDriver::Status status;
    ::nnt::psm::driver::detail::ChargeConfiguration chargeConfiguration;

    void Clear() NN_NOEXCEPT
    {
        inputCurrentLimit      = 0;
        inputVoltageLimitMilliVolt = 0;
        fastChargeCurrentLimit = 0;
        outputCurrentLimit     = 0;
        chargeVoltageLimit     = 0;
        hiZCondition           = false;

        chargeCurrentLimitOption = ::nn::psm::driver::detail::ChargerDriver::FastChargeCurrentLimitOption::None;

        fault                  = 0;

        status                 = ::nn::psm::driver::detail::ChargerDriver::Status::NotCharging;
        chargeConfiguration    = ::nnt::psm::driver::detail::ChargeConfiguration_Disable;
    }
};

ChargerStatus g_ChargerStatus;

} // namespace

namespace nn { namespace psm { namespace driver { namespace detail {

const ::nn::TimeSpan ChargerDriver::WatchdogTimerPeriodMax = ::nn::TimeSpan::FromSeconds(160);

ChargerDriver::ChargerDriver() NN_NOEXCEPT
    : m_PowerSupplySwitchingMutex(false)
{
    // 何もしません。
}

void ChargerDriver::Initialize() NN_NOEXCEPT
{
    g_ChargerStatus.Clear();

    // Charger IC のデフォルト値。
    // PSM が編集しない値および Initialize 関数内で上書きされる値は未実装。

    // Charger ドライバ内のデフォルト値。
    // PSM が編集しない値は未実装。
    g_ChargerStatus.chargeVoltageLimit = 4208;
    g_ChargerStatus.fastChargeCurrentLimit = 512;
    // Disable Force 20 Percent Charge Current
    // Set PreCharge Current Limit
    // Set Termination Current Limit
    // Set Minimum System Voltage Limit
    // Disable Watchdog Timer (set timer to 0 secs)
    // Disable Safety Timer
    // Reset I2C Watchdog Timer

    // PSM が上書きする値。
    g_ChargerStatus.inputCurrentLimit = 500;
    g_ChargerStatus.outputCurrentLimit = 500;
    g_ChargerStatus.hiZCondition = false;
    g_ChargerStatus.chargeConfiguration = ::nnt::psm::driver::detail::ChargeConfiguration_ChargeBattery;
    g_ChargerStatus.isWatchdogTimerEnabled = false;
    g_ChargerStatus.watchdogTimerPeriod = WatchdogTimerPeriodMax;

    // GPIO 初期化
    ::nn::gpio::Initialize();

    // 割込み用 GPIO 初期化
    ::nn::gpio::OpenSession(&m_ChargerInterruptSession, ::nn::gpio::GpioPadName_Bq24190Irq);
    ::nn::gpio::SetDirection(&m_ChargerInterruptSession, ::nn::gpio::Direction_Input);
    ::nn::gpio::SetInterruptMode(&m_ChargerInterruptSession, ::nn::gpio::InterruptMode_FallingEdge);
    ::nn::gpio::SetInterruptEnable(&m_ChargerInterruptSession, true);

    // SoC の電源が切れた場合に充電を有効に戻すために GPIO 経由で充電許可を操作する
    ::nn::gpio::OpenSession(&m_ChargeEnableSession, ::nn::gpio::GpioPadName_BattChgEnableN);
    ::nn::gpio::SetDirection(&m_ChargeEnableSession, ::nn::gpio::Direction_Output);
}

void ChargerDriver::Finalize() NN_NOEXCEPT
{
    ::nn::gpio::CloseSession(&m_ChargeEnableSession);

    // 割込み用 GPIO 終了処理
    ::nn::gpio::UnbindInterrupt(&m_ChargerInterruptSession);
    ::nn::gpio::CloseSession(&m_ChargerInterruptSession);

    // GPIO 終了処理
    ::nn::gpio::Finalize();
}

::nn::Result ChargerDriver::SetEnoughPowerChargeEmulation(bool isEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(SetupPowerSupply(m_IsOnTheGoEnabled, isEnabled));

    NN_RESULT_SUCCESS;
}

void ChargerDriver::SetUpUsbVbusChargingGpio() NN_NOEXCEPT
{
    // ポリシーテストの評価対象になっていないので実装を省略します。
}

::nn::Result ChargerDriver::EnableOnTheGo() NN_NOEXCEPT
{
    NN_RESULT_DO(SetupPowerSupply(true, m_IsEnoughPowerChargeEmulationEnabled));

    NN_RESULT_SUCCESS;
}

void ChargerDriver::SetUpVdd15HbChargingGpio() NN_NOEXCEPT
{
    // ポリシーテストの評価対象になっていないので実装を省略します。
}

::nn::Result ChargerDriver::EnableBatteryCharging() NN_NOEXCEPT
{
    NN_RESULT_DO(SetupPowerSupply(false, m_IsEnoughPowerChargeEmulationEnabled));

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetupPowerSupply(bool isOnTheGoRequested, bool isEnoughPowerChargeEmulationRequested) NN_NOEXCEPT
{
    if ( isOnTheGoRequested && !isEnoughPowerChargeEmulationRequested )
    {
        g_ChargerStatus.chargeConfiguration = ::nnt::psm::driver::detail::ChargeConfiguration_ChargeOTG;
    }
    else
    {
        g_ChargerStatus.chargeConfiguration = ::nnt::psm::driver::detail::ChargeConfiguration_ChargeBattery;
    }

    m_IsEnoughPowerChargeEmulationEnabled = isEnoughPowerChargeEmulationRequested;
    m_IsOnTheGoEnabled = isOnTheGoRequested;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::EnterHiZMode() NN_NOEXCEPT
{
    g_ChargerStatus.hiZCondition = true;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::ExitHiZMode() NN_NOEXCEPT
{
    g_ChargerStatus.hiZCondition = false;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetInputCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT
{
    NN_ASSERT(limitMilliAmpere >= 0);

    g_ChargerStatus.inputCurrentLimit = limitMilliAmpere;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetInputVoltageLimitMilliVolt(int inputVoltageLimitMilliVolt) NN_NOEXCEPT
{
    NN_ASSERT(inputVoltageLimitMilliVolt >= 0);

    g_ChargerStatus.inputVoltageLimitMilliVolt = inputVoltageLimitMilliVolt;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetFastChargeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT
{
    NN_ASSERT(limitMilliAmpere >= 0);

    g_ChargerStatus.fastChargeCurrentLimit = limitMilliAmpere;
    g_ChargerStatus.chargeCurrentLimitOption = ChargerDriver::FastChargeCurrentLimitOption::None;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetFastChargeCurrentLimit(int limitMilliAmpere, ChargerDriver::FastChargeCurrentLimitOption fastChargeCurrentLimitOption) NN_NOEXCEPT
{
    NN_UNUSED(fastChargeCurrentLimitOption);

    NN_ASSERT(limitMilliAmpere >= 0);

    g_ChargerStatus.fastChargeCurrentLimit = limitMilliAmpere;
    g_ChargerStatus.chargeCurrentLimitOption = fastChargeCurrentLimitOption;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetBoostModeCurrentLimit(int limitMilliAmpere) NN_NOEXCEPT
{
    NN_ASSERT(limitMilliAmpere >= 0);

    g_ChargerStatus.outputCurrentLimit = limitMilliAmpere;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::GetStatus(Status* pStatus) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pStatus);

    *pStatus = g_ChargerStatus.status;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetChargeVoltageLimit(int limitMilliVolt) NN_NOEXCEPT
{
    g_ChargerStatus.chargeVoltageLimit = limitMilliVolt;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::GetFaultStatus(FaultStatus* pFaultStatus) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pFaultStatus);

    pFaultStatus->Clear();
    pFaultStatus->isBoostFaultGenerated = (g_ChargerStatus.fault & FaultTypeBoostFault) == FaultTypeBoostFault;

    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetWatchdogTimerPeriod(::nn::TimeSpan period) NN_NOEXCEPT
{
    NN_ASSERT_MINMAX(
        period.GetSeconds(),
        static_cast<int64_t>(1),
        WatchdogTimerPeriodMax.GetSeconds());

    g_ChargerStatus.watchdogTimerPeriod = period;

    NN_RESULT_SUCCESS;
}

//!< ウォッチドッグを有効にします。
::nn::Result ChargerDriver::EnableWatchdogTimer() NN_NOEXCEPT
{
    g_ChargerStatus.isWatchdogTimerEnabled = true;

    NN_RESULT_SUCCESS;
}

//!< ウォッチドッグを無効にします。
::nn::Result ChargerDriver::DisableWatchdogTimer() NN_NOEXCEPT
{
    g_ChargerStatus.isWatchdogTimerEnabled = false;

    NN_RESULT_SUCCESS;
}

//!< ウォッチドッグをリセットします。
::nn::Result ChargerDriver::ResetWatchdogTimer() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

//!< ウォッチドッグが有効かどうか取得します。
bool ChargerDriver::IsWatchdogTimerEnabled() NN_NOEXCEPT
{
    return g_ChargerStatus.isWatchdogTimerEnabled;
}

::nn::Result ChargerDriver::BindInterrupt(::nn::os::SystemEventType* pOutInterruptEvent) NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::gpio::BindInterrupt(pOutInterruptEvent, &m_ChargerInterruptSession));
    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::ClearInterrupt(::nn::os::SystemEventType* pInterruptEvent) NN_NOEXCEPT
{
    // 連続で割り込みを受け付けないように最後に有効化します。
    ::nn::gpio::ClearInterruptStatus(&m_ChargerInterruptSession);
    ::nn::os::ClearSystemEvent(pInterruptEvent);
    ::nn::gpio::SetInterruptEnable(&m_ChargerInterruptSession, true);
    NN_RESULT_SUCCESS;
}

::nn::Result ChargerDriver::SetChargeEnable(bool enabled) NN_NOEXCEPT
{
    ::nn::gpio::SetValue(&m_ChargeEnableSession, enabled ? (::nn::gpio::GpioValue_Low) : (::nn::gpio::GpioValue_High));
    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::psm::driver::detail

namespace nnt { namespace psm { namespace driver { namespace detail {

int GetChargeCurrentLimitMilliAmpere() NN_NOEXCEPT
{
    return g_ChargerStatus.fastChargeCurrentLimit;
}

int GetInputCurrentLimitMilliAmpere() NN_NOEXCEPT
{
    return g_ChargerStatus.inputCurrentLimit;
}

int GetInputVoltageLimitMilliVolt() NN_NOEXCEPT
{
    return g_ChargerStatus.inputVoltageLimitMilliVolt;
}

int GetOutputCurrentLimitMilliAmpere() NN_NOEXCEPT
{
    return g_ChargerStatus.outputCurrentLimit;
}

int GetChargeVoltageLimitMilliVolt() NN_NOEXCEPT
{
    return g_ChargerStatus.chargeVoltageLimit;
}

bool IsWatchdogTimerEnabled() NN_NOEXCEPT
{
    return g_ChargerStatus.isWatchdogTimerEnabled;
}

::nnt::psm::driver::detail::ChargeConfiguration GetChargeConfiguration() NN_NOEXCEPT
{
    return g_ChargerStatus.chargeConfiguration;
}

::nn::psm::driver::detail::ChargerDriver::FastChargeCurrentLimitOption GetChargeCurrentLimitOption() NN_NOEXCEPT
{
    return g_ChargerStatus.chargeCurrentLimitOption;
}

void SetHiZCondition(bool hiZCondition) NN_NOEXCEPT
{
    g_ChargerStatus.hiZCondition = hiZCondition;
}

bool GetHiZCondition() NN_NOEXCEPT
{
    return g_ChargerStatus.hiZCondition;
}

void SetStatus(::nn::psm::driver::detail::ChargerDriver::Status status) NN_NOEXCEPT
{
    g_ChargerStatus.status = status;
}

}}}} // namespace nnt::psm::driver::detail
