﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/i2c/i2c.h>

#include <nnt/nntest.h>

namespace nnt { namespace psm {

class ChargerRegisterTest : public ::testing::Test
{
    virtual void SetUp() NN_OVERRIDE
    {
        nn::i2c::Initialize();
    }

    virtual void TearDown() NN_OVERRIDE
    {
        nn::i2c::Finalize();
    }
};

struct ExpectedEntry
{
    uint8_t address;
    uint8_t value;
    uint8_t mask;
};

// 期待するレジスタの値。
const ExpectedEntry g_Bq24193ChargerExpectedEntries[] =
{
    { 0x00, 0x32, 0x80 }, // [6:3]PINDPM は可変、[2:0]Ilim も可変、0x32 は VINDPM=4.36V, Ilim=500mA で非接続時の期待値（使用されないけど参考に残しておく）
    { 0x01, 0x00, 0xcf }, // [5:4]CHG_CONFIG は可変
    { 0x02, 0x00, 0x02 }, // [7:2]Ichg は可変、[0]FORCE_20PCT も可変
    { 0x03, 0x00, 0xff },
    { 0x04, 0x02, 0x03 }, // [7:2]Vreg は可変
    { 0x05, 0x82, 0xf9 }, // [2:1]CHG_TIMER はデフォルト値が期待値だが挙動には影響しないので無視
    { 0x06, 0x03, 0xff },
    { 0x07, 0x4b, 0xff },
    { 0x08, 0x00, 0x03 }, // 温度と Vsys は見る
    { 0x09, 0x00, 0xf8 }, // [3:0]NTC_FAULT Hot を取りうる
    { 0x0a, 0x2f, 0xff },
};

// システム起動後の bq24193 レジスタの値が期待通りであることの確認。
TEST_F(ChargerRegisterTest, Bq24193Test)
{
    nn::i2c::I2cSession session;
    nn::i2c::OpenSession(&session, nn::i2c::I2cDevice_Bq24193);

    // SIGLO-48880: HasDevice の代替。
    uint8_t value = 0x00;
    const uint8_t Address = 0x0a;
    nn::Result result = nn::i2c::ReadSingleRegister(&value, session, &Address);
    if ( result <= nn::i2c::ResultNoAck() )
    {
        NN_LOG("The platform doesn't have bq24913.\n");
        return;
    }

    for ( const ExpectedEntry& expected : g_Bq24193ChargerExpectedEntries )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&value, session, &(expected.address)));

        EXPECT_EQ(expected.value & expected.mask, value & expected.mask);
        if ( (expected.value & expected.mask) != (value & expected.mask) )
        {
            NN_LOG("0x%02x:0x%02x should be 0x%02x(mask:0x%02x).\n", expected.address, value, expected.value, expected.mask);
        }

        // SIGLO-81834: REG00 の VINDPM のチェック
        // 期待値は Mask:0x78 で 0x00 か 0x30
        if ( expected.address == 0x00 )
        {
            EXPECT_TRUE(((value & 0x78) == 0x00) || ((value & 0x78) == 0x30));
            if ( !(((value & 0x78) == 0x00) || ((value & 0x78) == 0x30)) )
            {
                NN_LOG("0x00:0x%02x should be 0x00 or 0x30(mask:0x78).\n", value);
            }
        }
    }

    nn::i2c::CloseSession(session);
}

}} // namespace nnt::psm
