﻿param(
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [Nintendo.CPU.Profiler.Scripting.NX.ScriptingSampleMode]$SampleMode = [Nintendo.CPU.Profiler.Scripting.NX.ScriptingSampleMode]::InProcess,
    
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [string]$Platform,
    
    [Parameter(ValueFromPipelineByPropertyName = $true)]
    [string]$BuildType
)

. "$([Nintendo.CPU.Profiler.Scripting.ScriptingCore]::ScriptDirectory)/testProfiler_Common.ps1"

write-output "Starting testProfiler_ThreadMigration"

[string[]]$nvnFunctionList = @()

function LoadNvnFunctionList()
{
    $containedFailure = $false
    
    $scriptDir = [Nintendo.CPU.Profiler.Scripting.ScriptingCore]::ScriptDirectory | resolve-path
    $sigloSdkRoot = "$scriptDir/../../../../" | resolve-path
    $nvnSrcFolder = "$sigloSdkRoot/Externals/odin2/common-headers/nvn/nvn" | resolve-path
    if ($false -eq (test-path $nvnSrcFolder))
    {
        WriteError "Could not find NVN Src Folder: $nvnSrcFolder"
        return $true
    }
    
    $nvnCsrc = "$nvnSrcFolder/nvn_FuncPtrImpl.h"
    if ($false -eq (test-path $nvnCsrc))
    {
        WriteError "Could not find: $nvnCsrc"
        return $true
    }
    
    
    [regex]$searchFuncNames = [regex]'.*=.*\"(.*)\"'
    $fileContents = [string[]](get-content $nvnCsrc)
    foreach ($line in $fileContents)
    {
        $match = $searchFuncNames.Match($line)
        if ($match.Success)
        {
            # Leading ',' enforces treating this as an array
            #   (not needed with initialization, but here to be safe)
            $global:nvnFunctionList += ,$match.Groups[1].Value
        }
    }
    
    return $containedFailure
}

function CheckForNvnFunctions()
{
    [string[]]$missingFunctions = @()
    
    profiler-showtab CodeCoverage
    
    profiler-filter CodeCoverage -Negate $false -RegEx $true -FilterName "^nvn"
    $notSeenFunctions = profiler-copy CodeCoverage NotSeen
    
    [string[]]$notSeenFunctionList = $notSeenFunctions -split [Environment]::NewLine
    
    foreach ($nvnFunc in $nvnFunctionList)
    {
        if (($notSeenFunctionList -contains $nvnFunc) -eq $false)
        {
            $missingFunctions += ,$nvnFunc
        }
    }
    
    return $missingFunctions
}

function BaseProfile()
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count
    
    SyncWhenAvailable
    profiler-waitfor ReadyToProfile
    
    profiler-start Sampled
    profiler-wait 1.0
    profiler-stop
    
    $missingFunctionList = CheckForNvnFunctions
    if ($missingFunctionList.Count -gt 0)
    {
        $containedFailure = $false
    }
    else
    {
        $containedFailure = $true
    }
    
    profiler-unsync
    sleep 1
    
    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }
    
    return $containedFailure
}

function NvnCFunctions()
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count
    
    SyncWhenAvailable
    profiler-waitfor ReadyToProfile
    
    profiler-start Sampled
    profiler-wait 1.0
    profiler-stop
    
    $missingFunctionList = CheckForNvnFunctions
    if ($missingFunctionList -eq $null -or $missingFunctionList.Count -lt ($global:nvnFunctionList.Count / 2))
    {
        $containedFailure = $false
    }
    else
    {
        $containedFailure = $true
        WriteError "Missing NVN Functions"
        foreach ($func in $missingFunctionList)
        {
            write-output "Missing: $func"
        }
    }
    
    profiler-unsync
    sleep 1
    
    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }
    
    return $containedFailure
}

function UninitializedNvn()
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count
    
    SyncWhenAvailable
    profiler-waitfor ReadyToProfile
    
    profiler-start Sampled
    profiler-wait 1.0
    profiler-stop
    
    $missingFunctionList = CheckForNvnFunctions
    if ($missingFunctionList -eq $null -or $missingFunctionList.Count -gt 0)
    {
        $containedFailure = $false
    }
    else
    {
        $containedFailure = $true
    }
    
    profiler-unsync
    sleep 1
    
    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }
    
    return $containedFailure
}

function NvnCppFunctions()
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count
    
    SyncWhenAvailable
    profiler-waitfor ReadyToProfile
    
    profiler-start Sampled
    profiler-wait 1.0
    profiler-stop
    
    $missingFunctionList = CheckForNvnFunctions
    if ($missingFunctionList -eq $null -or $missingFunctionList.Count -lt ($global:nvnFunctionList.Count / 2))
    {
        $containedFailure = $false
    }
    else
    {
        $containedFailure = $true
        WriteError "Missing NVN Functions"
        foreach ($func in $missingFunctionList)
        {
            write-output "Missing: $func"
        }
    }
    
    profiler-unsync
    sleep 1
    
    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }
    
    return $containedFailure
}

function CheckSvcFunctions()
{
    $containedFailure = $false
    $warningCount = [Nintendo.SDSG.Utilities.Logging]::Warnings.Count
    
    profiler-showtab CodeCoverage
    
    profiler-filter CodeCoverage -Negate $false -RegEx $true -FilterName "^nn::svc::profiler::"
    [string[]]$svcOverridesList = (profiler-copy CodeCoverage NotSeen) -split [Environment]::NewLine
    [regex]$searchFunctionName = [regex]'^nn::svc::profiler::([A-Za-z0-9_]+).*'
    
    [string[]]$svcNameList = @()
    foreach ($line in $svcOverridesList)
    {
        $match = $searchFunctionName.Match($line)
        if ($match.Success)
        {
            $svcNameList += ,$match.Groups[1]
        }
    }
    
    foreach ($name in $svcNameList)
    {
        $filterName = "^nn::svc.*::$name\b.*\)$"
        profiler-showtab CodeCoverage
        profiler-filter CodeCoverage -Negate $false -RegEx $true -FilterName $filterName
        write-output "Searched for: $filterName"
        [string[]]$foundSvc = (profiler-copy CodeCoverage NotSeen) -split [Environment]::NewLine
        write-output $foundSvc
        [string[]]$lastAsm = @()
        profiler-showtab assembly
        foreach ($svc in $foundSvc)
        {
            if ($svc -match $filterName)
            {
                write-output "Checking: $svc"
                profiler-assembly -FunctionName "$svc"
                [string[]]$rawAssembly = (profiler-copy Assembly) -split [Environment]::NewLine
                [regex]$asmRegex = [regex]'0x[A-Fa-f0-9]+\s*.*([A-Fa-f0-9]{8}.*)'
                [string[]]$asmCurrent = @()
                foreach ($line in $rawAssembly)
                {
                    $match = $asmRegex.Match($line)
                    if ($match.Success)
                    {
                        $asmCurrent += ,$match.Groups[1]
                    }
                }
                if ($lastAsm.Length -gt 0)
                {
                    if ($lastAsm.Length -eq $asmCurrent.Length)
                    {
                        for ([int]$i = 0; $i -lt $lastAsm.Length; ++$i)
                        {
                            if ($lastAsm[$i].CompareTo($asmCurrent[$i]) -ne 0)
                            {
                                $lastLine = $lastAsm[$i]
                                $currLine = $asmCurrent[$i]
                                write-output "$lastLine"
                                write-output "$currLine"
                                $containedFailure = $true
                            }
                        }
                    }
                    else
                    {
                        $containedFailure = $true
                    }
                }
                $lastAsm = $asmCurrent
            }
        }
    }
    
    if ([Nintendo.SDSG.Utilities.Logging]::Warnings.Count -gt $warningCount)
    {
        WriteError "Generated a warning"
        $containedFailure = $true
    }
    
    return $containedFailure
}

RunTest("LoadNvnFunctionList")
RunTest("BaseProfile")
RunTest("NvnCFunctions")
RunTest("UninitializedNvn")
RunTest("NvnCppFunctions")
RunTest("CheckSvcFunctions")

TestsFinished
