﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/socket.h>
#include <nn/nifm/nifm_ApiForSystem.h>

#include <nn/prepo/prepo_ApiConfig.h>
#include <nn/prepo/prepo_ResultPrivate.h>
#include <nn/prepo/detail/service/core/prepo_CategoryObject.h>
#include <nn/prepo/detail/service/core/prepo_FileSystem.h>
#include <nn/prepo/detail/service/core/prepo_StatisticsManager.h>
#include <nn/prepo/detail/service/core/prepo_UserAgreementChecker.h>

#include "../Common/testPrepo_Common.h"

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory = {};

    int g_UserCount = 0;
    nn::account::Uid g_Users[nn::account::UserCountMax] = {};
}

namespace
{
    void Initialize() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
        nn::account::InitializeForSystemService();
        nn::prepo::detail::service::core::FileSystem::EnableTestMode();
        nn::prepo::detail::service::core::FileSystem::MountAll();
        nn::prepo::detail::service::core::SystemInfo::SetOperationMode(0);
    }

    void Finalize() NN_NOEXCEPT
    {
        nn::prepo::detail::service::core::FileSystem::UnmountAll();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Finalize());
    }

    nn::Result WriteReport(nn::prepo::detail::service::core::ReportFileManager& manager, const nn::account::Uid& user) NN_NOEXCEPT
    {
        const char eventId[] = "test_prepo_upload_executer";
        const nn::ApplicationId appId = { 0x0123456789ABCDEF };

        nn::Bit8 buffer[2 * 1024];

        nnt::prepo::ReportWriter writer(buffer, sizeof(buffer));

        NN_RESULT_DO(writer.Initialize(eventId, appId, user));
        NN_RESULT_DO(writer.Add("x", 1.23));
        NN_RESULT_DO(writer.Add("y", 4.56));
        NN_RESULT_DO(writer.Add("z", 7.89));

        nn::prepo::detail::service::ReportDataSummary summary;
        NN_RESULT_DO(manager.WriteFile(&summary, buffer, writer.GetSize(), 1));

        NN_RESULT_SUCCESS;
    }

    nn::Result WriteReport(nn::prepo::detail::service::core::ReportFileManager& manager) NN_NOEXCEPT
    {
        return WriteReport(manager, nn::account::InvalidUid);
    }
}

class UploadExecuterTest : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        Initialize();

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&g_UserCount,
            g_Users, NN_ARRAY_SIZE(g_Users)));
        NN_ABORT_UNLESS_EQUAL(g_UserCount, 2);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().Update());

        // user0 は、情報送信に同意していない。
        nn::account::NetworkServiceAccountId account0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::prepo::GetNetworkServiceAccountId(&account0, g_Users[0]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsUserAgreedWithNaEula(account0));
        NN_ABORT_UNLESS(nn::prepo::ResultTransmissionNotAgreed::Includes(
            nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsUserAgreedWithAnalystics(account0)));

        // user1 は、情報送信に同意している。
        nn::account::NetworkServiceAccountId account1;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::prepo::GetNetworkServiceAccountId(&account1, g_Users[1]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsUserAgreedWithNaEula(account1));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsUserAgreedWithAnalystics(account1));

        // user0, 1 のいずれも NA 連携しているので、いずれかのユーザーが NA EULA に同意していると判定される。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsAnyoneAgreedWithNaEula());

        // user1 が情報送信に同意しているので、いずれかのユーザーが情報送信に同意していると判定される。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsAnyoneAgreedWithAnalystics());
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        Finalize();
    }
};

TEST_F(UploadExecuterTest, Normal)
{
    const auto category = nn::prepo::ReportCategory_Normal;

    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo");

    auto executeUpload = nn::prepo::detail::service::core::CategoryObject::ExecuteUpload;
    auto&& statisticsManager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();
    auto&& reportFileManager = nn::prepo::detail::service::core::CategoryObject::GetReportFileManager(category);

    // 情報送信に同意していないユーザーのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[0]));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(1, statistics.groups[category].lostByDisagree.count);
    }

    // 情報送信に同意しているユーザーのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[1]));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(1, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }

    // ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(1, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }

    // 情報送信に同意していないユーザーと、同意しているユーザーと、ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[0]));
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[1]));
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(2, statistics.groups[category].uploaded.count);
        EXPECT_EQ(1, statistics.groups[category].lostByDisagree.count);
    }
}

TEST_F(UploadExecuterTest, WithoutUserAgreementCheck)
{
    const auto category = nn::prepo::ReportCategory_AntiPiracy;

    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo-ap");

    auto executeUpload = nn::prepo::detail::service::core::CategoryObject::ExecuteUpload;
    auto&& statisticsManager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();
    auto&& reportFileManager = nn::prepo::detail::service::core::CategoryObject::GetReportFileManager(category);

    // 情報送信に同意していないユーザーのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[0]));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(1, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }

    // 情報送信に同意しているユーザーのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[1]));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(1, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }

    // ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(1, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }

    // 情報送信に同意していないユーザーと、同意しているユーザーと、ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[0]));
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[1]));
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(3, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }
}

class UploadExecuterTestNoAgreedUser : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        Initialize();

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&g_UserCount,
            g_Users, NN_ARRAY_SIZE(g_Users)));
        NN_ABORT_UNLESS_EQUAL(g_UserCount, 2);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().Update());

        // user0 は、情報送信に同意していない。
        nn::account::NetworkServiceAccountId account0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nnt::prepo::GetNetworkServiceAccountId(&account0, g_Users[0]));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsUserAgreedWithNaEula(account0));
        NN_ABORT_UNLESS(nn::prepo::ResultTransmissionNotAgreed::Includes(
            nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsUserAgreedWithAnalystics(account0)));

        // user1 は、NA 連携していない。
        nn::account::NetworkServiceAccountId account1;
        NN_ABORT_UNLESS(nn::account::ResultNetworkServiceAccountUnavailable::Includes(nnt::prepo::GetNetworkServiceAccountId(&account1, g_Users[1])));

        // user0 は NA 連携しているので、いずれかのユーザーが NA EULA に同意していると判定される。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsAnyoneAgreedWithNaEula());

        // user0 が情報送信に同意していないので、いずれのユーザーも情報送信に同意していないと判定される。
        NN_ABORT_UNLESS(nn::prepo::ResultTransmissionNotAgreed::Includes(
            nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsAnyoneAgreedWithAnalystics()));
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        Finalize();
    }
};

TEST_F(UploadExecuterTestNoAgreedUser, Normal)
{
    const auto category = nn::prepo::ReportCategory_Normal;

    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo");

    auto executeUpload = nn::prepo::detail::service::core::CategoryObject::ExecuteUpload;
    auto&& statisticsManager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();
    auto&& reportFileManager = nn::prepo::detail::service::core::CategoryObject::GetReportFileManager(category);

    // 情報送信に同意していないユーザーのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[0]));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count); // 一人も情報送信に同意していないときは、レポートを破棄しない。
    }

    // ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count); // 一人も情報送信に同意していないときは、レポートを破棄しない。
    }

    // 情報送信に同意していないユーザーと、ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[0]));
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count); // 一人も情報送信に同意していないときは、レポートを破棄しない。
    }
}

TEST_F(UploadExecuterTestNoAgreedUser, WithoutUserAgreementCheck)
{
    const auto category = nn::prepo::ReportCategory_AntiPiracy;

    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo-ap");

    auto executeUpload = nn::prepo::detail::service::core::CategoryObject::ExecuteUpload;
    auto&& statisticsManager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();
    auto&& reportFileManager = nn::prepo::detail::service::core::CategoryObject::GetReportFileManager(category);

    // 情報送信に同意していないユーザーのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[0]));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(1, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }

    // ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(1, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }

    // 情報送信に同意していないユーザーと、ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager, g_Users[0]));
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(2, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count);
    }
}

class UploadExecuterTestNoNaLinkedUser : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        Initialize();

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListAllUsers(&g_UserCount,
            g_Users, NN_ARRAY_SIZE(g_Users)));
        NN_ABORT_UNLESS_EQUAL(g_UserCount, 2);

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().Update());

        // user0 は、NA 連携していない。
        nn::account::NetworkServiceAccountId account0;
        NN_ABORT_UNLESS(nn::account::ResultNetworkServiceAccountUnavailable::Includes(nnt::prepo::GetNetworkServiceAccountId(&account0, g_Users[0])));

        // user1 は、NA 連携していない。
        nn::account::NetworkServiceAccountId account1;
        NN_ABORT_UNLESS(nn::account::ResultNetworkServiceAccountUnavailable::Includes(nnt::prepo::GetNetworkServiceAccountId(&account1, g_Users[1])));

        // 誰も NA 連携していないので、いずれのユーザーも NA EULA に同意していないと判定される。
        NN_ABORT_UNLESS(nn::prepo::ResultTransmissionNotAgreed::Includes(
            nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsAnyoneAgreedWithNaEula()));

        // 誰も NA 連携していないので、いずれのユーザーも情報送信に同意していないと判定される。
        NN_ABORT_UNLESS(nn::prepo::ResultTransmissionNotAgreed::Includes(
            nn::prepo::detail::service::core::UserAgreementChecker::GetInstance().IsAnyoneAgreedWithAnalystics()));
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        Finalize();
    }
};

TEST_F(UploadExecuterTestNoNaLinkedUser, Normal)
{
    const auto category = nn::prepo::ReportCategory_Normal;

    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo");

    auto executeUpload = nn::prepo::detail::service::core::CategoryObject::ExecuteUpload;
    auto&& statisticsManager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();
    auto&& reportFileManager = nn::prepo::detail::service::core::CategoryObject::GetReportFileManager(category);

    // ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count); // 一人も NA 連携してないときは、レポートを破棄しない。
    }
}

TEST_F(UploadExecuterTestNoNaLinkedUser, WithoutUserAgreementCheck)
{
    const auto category = nn::prepo::ReportCategory_AntiPiracy;

    nn::prepo::detail::service::core::FileSystem::ClearStorage("prepo-ap");

    auto executeUpload = nn::prepo::detail::service::core::CategoryObject::ExecuteUpload;
    auto&& statisticsManager = nn::prepo::detail::service::core::StatisticsManager::GetInstance();
    auto&& reportFileManager = nn::prepo::detail::service::core::CategoryObject::GetReportFileManager(category);

    // ユーザー指定なしのレポートを送信。
    {
        NNT_ASSERT_RESULT_SUCCESS(WriteReport(reportFileManager));

        statisticsManager.Clear();

        NNT_ASSERT_RESULT_SUCCESS(executeUpload(true));

        nn::prepo::Statistics statistics;
        statisticsManager.GetStatistics(&statistics);

        EXPECT_EQ(0, statistics.groups[category].uploaded.count);
        EXPECT_EQ(0, statistics.groups[category].lostByDisagree.count); // 一人も NA 連携してないときは、レポートを破棄しない。
    }
}
