﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include <string>
#include <sstream>

#include <nnt.h>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/prepo/detail/service/core/prepo_Storage.h>

#include "../Common/testPrepo_Common.h"

namespace
{
    const nn::fs::SystemSaveDataId TestStorageSaveDataId = 0x80000000000000AF;
    const int64_t TestStorageSize = 10 * 1024 * 1024;
    const int64_t TestStorageJournalSize = 1 * 1024 * 1024;
    const char TestStorageName[] = "prepo-test";
}

namespace
{
    const size_t FileSizeMax = 128 * 1024;
}

namespace
{
    nn::prepo::detail::service::core::Storage& GetTestStorage() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::prepo::detail::service::core::Storage, s_Storage,
            = { TestStorageName, TestStorageSaveDataId, TestStorageSize + TestStorageJournalSize, TestStorageJournalSize });
        return s_Storage;
    }
}

namespace
{
    int64_t GetGenuineFreeSpaceSize() NN_NOEXCEPT
    {
        return GetTestStorage().GetFreeSpaceSize();
    }

    int64_t GetEmulatedFreeSpaceSize() NN_NOEXCEPT
    {
        return TestStorageSize - GetTestStorage().GetUsage();
    }
}

TEST(Initialize, Main)
{
    NN_RESULT_ABORTING_BLOCK
    {
        nn::fs::SetEnabledAutoAbort(false);
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::SetEnabledAutoAbort(true);
        };

        NN_RESULT_TRY(nn::fs::DeleteSaveData(TestStorageSaveDataId))
            NN_RESULT_CATCH(nn::fs::ResultTargetNotFound)
            {
            }
        NN_RESULT_END_TRY;
        NN_RESULT_DO(GetTestStorage().Mount());
        NN_RESULT_SUCCESS;
    };
}

TEST(GetStorageUsage, CompareWithFsGetFreeSpaceSize)
{
    auto freeSpaceSize = GetGenuineFreeSpaceSize();

    EXPECT_EQ(freeSpaceSize, GetEmulatedFreeSpaceSize());

    std::vector<std::string> directories;
    directories.push_back(std::string(TestStorageName) + ":");

    for (int i = 1; ; i++)
    {
        char entryName[8 + 1];
        NN_ABORT_UNLESS(nn::util::SNPrintf(entryName, sizeof (entryName), "%08d", i) < sizeof (entryName));

        std::string path;
        do
        {
            auto directory = directories[nnt::prepo::GetRandom<uint64_t>() % directories.size()];
            path = directory + "/" + entryName;

        } while (!(path.size() <= nn::fs::EntryNameLengthMax));

        nn::Result result;
        std::stringstream message;

        if (nnt::prepo::GetRandom<uint64_t>() % 2 == 0)
        {
            const auto fileSize = nnt::prepo::GetRandom<size_t>() % (FileSizeMax + 1);
            result = nn::fs::CreateFile(path.c_str(), fileSize);
            message << "Create a file.      (path=" << path << ", size=" << fileSize << ")";
        }
        else
        {
            result = nn::fs::CreateDirectory(path.c_str());
            directories.push_back(path);
            message << "Create a directory. (path=" << path << ")";
        }

        if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
        {
            break;
        }

        NNT_ASSERT_RESULT_SUCCESS(result);

        NNT_ASSERT_RESULT_SUCCESS(GetTestStorage().Commit());

        const auto newFreeSpaceSize = GetGenuineFreeSpaceSize();

        NN_LOG("%8lld -> %8lld (%8lld) | %s\n", freeSpaceSize, newFreeSpaceSize, newFreeSpaceSize - freeSpaceSize, message.str().c_str());

        EXPECT_EQ(newFreeSpaceSize, GetEmulatedFreeSpaceSize());

        freeSpaceSize = newFreeSpaceSize;
    }
}

TEST(Finalize, Main)
{
    GetTestStorage().Unmount();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::DeleteSaveData(TestStorageSaveDataId));
}
