﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/prepo.h>
#include <nn/prepo/prepo_SystemPlayReport.h>
#include <nn/prepo/prepo_ApiAdmin.h>
#include <nn/prepo/prepo_ApiConfig.h>
#include <nn/prepo/prepo_ApiDebug.h>
#include <cstdlib>
#include "../Common/testPrepo_Common.h"

namespace
{
    nn::nifm::NetworkConnection& GetNetworkConnection() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(nn::nifm::NetworkConnection, connection);
        return connection;
    }

    void SavePlayReport(const char* eventId)
    {
        nn::prepo::SystemPlayReport report(eventId);
        char buffer[1024];

        nn::ApplicationId appId = {0x0100000000003006};

        report.SetApplicationId(appId);
        report.SetBuffer(buffer, sizeof (buffer));

        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:x", 1.23));
        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:y", 0.00));
        NNT_EXPECT_RESULT_SUCCESS(report.Add("position:z", -5.55));

        report.Save();
    }
}

TEST(RequestImmediateTransmission, Initialize)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::ClearStorage());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(false));

    NN_LOG("Wait for completion of upload playreport by PDM.\n");
    for (int i = 10; i >= 0; i--)
    {
        NN_LOG("%d\n", i);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_LOG("Request network connection from application.\n");
    nn::nifm::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(GetNetworkConnection().GetRequestHandle(),
        nn::nifm::RequirementPreset_InternetBestEffort));
    GetNetworkConnection().SubmitRequestAndWait();
    ASSERT_TRUE(GetNetworkConnection().IsAvailable());
}

TEST(RequestImmediateTransmission, Save1)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::ClearStorage());

    NN_LOG("Save play report.\n");
    SavePlayReport("immediate_transmission1");

    ASSERT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(
        nn::prepo::TransmissionStatus_Pending, nn::TimeSpan::FromSeconds(20)));

    int64_t used;
    int64_t capacity;
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::GetStorageUsage(&used, &capacity));
    EXPECT_GT(used, 0);
}

TEST(RequestImmediateTransmission, SaveWithRequestImmediateTransmission1)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::ClearStorage());

    NN_LOG("Save play report.\n");
    SavePlayReport("immediate_transmission2");
    NN_LOG("Request immediate transmission.\n");
    NNT_EXPECT_RESULT_SUCCESS(nn::prepo::RequestImmediateTransmission());

    ASSERT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(
        nn::prepo::TransmissionStatus_Processing, nn::TimeSpan::FromSeconds(20)));

    ASSERT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(
        nn::prepo::TransmissionStatus_Idle, nn::TimeSpan::FromSeconds(20)));

    int64_t used;
    int64_t capacity;
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::GetStorageUsage(&used, &capacity));
    // EXPECT_EQ(0, used); // 他のアプリケーションのプレイレポートを書き込みにより、ストレージ使用量が 0 にならないことがある。
}

TEST(RequestImmediateTransmission, Save2)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::ClearStorage());

    NN_LOG("Save play report.\n");
    SavePlayReport("immediate_transmission3");

    ASSERT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(
        nn::prepo::TransmissionStatus_Pending, nn::TimeSpan::FromSeconds(20)));

    int64_t used;
    int64_t capacity;
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::GetStorageUsage(&used, &capacity));
    EXPECT_GT(used, 0);
}

TEST(RequestImmediateTransmission, SaveWithRequestImmediateTransmission2)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::prepo::ClearStorage());

    NN_LOG("Save play report.\n");
    SavePlayReport("immediate_transmission4");

    NN_LOG("Sleep 5 seconds...\n"); // ファイルへの書き出しを待つ。

    for (int i = 5; i >= 0; i--)
    {
        NN_LOG("%d\n", i);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_LOG("Sleep 5 seconds done!\n");

    ASSERT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(
        nn::prepo::TransmissionStatus_Pending, nn::TimeSpan::FromSeconds(20)));

    int64_t used;
    int64_t capacity;

    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::GetStorageUsage(&used, &capacity));
    EXPECT_GT(used, 0);

    // ファイルへの書き出しが完了した後に即時実行要求を出す。
    NN_LOG("Request immediate transmission.\n");
    NNT_EXPECT_RESULT_SUCCESS(nn::prepo::RequestImmediateTransmission());

    ASSERT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(
        nn::prepo::TransmissionStatus_Processing, nn::TimeSpan::FromSeconds(20)));

    ASSERT_TRUE(nnt::prepo::WaitForTransmissionStatusChanged(
        nn::prepo::TransmissionStatus_Idle, nn::TimeSpan::FromSeconds(20)));

    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::GetStorageUsage(&used, &capacity));
    // EXPECT_EQ(0, used); // 他のアプリケーションのプレイレポートを書き込みにより、ストレージ使用量が 0 にならないことがある。
}

TEST(RequestImmediateTransmission, Finalize)
{
    GetNetworkConnection().CancelRequest();

    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::SetUserAgreementCheckEnabled(true));
    NNT_ASSERT_RESULT_SUCCESS(nn::prepo::ClearStorage());
}
