﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/fgm/fgm.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntrusiveList.h>

namespace {

struct StaticMutex
{
    ::nn::os::MutexType mutex;

    void lock() NN_NOEXCEPT
    {
        ::nn::os::LockMutex(&mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        ::nn::os::UnlockMutex(&mutex);
    }
};

StaticMutex g_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

class SessionContext : public nn::util::IntrusiveListBaseNode<SessionContext>
{
public:
    explicit SessionContext(const void* objectAddress) NN_NOEXCEPT
    : m_ObjectAddress(objectAddress)
    , m_Module(nn::fgm::Module_Test)
    , m_Priority(nn::fgm::Priority_Default)
    , m_Min(static_cast<nn::fgm::Setting>(nn::fgm::Setting_Min))
    , m_Max(static_cast<nn::fgm::Setting>(nn::fgm::Setting_Max))
    {
    }

    void Initialize(nn::fgm::Module module, nn::fgm::Priority priority) NN_NOEXCEPT
    {
        m_Module = module;
        m_Priority = priority;
    }

    void Set(nn::fgm::Setting min, nn::fgm::Setting max) NN_NOEXCEPT
    {
        m_Min = min;
        m_Max = max;
    }

    const void* GetObjectAddress() NN_NOEXCEPT
    {
        return m_ObjectAddress;
    }

    nn::fgm::Module GetModule() NN_NOEXCEPT
    {
        return m_Module;
    }

    nn::fgm::Priority GetPriority() NN_NOEXCEPT
    {
        return m_Priority;
    }

    nn::fgm::Setting GetMin() NN_NOEXCEPT
    {
        return m_Min;
    }

    nn::fgm::Setting GetMax() NN_NOEXCEPT
    {
        return m_Max;
    }

//private:
public:
    const void* m_ObjectAddress;
    nn::fgm::Module m_Module;
    nn::fgm::Priority m_Priority;
    nn::fgm::Setting m_Min;
    nn::fgm::Setting m_Max;
};

nn::util::IntrusiveList<SessionContext, nn::util::IntrusiveListBaseNodeTraits<SessionContext>> g_SessionContextList;

const size_t MaxSessionContexts = 30;

SessionContext* FindSessionContext(const void* objectAddress) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( (*itr).GetObjectAddress() == objectAddress )
        {
            return &(*itr);
        }
    }

    return nullptr;
}

// 必要に応じて改善する。Priority 順に並ぶべき。
void CreateSessionContext(const void* objectAddress, nn::fgm::Module module, nn::fgm::Priority priority) NN_NOEXCEPT
{
    NN_ASSERT(g_SessionContextList.size() < MaxSessionContexts)

    NN_ASSERT_EQUAL(nullptr, FindSessionContext(objectAddress));

    SessionContext* pSessionContext = new SessionContext(objectAddress);

    g_SessionContextList.push_front(*pSessionContext);
    pSessionContext->Initialize(module, priority);
}

void DestroySessionContext(const void* objectAddress) NN_NOEXCEPT
{
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( (*itr).GetObjectAddress() == objectAddress )
        {
            g_SessionContextList.erase(itr);
            delete &(*itr);

            return;
        }
    }

    NN_ASSERT(false);
}

// 必要に応じて改善する。max, min の範囲を評価するべき。
nn::fgm::Setting GetCurrentSetting(const void* objectAddress) NN_NOEXCEPT
{
    SessionContext* pSessionContext = FindSessionContext(objectAddress);

    NN_ASSERT_NOT_NULL(pSessionContext);

    nn::fgm::Module targetModule = pSessionContext->GetModule();
    nn::fgm::Priority currentPriority = nn::fgm::Priority_Lowest;
    nn::fgm::Setting min = static_cast<nn::fgm::Setting>(nn::fgm::Setting_Min);

    // Stub なので雑に見る。全 Request の矛盾はない想定。下側しか見ない。
    for ( auto itr = g_SessionContextList.begin(); itr != g_SessionContextList.end(); itr++ )
    {
        if ( (*itr).GetModule() == targetModule && (*itr).GetPriority() <= currentPriority && (*itr).GetMin() > min )
        {
            min = (*itr).GetMin();
        }
    }

    return min;
}

} // namespace

namespace nn { namespace fgm {

// Request の番地自体をキーにして管理情報を手に入れる。
Request::Request() NN_NOEXCEPT
{
}

Request::~Request() NN_NOEXCEPT
{
}

nn::Result Request::Initialize(Module id, Priority priority) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    CreateSessionContext(this, id, priority);

    NN_RESULT_SUCCESS;
}

nn::Result Request::SetAndWait(Setting min, Setting max) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    FindSessionContext(this)->Set(min, max);

    NN_RESULT_SUCCESS;
}

nn::Result Request::Get(Setting* pCurrent) NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    NN_ASSERT_NOT_NULL(pCurrent);

    *pCurrent = GetCurrentSetting(this);

    NN_RESULT_SUCCESS;
}

nn::Result Request::Finalize() NN_NOEXCEPT
{
    ::std::lock_guard<StaticMutex> lock(g_Mutex);

    DestroySessionContext(this);

    NN_RESULT_SUCCESS;
}

}} // namespace nn::fgm
