﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/pcv/pcv.h>
#include <nn/nn_Common.h>
#include <nnt/nntest.h>

class SetVoltageValueTest : public ::testing::TestWithParam<nn::pcv::PowerDomain>{};

const nn::pcv::PowerDomain supportedPDs[] =
{
    // Insert safe power domains to set voltage values
    nn::pcv::PowerDomain_Max77620_Ldo2,
};

INSTANTIATE_TEST_CASE_P(SetVoltageModules,
    SetVoltageValueTest,
    ::testing::ValuesIn(supportedPDs));

TEST_P(SetVoltageValueTest, PowerDomain)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());

    bool enabled;
    nn::Result result;

    //Get Range of voltage and step voltage
    result = nn::pcv::GetVoltageEnabled(&enabled, static_cast<nn::pcv::PowerDomain>(GetParam()));
    EXPECT_TRUE(result.IsSuccess());
    if (!result.IsSuccess())
    {
            NN_LOG("Module: %3d\n", result.GetModule());
            NN_LOG("Description: %3d\n", result.GetDescription());
    }
    if (enabled)
    {
        nn::pcv::MicroVolt minVoltage, maxVoltage, stepVoltage = 0;
        result = nn::pcv::GetVoltageRange(&minVoltage, &maxVoltage, &stepVoltage, static_cast<nn::pcv::PowerDomain>(GetParam()));
        EXPECT_TRUE(result.IsSuccess());
        if (!result.IsSuccess())
        {
            NN_LOG("Module: %3d\n", result.GetModule());
            NN_LOG("Description: %3d\n", result.GetDescription());
        }
        NN_LOG("PowerDomain %2d: min=%10d uV, max=%10d uV, step=%10d uV\n",
               GetParam(), minVoltage, maxVoltage, stepVoltage);

        ASSERT_GE(maxVoltage, minVoltage);
        ASSERT_GT(stepVoltage, 0);

        // Set all possible voltage values
        nn::pcv::MicroVolt setVoltage = minVoltage;
        while (setVoltage <= maxVoltage)
        {
            result = nn::pcv::SetVoltageValue(
                static_cast<nn::pcv::PowerDomain>(GetParam()),
                setVoltage);

            EXPECT_TRUE(result.IsSuccess());
            if (!result.IsSuccess())
            {
                NN_LOG("Module: %3d\n", result.GetModule());
                NN_LOG("Description: %3d\n", result.GetDescription());
            }

            // Verify
            nn::pcv::MicroVolt actualSetVoltage = 0;
            result = nn::pcv::GetVoltageValue(
                &actualSetVoltage,
                static_cast<nn::pcv::PowerDomain>(GetParam()));

            EXPECT_TRUE(result.IsSuccess());
            if (!result.IsSuccess())
            {
                NN_LOG("Module: %3d\n", result.GetModule());
                NN_LOG("Description: %3d\n", result.GetDescription());
            }
            EXPECT_EQ(setVoltage, actualSetVoltage);

            setVoltage += stepVoltage;
        }

    }
    else
    {
        NN_LOG("PowerDomain %2d: not enabled\n", GetParam());
    }
}

TEST_F(SetVoltageValueTest, InvalidPowerDomain)
{
    ASSERT_TRUE(nn::pcv::IsInitialized());
    nn::Result invalidResult;

    invalidResult = nn::pcv::SetVoltageValue(nn::pcv::PowerDomain_NumPowerDomain, 0);
    ASSERT_FALSE(invalidResult.IsSuccess());
    NN_LOG("Module: %3d\n", invalidResult.GetModule());
    NN_LOG("Description: %3d\n", invalidResult.GetDescription());
}
