﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/pctl/detail/service/json/pctl_Json.h>
#include <nn/pctl/detail/service/json/pctl_JsonDataHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonErrorHandler.h>
#include <nn/pctl/detail/service/json/pctl_JsonInputStream.h>

using namespace nn::pctl::detail::service;

static char JsonStringBuffer[4096];
static char JsonStreamBuffer[1024];

#define EMPTY_SPECIFIER

TEST(Json, ParseEmpty)
{
    static const char JsonDataPattern[] =
        "";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, nullptr, 0);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_TRUE(parseResult.IsError());
    EXPECT_EQ(RAPIDJSON_NAMESPACE::kParseErrorDocumentEmpty, parseResult.Code());
}

TEST(Json, ParseInt_Unsigned)
{
    static const char JsonDataPattern[] =
        "123";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_EQ(123, value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseInt_Signed)
{
    static const char JsonDataPattern[] =
        "-456";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_EQ(-456, value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseInt_Overflow)
{
    static const char JsonDataPattern[] =
        "9223372036854775808"; // 0x7FFFFFFF_FFFFFFFF + 1
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, int64_t) = [](void*, int, int64_t) -> bool
    {
        // should not reach here
        ADD_FAILURE();
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_TRUE(parseResult.IsError());
}

TEST(JsonInput, ParseInt_Double)
{
    static const char JsonDataPattern[] =
        "123.45";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, int64_t) = [](void*, int, int64_t) -> bool
    {
        // should not reach here
        ADD_FAILURE();
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_TRUE(parseResult.IsError());
}

TEST(JsonInput, ParseInt_Nullable_Null)
{
    static const char JsonDataPattern[] =
        "null";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, nn::util::optional<int64_t>) = [](void*, int, nn::util::optional<int64_t> value) -> bool
    {
        EXPECT_FALSE(value.operator bool());
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64_NULLABLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseInt_Nullable_NotNull)
{
    static const char JsonDataPattern[] =
        "-23";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, nn::util::optional<int64_t>) = [](void*, int, nn::util::optional<int64_t> value) -> bool
    {
        EXPECT_TRUE(value.operator bool());
        EXPECT_EQ(-23, *value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64_NULLABLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseUint_Unsigned)
{
    static const char JsonDataPattern[] =
        "123";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(123, value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseUint_Signed)
{
    static const char JsonDataPattern[] =
        "-456";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, uint64_t) = [](void*, int, uint64_t) -> bool
    {
        // should not reach here
        ADD_FAILURE();
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_TRUE(parseResult.IsError());
}

TEST(JsonInput, ParseUint_Double)
{
    static const char JsonDataPattern[] =
        "123.45";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, uint64_t) = [](void*, int, uint64_t) -> bool
    {
        // should not reach here
        ADD_FAILURE();
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_TRUE(parseResult.IsError());
}

TEST(JsonInput, ParseUint_Nullable_Null)
{
    static const char JsonDataPattern[] =
        "null";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, nn::util::optional<uint64_t>) = [](void*, int, nn::util::optional<uint64_t> value) -> bool
    {
        EXPECT_FALSE(value.operator bool());
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64_NULLABLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseUint_Nullable_NotNull)
{
    static const char JsonDataPattern[] =
        "456";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, nn::util::optional<uint64_t>) = [](void*, int, nn::util::optional<uint64_t> value) -> bool
    {
        EXPECT_TRUE(value.operator bool());
        EXPECT_EQ(456, *value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64_NULLABLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseDouble_Unsigned)
{
    static const char JsonDataPattern[] =
        "123";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, double) = [](void*, int, double value) -> bool
    {
        EXPECT_EQ(123, value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseDouble_Signed)
{
    static const char JsonDataPattern[] =
        "-456";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, double) = [](void*, int, double value) -> bool
    {
        EXPECT_EQ(-456, value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseDouble_Double)
{
    static const char JsonDataPattern[] =
        "-123.25";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool(*functionValueHandler)(void*, int, double) = [](void*, int, double value) -> bool
    {
        EXPECT_EQ(-123.25, value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseDouble_Nullable_Null)
{
    static const char JsonDataPattern[] =
        "null";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, nn::util::optional<double>) = [](void*, int, nn::util::optional<double> value) -> bool
    {
        EXPECT_FALSE(value.operator bool());
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE_NULLABLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseDouble_Nullable_NotNull)
{
    static const char JsonDataPattern[] =
        "123.25";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, nn::util::optional<double>) = [](void*, int, nn::util::optional<double> value) -> bool
    {
        EXPECT_TRUE(value.operator bool());
        EXPECT_EQ(123.25, *value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE_NULLABLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseBoolean)
{
    static const char JsonDataPattern[] =
        "true";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, bool) = [](void*, int, bool value) -> bool
    {
        EXPECT_EQ(true, value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseBooleanNullable_Null)
{
    static const char JsonDataPattern[] =
        "null";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, nn::util::optional<bool>) = [](void*, int, nn::util::optional<bool> value) -> bool
    {
        EXPECT_FALSE(value.operator bool());
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN_NULLABLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseBooleanNullable_NotNull)
{
    static const char JsonDataPattern[] =
        "false";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, nn::util::optional<bool>) = [](void*, int, nn::util::optional<bool> value) -> bool
    {
        EXPECT_TRUE(value.operator bool());
        EXPECT_EQ(false, *value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN_NULLABLE(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseString)
{
    static const char JsonDataPattern[] =
        "\"The string\"";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, const char*, size_t) = [](void*, int, const char* value, size_t length) -> bool
    {
        EXPECT_EQ(0, std::strcmp("The string", value));
        EXPECT_EQ(10, length);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseString_Null)
{
    static const char JsonDataPattern[] =
        "null";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    bool (* functionValueHandler)(void*, int, const char*, size_t) = [](void*, int, const char* value, size_t length) -> bool
    {
        NN_UNUSED(length);
        EXPECT_EQ(nullptr, value);
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(*functionValueHandler)
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
}

TEST(JsonInput, ParseObject_OneValue)
{
    static const char JsonDataPattern[] =
        "{\"number1\":2}";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached;
    reached = false;
    bool (* functionValueHandler)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reached);
        EXPECT_EQ(2, value);
        reached = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number1") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reached);
}

TEST(JsonInput, ParseObject_ThreeValue)
{
    static const char JsonDataPattern[] =
        "{\"number1\": 2, \"number3\": 4.5, \"number2\": -20}";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached1;
    static bool reached2;
    static bool reached3;
    reached1 = false;
    reached2 = false;
    reached3 = false;
    bool (* functionValue1Handler)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reached1);
        EXPECT_EQ(2, value);
        reached1 = true;
        return true;
    };
    bool (* functionValue2Handler)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_EQ(false, reached2);
        EXPECT_EQ(-20, value);
        reached2 = true;
        return true;
    };
    bool (* functionValue3Handler)(void*, int, double) = [](void*, int, double value) -> bool
    {
        EXPECT_EQ(false, reached3);
        EXPECT_EQ(4.5, value);
        reached3 = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number1") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValue1Handler)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number2") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValue2Handler)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number3") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE(*functionValue3Handler)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reached1);
    EXPECT_EQ(true, reached2);
    EXPECT_EQ(true, reached3);
}

TEST(JsonInput, ParseObject_NotAllowUnknownKey)
{
    static const char JsonDataPattern[] =
        "{\"number1\":2,\"number2\":4}";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached;
    reached = false;
    bool (* functionValueHandler)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reached);
        EXPECT_EQ(2, value);
        reached = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN_FIXED(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number1") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_TRUE(parseResult.IsError());
    EXPECT_EQ(true, reached);
}

TEST(JsonInput, ParseObject_AllowUnknownKey)
{
    static const char JsonDataPattern[] =
        "{\"number1\":2,\"number2\":4}";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached;
    reached = false;
    bool (* functionValueHandler)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reached);
        EXPECT_EQ(2, value);
        reached = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number1") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reached);
}

TEST(JsonInput, ParseObject_ExpectAnyKey)
{
    static const char JsonDataPattern[] =
        "{\"number1\":2,\"number2\":4,\"number3\":-6,\"number4\":-8}";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static int reachedCount;
    static bool reachedValue2;
    static bool reachedValue4;
    static int nextExpectValue = 0;
    reachedCount = 0;
    reachedValue2 = false;
    reachedValue4 = false;
    bool (* functionKeyHandler)(bool*, void*, int, const char*, size_t) = [](bool* outIgnore, void*, int, const char* key, size_t length) -> bool
    {
        if (std::strncmp("number1", key, 7) == 0 && length == 7)
        {
            nextExpectValue = 2;
        }
        else if (std::strncmp("number2", key, 7) == 0 && length == 7)
        {
            // should not reach here
            ADD_FAILURE();
        }
        else if (std::strncmp("number3", key, 7) == 0 && length == 7)
        {
            nextExpectValue = -6;
        }
        else if (std::strncmp("number4", key, 7) == 0 && length == 7)
        {
            reachedValue4 = true;
            *outIgnore = true;
        }
        else
        {
            // should not reach here
            ADD_FAILURE();
            *outIgnore = true;
        }
        return true;
    };
    bool (* functionValueHandler)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_EQ(nextExpectValue, value);
        ++reachedCount;
        return true;
    };
    bool (* functionValueHandler2)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reachedValue2);
        EXPECT_EQ(4, value);
        reachedValue2 = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_ANY_KEY(functionKeyHandler) NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number2") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler2)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reachedValue2);
    EXPECT_EQ(true, reachedValue4);
    EXPECT_EQ(2, reachedCount);
}

TEST(JsonInput, ParseObject_NestObject)
{
    static const char JsonDataPattern[] =
        "{"
            "\"number1\":2,"
            "\"data2\":{"
                "\"hoge\":true,"
                "\"piyo\":\"string\""
            "},"
            "\"number2\":-2"
        "}";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached1;
    static bool reached2;
    static bool reached3;
    static bool reached4;
    reached1 = false;
    reached2 = false;
    reached3 = false;
    reached4 = false;
    bool (* functionValueHandler1)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reached1);
        EXPECT_EQ(2, value);
        reached1 = true;
        return true;
    };
    bool (* functionValueHandler2)(void*, int, bool) = [](void*, int, bool value) -> bool
    {
        EXPECT_EQ(false, reached2);
        EXPECT_EQ(true, value);
        reached2 = true;
        return true;
    };
    bool (* functionValueHandler3)(void*, int, const char*, size_t) = [](void*, int, const char* value, size_t length) -> bool
    {
        EXPECT_EQ(false, reached3);
        EXPECT_EQ(0, std::strncmp(value, "string", 6));
        EXPECT_EQ(6, length);
        reached3 = true;
        return true;
    };
    bool (* functionValueHandler4)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_EQ(false, reached4);
        EXPECT_EQ(-2, value);
        reached4 = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number1") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler1)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("data2") NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("hoge") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN(*functionValueHandler2)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("piyo") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(*functionValueHandler3)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number2") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler4)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reached1);
    EXPECT_EQ(true, reached2);
    EXPECT_EQ(true, reached3);
    EXPECT_EQ(true, reached4);
}

TEST(JsonInput, ParseObject_NestNullObject)
{
    static const char JsonDataPattern[] =
        "{"
            "\"number1\":2,"
            "\"data2\":null,"
            "\"number2\":-2"
        "}";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached1;
    static bool reached2;
    static bool reached3;
    static bool reached4;
    reached1 = false;
    reached2 = false;
    reached3 = false;
    reached4 = false;
    bool (* functionValueHandler1)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reached1);
        EXPECT_EQ(2, value);
        reached1 = true;
        return true;
    };
    bool (* functionValueHandler2)(void*, int, bool) = [](void*, int, bool) -> bool
    {
        ADD_FAILURE();
        return true;
    };
    bool (* functionValueHandler3)(void*, int, const char*, size_t) = [](void*, int, const char*, size_t) -> bool
    {
        ADD_FAILURE();
        return true;
    };
    bool (* functionValueHandler4)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_EQ(false, reached4);
        EXPECT_EQ(-2, value);
        reached4 = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number1") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler1)
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("data2") NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(true)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("hoge") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_BOOLEAN(*functionValueHandler2)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("piyo") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(*functionValueHandler3)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
            NN_DETAIL_PCTL_JSON_EXPECT_KEY("number2") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler4)
        NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reached1);
    EXPECT_EQ(false, reached2);
    EXPECT_EQ(false, reached3);
    EXPECT_EQ(true, reached4);
}

TEST(JsonInput, ParseArray_OneValue)
{
    static const char JsonDataPattern[] =
        "[2]";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached;
    reached = false;
    bool (* functionValueHandler)(void*, int, uint64_t) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reached);
        EXPECT_EQ(2, value);
        reached = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValueHandler)
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reached);
}

TEST(JsonInput, ParseArray_ThreeValue)
{
    static const char JsonDataPattern[] =
        "[2, -20, 4.5]";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached1;
    static bool reached2;
    static bool reached3;
    reached1 = false;
    reached2 = false;
    reached3 = false;
    bool(*functionValue1Handler)(void* param, int index, uint64_t value) = [](void*, int, uint64_t value) -> bool
    {
        EXPECT_EQ(false, reached1);
        EXPECT_EQ(2, value);
        reached1 = true;
        return true;
    };
    bool (* functionValue2Handler)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_EQ(false, reached2);
        EXPECT_EQ(-20, value);
        reached2 = true;
        return true;
    };
    bool (* functionValue3Handler)(void*, int, double) = [](void*, int, double value) -> bool
    {
        EXPECT_EQ(false, reached3);
        EXPECT_EQ(4.5, value);
        reached3 = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN_FIXED(false)
            NN_DETAIL_PCTL_JSON_EXPECT_VALUE_UINT64(*functionValue1Handler)
            NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValue2Handler)
            NN_DETAIL_PCTL_JSON_EXPECT_VALUE_DOUBLE(*functionValue3Handler)
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reached1);
    EXPECT_EQ(true, reached2);
    EXPECT_EQ(true, reached3);
}

TEST(JsonInput, ParseArray_RepeatValue)
{
    static const char JsonDataPattern[] = "[2, 8, -10, 6]";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static const int64_t ExpectValues[] = { 2, 8, -10, 6 };
    static const int ExpectValuesLength = static_cast<int>(std::extent<decltype(ExpectValues)>::value);
    static int reached;
    reached = 0;
    bool (* functionValueHandler)(void*, int, int64_t) = [](void*, int index, int64_t value) -> bool
    {
        EXPECT_GE(index, 0);
        EXPECT_LT(index, ExpectValuesLength);
        EXPECT_EQ(ExpectValues[index], value);
        ++reached;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler)
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(ExpectValuesLength, reached);
}

TEST(JsonInput, ParseArray_WithObjectValue)
{
    static const char JsonDataPattern[] =
        "[-3, {"
        "    \"good\": \"bye\""
        "}]";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static bool reached1;
    static bool reached2;
    reached1 = false;
    reached2 = false;
    bool (* functionValueHandler1)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_EQ(false, reached1);
        EXPECT_EQ(-3, value);
        reached1 = true;
        return true;
    };
    bool (* functionValueHandler2)(void*, int, const char*, size_t) = [](void*, int, const char* value, size_t length) -> bool
    {
        EXPECT_EQ(false, reached2);
        EXPECT_EQ(0, std::strncmp("bye", value, 3));
        EXPECT_EQ(3, length);
        reached2 = true;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN_FIXED(false)
            NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler1)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("good") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_STRING(*functionValueHandler2)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(true, reached1);
    EXPECT_EQ(true, reached2);
}

TEST(JsonInput, ParseArray_RepeatObjectValue)
{
    static const char JsonDataPattern[] =
        "[{"
        "    \"good\": 1"
        "},{"
        "    \"good\": 2"
        "},{"
        "    \"good\": 3"
        "}]";
    static const size_t JsonDataPatternLength = std::extent<decltype(JsonDataPattern)>::value - 1; // NULL文字除く
    static const int64_t ExpectValues[] = { 1, 2, 3 };
    static const int ExpectValuesLength = static_cast<int>(std::extent<decltype(ExpectValues)>::value);
    static int reached;
    reached = 0;
    bool (* functionValueHandler)(void*, int, int64_t) = [](void*, int, int64_t value) -> bool
    {
        EXPECT_GE(reached, 0);
        EXPECT_LT(reached, ExpectValuesLength);
        EXPECT_EQ(ExpectValues[reached], value);
        ++reached;
        return true;
    };
    NN_DETAIL_PCTL_JSON_BEGIN_EXPECTS(EMPTY_SPECIFIER, JsonDataExpect)
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_BEGIN(false)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_BEGIN(false)
                NN_DETAIL_PCTL_JSON_EXPECT_KEY("good") NN_DETAIL_PCTL_JSON_EXPECT_VALUE_INT64(*functionValueHandler)
            NN_DETAIL_PCTL_JSON_EXPECT_OBJECT_END()
        NN_DETAIL_PCTL_JSON_EXPECT_ARRAY_END()
    NN_DETAIL_PCTL_JSON_END_EXPECTS()

    json::NullAllocator allocator;
    json::JsonMemoryInputStream stream;
    json::JsonDataHandler handler(nullptr, JsonDataExpect);
    json::JsonReader reader(&allocator, 1);
    static const int parseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag |
        RAPIDJSON_NAMESPACE::kParseStopWhenDoneFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;
    stream.SetBuffer(JsonStringBuffer, sizeof(JsonStringBuffer), JsonStreamBuffer, sizeof(JsonStreamBuffer));

    NNT_EXPECT_RESULT_SUCCESS(stream.Open(JsonDataPattern, JsonDataPatternLength));
    auto parseResult = reader.Parse<parseFlags>(stream, handler);
    EXPECT_FALSE(parseResult.IsError());
    EXPECT_EQ(ExpectValuesLength, reached);
}
