﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nnt.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>

#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/util/util_Execution.h>
#include <nn/pctl/pctl_ApiSettings.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiStereoVision.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiForDebug.h>
#include <nn/pctl/pctl_ApiPairing.h>
#include <nn/pctl/pctl_ApiWatcher.h>

#include <nn/time/time_ApiForMenu.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nsd/nsd_ApiForTest.h>
#else
#include <nn/nsd/nsd_ApiForMenu.h>
#endif

#include <nn/os/os_SystemEvent.h>
#include <nn/ae.h>
#include <nn/applet/applet.h>
#include <nn/err.h>

#include "../Common/testPctl_TestUtil.h"


namespace
{
    void WaitApplicationExit(nn::os::SystemEventType& e) NN_NOEXCEPT
    {
        nn::Bit32 messageFlags = 0;
        while (messageFlags != 0x3)
        {
            auto message = nn::ae::WaitForNotificationMessage(&e);
            if (message == nn::ae::Message_ChangeIntoForeground)
            {
                nn::ae::AcquireForegroundRights();
                messageFlags |= 0x1;
            }
            else if (message == nn::ae::Message_ApplicationExited)
            {
                messageFlags |= 0x2;
            }
        }
    }

    // nn::ae のメッセージ処理
    bool ExpectMessage(nn::ae::Message expectMessage, nn::os::SystemEventType& e) NN_NOEXCEPT
    {
        auto message = nn::ae::WaitForNotificationMessage(&e);
        if (message != expectMessage)
        {
            NN_LOG("Unexpected message 0x%08x (expect=0x%08x)\n", message, expectMessage);
            return false;
        }
        return true;
    }

    void RunApplication(nn::ae::SystemAppletParameters* param) NN_NOEXCEPT
    {
        // SA の起動
        nn::os::SystemEventType e;
        nn::ae::InitializeNotificationMessageEvent(&e);
        NN_ABORT_UNLESS(ExpectMessage(nn::ae::Message_ChangeIntoForeground, e));

        auto ret = RUN_ALL_TESTS();

        nnt::Exit(ret);
    }

    char* GetEventNameFromEventType(nn::pctl::detail::service::watcher::EventType e)
    {
        NN_FUNCTION_LOCAL_STATIC(char, name[32], = "");

        switch (e)
        {
        case nn::pctl::detail::service::watcher::EventType::Invalid:
            nn::util::Strlcpy(name, "Invalid", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidDeviceLaunch:
            nn::util::Strlcpy(name, "DidDeviceLaunch", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidWakeup:
            nn::util::Strlcpy(name, "DidWakeup", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidSleep:
            nn::util::Strlcpy(name, "DidSleep", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationLaunch:
            nn::util::Strlcpy(name, "DidApplicationLaunch", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationTerminate:
            nn::util::Strlcpy(name, "DidApplicationTerminate", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::Idle:
            nn::util::Strlcpy(name, "Idle", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidUnlock:
            nn::util::Strlcpy(name, "DidUnlock", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidLock:
            nn::util::Strlcpy(name, "DidLock", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidWrongUnlockCode:
            nn::util::Strlcpy(name, "DidWrongUnlockCode", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidComeOnline:
            nn::util::Strlcpy(name, "DidComeOnline", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidGoOffline:
            nn::util::Strlcpy(name, "DidGoOffline", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidAddNewManagedApplication:
            nn::util::Strlcpy(name, "DidAddNewManagedApplication", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidRemoveManagedApplication:
            nn::util::Strlcpy(name, "DidRemoveManagedApplication", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidInterruptPlaying:
            nn::util::Strlcpy(name, "DidInterruptPlaying", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationPlay:
            nn::util::Strlcpy(name, "DidApplicationPlay", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidAlarmMakeInvisible:
            nn::util::Strlcpy(name, "DidAlarmMakeInvisible", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidAlarmMakeVisible:
            nn::util::Strlcpy(name, "DidAlarmMakeVisible", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationDownloadStart:
            nn::util::Strlcpy(name, "DidApplicationDownloadStart", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidReachLimitTime:
            nn::util::Strlcpy(name, "DidReachLimitTime", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationSuspend:
            nn::util::Strlcpy(name, "DidApplicationSuspend", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidApplicationResume:
            nn::util::Strlcpy(name, "DidApplicationResume", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidUserOpen:
            nn::util::Strlcpy(name, "DidUserOpen", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidUserClose:
            nn::util::Strlcpy(name, "DidUserClose", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidShutdown:
            nn::util::Strlcpy(name, "DidShutdown", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidPlayTimerStart:
            nn::util::Strlcpy(name, "DidPlayTimerStart", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidPlayTimerStop:
            nn::util::Strlcpy(name, "DidPlayTimerStop", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidDeviceActivate: // 連携完了直後の最初のイベント
            nn::util::Strlcpy(name, "DidDeviceActivate", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidLocationNameChange:
            nn::util::Strlcpy(name, "DidLocationNameChange", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidUnexpectedShutdownOccur:
            nn::util::Strlcpy(name, "DidUnexpectedShutdownOccur", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::DidLimitedApplicationLaunch:
            nn::util::Strlcpy(name, "DidLimitedApplicationLaunch", sizeof(name));
            break;
        case nn::pctl::detail::service::watcher::EventType::EventTypeMaxCount:
            nn::util::Strlcpy(name, "EventTypeMaxCount", sizeof(name));
            break;
        default:
            nn::util::Strlcpy(name, "Unknown", sizeof(name));
            break;
        }
        return name;
    } // NOLINT(impl/function_size)

    char* GetDateTimeString(char* outBuffer, size_t outBufferSize, nn::time::PosixTime posixTime)
    {
        if (posixTime < nn::time::InputPosixTimeMin || posixTime > nn::time::InputPosixTimeMax)
        {
            nn::util::SNPrintf(outBuffer, outBufferSize, "(Invalid)");
            return outBuffer;
        }

        nn::time::CalendarTime calendarTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        auto result = nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
        if (result.IsSuccess())
        {
            nn::util::SNPrintf(outBuffer, outBufferSize, "%04d-%02d-%02d %02d:%02d:%02d (%s)",
                calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second,
                calendarAdditionalInfo.timeZone.standardTimeName);
        }
        else
        {
            nn::util::SNPrintf(outBuffer, outBufferSize, "(GetDateTimeString failed : 0x%08x)", result.GetInnerValueForDebug());
        }

        return outBuffer;
    }

    void CompareEvent(const nn::pctl::detail::service::watcher::EventData& e1, const nn::pctl::detail::service::watcher::EventData& e2)
    {
        EXPECT_EQ(e1.eventType, e2.eventType);
        EXPECT_EQ(e1.isDaylightSavingTime, e2.isDaylightSavingTime);
        EXPECT_EQ(e1.timezoneOffsetSeconds, e2.timezoneOffsetSeconds);
        // timestamp
        EXPECT_EQ(e1.timezoneName, e2.timezoneName);
        EXPECT_EQ(e1.dataVersion, e2.dataVersion);
        // payload
    }

}

class DummySystemAppletTest : public ::testing::Test
{
public:
    static nn::pctl::test::PairingInfo PairingCodeInfo;
    static bool PairingEnabled;
    static bool UnlinkPairingEnabled;

protected:
    static void SetUpTestCase()
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::nifm::InitializeSystem());
        m_pNetworkConnection = new nn::nifm::NetworkConnection();
        m_pNetworkConnection->SubmitRequestAndWait();

        ASSERT_TRUE(m_pNetworkConnection->IsAvailable());

        NNT_ASSERT_RESULT_SUCCESS(nn::time::InitializeForMenu());

        nn::pctl::StartPlayTimer();
    }

    static void TearDownTestCase()
    {
        delete m_pNetworkConnection;
        m_pNetworkConnection = nullptr;
        NNT_ASSERT_RESULT_SUCCESS(nn::time::Finalize());
    }

    static nn::nifm::NetworkConnection* m_pNetworkConnection;
    static const nn::ncm::ApplicationId m_ApplicationId1;
    static const nn::ncm::ApplicationId m_ApplicationId2;
};

nn::pctl::test::PairingInfo DummySystemAppletTest::PairingCodeInfo = {};
bool DummySystemAppletTest::PairingEnabled = true;
bool DummySystemAppletTest::UnlinkPairingEnabled = true;

nn::nifm::NetworkConnection* DummySystemAppletTest::m_pNetworkConnection = nullptr;
const nn::ncm::ApplicationId DummySystemAppletTest::m_ApplicationId1 = { 0x010000000000B241 };
const nn::ncm::ApplicationId DummySystemAppletTest::m_ApplicationId2 = { 0x010000000000B242 };


TEST_F(DummySystemAppletTest, Pairing)
{
    if (PairingEnabled)
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::nsd::SetTd1EnvironmentForTest();
#else
        nn::nsd::EnvironmentIdentifier env;
        nn::nsd::GetEnvironmentIdentifier(&env);
        ASSERT_EQ(0, strncmp("td1", env.value, sizeof("td1")));
#endif

        if (PairingCodeInfo.pairingCode[0] == '\0')
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::pctl::test::GetPairingInfo(&PairingCodeInfo, "ido_junichi+pctl01@exmx.nintendo.co.jp", "Password"));
        }

        nn::pctl::SetPinCode(nullptr);
        nn::pctl::ClearFreeCommunicationApplicationListForDebug();
        nn::pctl::DeletePairing();

        nn::pctl::PairingInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::RequestPairing(&info, PairingCodeInfo.pairingCode));

        nn::pctl::PairingAccountInfo accountInfo;
        char nickname[64];
        size_t size = 0;

        ASSERT_TRUE(info.IsInstanceValid());
        ASSERT_EQ(nn::pctl::PairingState::PairingState_Processing, info.GetState());
        ASSERT_TRUE(info.GetAccountInfo(&accountInfo));
        ASSERT_TRUE(accountInfo.IsInstanceValid());
        EXPECT_GT(accountInfo.GetNickname(nickname, 64), static_cast<size_t>(0));
        NNT_EXPECT_RESULT_SUCCESS(accountInfo.GetMiiImageContentType(&size, nullptr, 0));
        NNT_EXPECT_RESULT_SUCCESS(accountInfo.GetMiiImage(&size, nullptr, 0));

        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::AuthorizePairing(&info));

        ASSERT_TRUE(info.IsInstanceValid());
        ASSERT_EQ(nn::pctl::PairingState::PairingState_Active, info.GetState());
    }
}

TEST_F(DummySystemAppletTest, Pre)
{
    // 一時解除状態を戻す
    nn::pctl::RevertRestrictionTemporaryUnlocked();
    ASSERT_FALSE(nn::pctl::IsRestrictionTemporaryUnlocked());

    // Event クリア
    int outCount;
    nn::pctl::detail::service::watcher::EventData eventData[1];
    NNT_ASSERT_RESULT_SUCCESS(nn::pctl::RequestPostEvent(&outCount, eventData, NN_ARRAY_SIZE(eventData)));

    // 他人との自由なコミュニケーションリストのクリア
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();
}

TEST_F(DummySystemAppletTest, Basic)
{
    NN_UNUSED(m_ApplicationId2);

    char pinCode[] = "1234";
    nn::pctl::SetPinCode(pinCode);

    nn::pctl::SafetyLevelSettings customSettings = {
        13,   // ratingAge
        true, // snsPostRestriction
        true, // freeCommunicationRestriction
    };
    nn::pctl::SetDefaultRatingOrganization(nn::ns::RatingOrganization::CERO);
    nn::pctl::SetSafetyLevel(nn::pctl::SafetyLevel::SafetyLevel_Custom);
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);

    auto applicationHandle = nn::ae::CreateApplication(m_ApplicationId1);
    NN_UTIL_SCOPE_EXIT
    {
        nn::ae::CloseApplicationHandle(applicationHandle);
    };

    // アプリ起動確認 （起動できない）
    nn::ns::ApplicationControlProperty prop;
    NNT_ASSERT_RESULT_SUCCESS(nn::ae::GetApplicationControlProperty(&prop, applicationHandle));
    {
        nn::pctl::RestrictionSettings settings;
        nn::pctl::GetCurrentSettings(&settings);
        NN_LOG("prop=%d, settings.age=%d\n", prop.GetAge(nn::ns::RatingOrganization::CERO), settings.ratingAge);
    }
    NNT_ASSERT_RESULT_FAILURE(nn::pctl::ResultRestrictedByRating, nn::pctl::ConfirmLaunchApplicationPermission(m_ApplicationId1, prop));

    customSettings.ratingAge = 20;
    nn::pctl::SetCustomSafetyLevelSettings(customSettings);

    {
        nn::pctl::RestrictionSettings settings;
        nn::pctl::GetCurrentSettings(&settings);
        NN_LOG("settings.freeCommunicationRestriction=%s\n", settings.freeCommunicationRestriction ? "true":"false");
    }

    // アプリ起動確認（起動できる）
    NNT_ASSERT_RESULT_SUCCESS(nn::pctl::ConfirmLaunchApplicationPermission(m_ApplicationId1, prop));

    // アプリ起動
    nn::ae::StartApplication(applicationHandle);

    nn::os::SystemEventType event;
    nn::ae::InitializeNotificationMessageEvent(&event);
    NNT_ASSERT_RESULT_SUCCESS(nn::ae::RequestApplicationGetForeground(applicationHandle));
    NN_ABORT_UNLESS(ExpectMessage(nn::ae::Message_ChangeIntoBackground, event));

    // アプリ終了待ち
    WaitApplicationExit(event);

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // 一時解除
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::UnlockRestrictionTemporarily(pinCode));
        NN_UTIL_SCOPE_EXIT
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            // 一時解除を元に戻す
            nn::pctl::RevertRestrictionTemporaryUnlocked();
            ASSERT_FALSE(nn::pctl::IsRestrictionTemporaryUnlocked());
        };
        ASSERT_TRUE(nn::pctl::IsRestrictionTemporaryUnlocked());
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // PinCode 間違い通知
    nn::pctl::NotifyWrongPinCodeInputManyTimes();

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // DownloadStart 通知
    nn::pctl::NotifyApplicationDownloadStarted(m_ApplicationId1);

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
}

TEST_F(DummySystemAppletTest, VerifyEvents)
{
    int outCount;
    nn::pctl::detail::service::watcher::EventData eventData[200];
    NNT_ASSERT_RESULT_SUCCESS(nn::pctl::RequestPostEvent(&outCount, eventData, NN_ARRAY_SIZE(eventData)));

    const nn::pctl::detail::service::watcher::EventData expectedEvents[] =
    {
        {
            nn::pctl::detail::service::watcher::EventType::DidLimitedApplicationLaunch,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidAddNewManagedApplication,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidApplicationLaunch,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidUserOpen,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidUserClose,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidApplicationSuspend,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidApplicationTerminate,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidUnlock,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidLock,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidWrongUnlockCode,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        {
            nn::pctl::detail::service::watcher::EventType::DidApplicationDownloadStart,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
        //
        {
            nn::pctl::detail::service::watcher::EventType::Idle,
            false, 0,{},
            { "UTC" },
            0,
            nn::pctl::detail::service::watcher::EventDataVersionForSys3,
            {},
            { { m_ApplicationId1 } },
        },
    };


    EXPECT_EQ(NN_ARRAY_SIZE(expectedEvents), outCount);
    auto eventCount = std::min(outCount, static_cast<int>(NN_ARRAY_SIZE(eventData)));
    for (int u = 0; u < eventCount; ++u)
    {
        CompareEvent(expectedEvents[u], eventData[u]);
    }

    NN_LOG("Event Count = %d\n", outCount);
    char dateTimeStr[64];
    for (int u = 0; u < eventCount; ++u)
    {
        auto& e = eventData[u];
        if (e.eventType == nn::pctl::detail::service::watcher::EventType::DidUserOpen ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidUserClose)
        {
            auto& uid = e.payload.openedUser.uid;
            NN_LOG("  [% 3lu] type = %s, timestamp = %s, uid = %08x_%08x_%08x_%08x\n",
                u, GetEventNameFromEventType(e.eventType), GetDateTimeString(dateTimeStr, sizeof(dateTimeStr), e.timestamp),
                static_cast<uint32_t>(uid._data[0] >> 32),
                static_cast<uint32_t>(uid._data[0] & 0xFFFFFFFFull),
                static_cast<uint32_t>(uid._data[1] >> 32),
                static_cast<uint32_t>(uid._data[1] & 0xFFFFFFFFull));
        }
        else if (
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationLaunch ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationTerminate ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidAddNewManagedApplication ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidRemoveManagedApplication ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationPlay ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationDownloadStart ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationSuspend ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidApplicationResume ||
            e.eventType == nn::pctl::detail::service::watcher::EventType::DidLimitedApplicationLaunch
            )
        {
            NN_LOG("  [% 3lu] type = %s, timestamp = %s, appId = 0x%016llX\n",
                u, GetEventNameFromEventType(e.eventType), GetDateTimeString(dateTimeStr, sizeof(dateTimeStr), e.timestamp),
                e.payload.applicationLaunch.applicationId.value);
        }
        else
        {
            NN_LOG("  [% 3lu] type = %s, timestamp = %s\n",
                u, GetEventNameFromEventType(e.eventType),
                GetDateTimeString(dateTimeStr, sizeof(dateTimeStr), e.timestamp));
        }
    }

} // NOLINT(impl/function_size)

TEST_F(DummySystemAppletTest, UnlinkPairing)
{
    if (UnlinkPairingEnabled)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlinkPairing(true));
    }
}

// アロケータ用のバッファ
NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[1 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    if (argc > 1 && (std::strcmp(argv[1], "--help") == 0 || std::strcmp(argv[1], "-h") == 0))
    {
        NN_LOG("Usage:\n" " testPctl_RequestUpdateExemptionList [pairing code] [--without_pairing] [--without_unlinkpairing]\n");
        return;
    }

    ::testing::InitGoogleTest(&argc, argv);

    for (int i = 1; i < argc; ++i)
    {
        if (std::strcmp(argv[i], "--without_pairing") == 0) // Pairing を行わない
        {
            DummySystemAppletTest::PairingEnabled = false;
        }
        else if (std::strcmp(argv[i], "--without_unlinkpairing") == 0) // UnlinkPairing を行わない
        {
            DummySystemAppletTest::UnlinkPairingEnabled = false;
        }
        else
        {
            nn::util::Strlcpy(DummySystemAppletTest::PairingCodeInfo.pairingCode, argv[i], sizeof(DummySystemAppletTest::PairingCodeInfo.pairingCode));
        }
    }

    // SystemApplet として起動する
    nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, RunApplication);
}
