﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiForRecovery.h>
#include <nn/pctl/pctl_ApiPairing.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/detail/ipc/pctl_IpcConfig.h>
#include <nn/pctl/detail/ipc/pctl_IpcSession.h>

NN_STATIC_ASSERT(NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS == NN_DETAIL_PCTL_CONFIG_SERVER_PROCESS_DFC);

namespace nn { namespace pctl { namespace detail { namespace service {
class ServiceWatcher;
extern ServiceWatcher* g_pWatcher;
ServiceWatcher* g_pWatcherTemp;

bool IsWatcherAvailable() NN_NOEXCEPT;

void StartDisabledWatcherEmulation() NN_NOEXCEPT
{
    // (初期化されていなければ)内部でWatcherを初期化
    ipc::GetServiceForWatcher();

    // 疑似的に無効化されている状態をつくる
    g_pWatcherTemp = g_pWatcher;
    g_pWatcher = nullptr;
    ASSERT_TRUE(!IsWatcherAvailable());
}

void FinishDisabledWatcherEmulation() NN_NOEXCEPT
{
    // 元に戻す
    g_pWatcher = g_pWatcherTemp;
}

}}}}

class DisableWatchersTest : public ::testing::Test
{
public:
    static void SetUpTestCase()
    {
        nn::nifm::Initialize();
        // 事前条件として通信必須の関数があるためリクエストを出す
        // (失敗した場合はそれらのテストを行わない)
        nn::nifm::SubmitNetworkRequestAndWait();

        nn::pctl::SafetyLevelSettings customSettings;
        customSettings.ratingAge = 15;
        customSettings.snsPostRestriction = true;
        customSettings.freeCommunicationRestriction = true;

        nn::pctl::SetPinCode("0123");
        nn::pctl::SetCustomSafetyLevelSettings(customSettings);

        EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());

        nn::pctl::PlayTimerSettings timerSettings;
        timerSettings.isEnabled = true;
        timerSettings.isWeekSettingsUsed = false;
        timerSettings.playTimerMode = nn::pctl::PlayTimerMode::PlayTimerMode_Alarm;
        timerSettings.dailySettings.isLimitTimeEnabled = true;
        timerSettings.dailySettings.limitTime = 60;
        timerSettings.dailySettings.isBedtimeEnabled = false;
        nn::pctl::SetPlayTimerSettingsForDebug(timerSettings);
        // 期限時間設定のみなので残り時間はちょうど60分になるはず
        EXPECT_EQ(nn::TimeSpan::FromMinutes(60), nn::pctl::GetPlayTimerRemainingTime());

        nn::pctl::detail::service::StartDisabledWatcherEmulation();
    }

    static void TearDownTestCase()
    {
        nn::pctl::detail::service::FinishDisabledWatcherEmulation();
        nn::pctl::DeleteSettings();
    }
};

TEST_F(DisableWatchersTest, TestAvailableNormalRestrictions)
{
    EXPECT_TRUE(nn::pctl::IsRestrictionEnabled());
    EXPECT_FALSE(nn::pctl::IsRestrictedSystemSettingsEntered());
    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultRestricted, nn::pctl::ConfirmSystemSettingsPermission());
}

TEST_F(DisableWatchersTest, TestAbortRequestPairing)
{
    if (!nn::nifm::IsNetworkAvailable())
    {
        return;
    }
    nn::pctl::PairingInfo info;
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::RequestPairing(&info, "123456"), "");
}

TEST_F(DisableWatchersTest, TestReturnFalseIsPairingActive)
{
    EXPECT_FALSE(nn::pctl::IsPairingActive());
}

TEST_F(DisableWatchersTest, TestAbortRetrievePairingInfo)
{
    if (!nn::nifm::IsNetworkAvailable())
    {
        return;
    }
    nn::pctl::PairingInfo info;
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::RetrievePairingInfo(&info), "");
}

TEST_F(DisableWatchersTest, TestAbortUnlinkPairing)
{
    if (!nn::nifm::IsNetworkAvailable())
    {
        return;
    }
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::UnlinkPairing(true), "");
}

TEST_F(DisableWatchersTest, TestAbortSynchronizeParentalControlSettings)
{
    if (!nn::nifm::IsNetworkAvailable())
    {
        return;
    }
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::SynchronizeParentalControlSettings(), "");
}

TEST_F(DisableWatchersTest, TestAbortGetSynchronizationEvent)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::GetSynchronizationEvent(), "");
}

TEST_F(DisableWatchersTest, TestAbortStartPlayTimer)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::StartPlayTimer(), "");
}

TEST_F(DisableWatchersTest, TestAbortStopPlayTimer)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::StopPlayTimer(), "");
}

TEST_F(DisableWatchersTest, TestReturnFalseIsPlayTimerEnabled)
{
    EXPECT_FALSE(nn::pctl::IsPlayTimerEnabled());
}

TEST_F(DisableWatchersTest, TestReturnZeroGetPlayTimerRemainingTime)
{
    EXPECT_EQ(nn::TimeSpan(0), nn::pctl::GetPlayTimerRemainingTime());
}

TEST_F(DisableWatchersTest, TestReturnFalseIsRestrictedByPlayTimer)
{
    EXPECT_FALSE(nn::pctl::IsRestrictedByPlayTimer());
}

TEST_F(DisableWatchersTest, TestReturnDisabledGetPlayTimerSettings)
{
    nn::pctl::PlayTimerSettings settings;
    nn::pctl::GetPlayTimerSettings(&settings);
    EXPECT_FALSE(settings.isEnabled);
}

TEST_F(DisableWatchersTest, TestAbortGetPlayTimerEventToRequestSuspension)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::GetPlayTimerEventToRequestSuspension(), "");
}

TEST_F(DisableWatchersTest, TestAbortSetPlayTimerSettingsForDebug)
{
    nn::pctl::PlayTimerSettings settings;
    settings.isEnabled = false;
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::SetPlayTimerSettingsForDebug(settings), "");
}

TEST_F(DisableWatchersTest, TestAbortCancelNetworkRequest)
{
    EXPECT_DEATH_IF_SUPPORTED(nn::pctl::CancelNetworkRequest(), "");
}

TEST_F(DisableWatchersTest, TestNothingOccurredNotifyWrongPinCodeInputManyTimes)
{
    nn::pctl::NotifyWrongPinCodeInputManyTimes();
}
