﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/nn_Log.h>

#include <nn/pctl/pctl_Api.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>
#include <nn/pctl/pctl_ApiPairing.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/pctl_ApiForDebug.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/nsd/nsd_ApiForTest.h>
#else
#include <nn/nsd/nsd_ApiForMenu.h>
#endif

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/fs.h>
#include <nn/fs/fs_Host.h>
#include <nn/http/http_Result.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Tick.h>

#include "../Common/testPctl_TestUtil.h"



class CancelTest : public ::testing::Test
{
public:
    static nn::pctl::test::PairingInfo PairingCodeInfo;
    static const size_t ThreadStackSize = 32 * 1024;
    static NN_OS_ALIGNAS_THREAD_STACK uint8_t ThreadStack[ThreadStackSize];
    static const char CancelTestThreadName[32];
    static void StaticCancelTestThreadProc(void* pThis) NN_NOEXCEPT;
    static nn::TimeSpan WaitTimeSpanForCancelRequestPairingTest;
    static nn::TimeSpan WaitTimeSpanForCancelAuthorizePairingTest;
protected:
    static void SetUpTestCase()
    {
        nn::nifm::Initialize();
        pNetworkConnection = new nn::nifm::NetworkConnection();
        pNetworkConnection->SubmitRequestAndWait();

        ASSERT_TRUE(pNetworkConnection->IsAvailable());
    }
    static void TearDownTestCase()
    {
        delete pNetworkConnection;
        pNetworkConnection = nullptr;
    }
    static nn::nifm::NetworkConnection* pNetworkConnection;
};

nn::pctl::test::PairingInfo CancelTest::PairingCodeInfo = {};
nn::TimeSpan CancelTest::WaitTimeSpanForCancelAuthorizePairingTest = nn::TimeSpan::FromMilliSeconds(500);
nn::TimeSpan CancelTest::WaitTimeSpanForCancelRequestPairingTest= nn::TimeSpan::FromMilliSeconds(1000);
nn::nifm::NetworkConnection* CancelTest::pNetworkConnection = nullptr;
const char CancelTest::CancelTestThreadName[32] = "testPctl_CancelTest";
NN_OS_ALIGNAS_THREAD_STACK uint8_t CancelTest::ThreadStack[CancelTest::ThreadStackSize];

void CancelTest::StaticCancelTestThreadProc(void* pArg) NN_NOEXCEPT
{
    nn::TimeSpan* pTimeSpan = reinterpret_cast<nn::TimeSpan*>(pArg);
    nn::os::SleepThread(*pTimeSpan);
    nn::pctl::CancelNetworkRequest();
}

TEST_F(CancelTest, CancelAuthorizePairing)
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::nsd::SetTd1EnvironmentForTest();
#else
    nn::nsd::EnvironmentIdentifier env;
    nn::nsd::GetEnvironmentIdentifier(&env);
    ASSERT_STREQ("td1", env.value);
#endif

    if (PairingCodeInfo.pairingCode[0] == '\0')
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::test::GetPairingInfo(&PairingCodeInfo, "ido_junichi+pctl01@exmx.nintendo.co.jp", "Password"));
    }

    nn::pctl::SetPinCode(nullptr);
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();
    nn::pctl::DeletePairing();

    nn::pctl::PairingInfo info;
    NNT_ASSERT_RESULT_SUCCESS(nn::pctl::RequestPairing(&info, PairingCodeInfo.pairingCode));

    nn::pctl::PairingAccountInfo accountInfo;
    char nickname[64];
    size_t size = 0;

    EXPECT_TRUE(info.IsInstanceValid());
    EXPECT_EQ(nn::pctl::PairingState::PairingState_Processing, info.GetState());
    EXPECT_TRUE(info.GetAccountInfo(&accountInfo));
    EXPECT_TRUE(accountInfo.IsInstanceValid());
    EXPECT_GT(accountInfo.GetNickname(nickname, 64), static_cast<size_t>(0));
    NNT_EXPECT_RESULT_SUCCESS(accountInfo.GetMiiImageContentType(&size, nullptr, 0));
    NNT_EXPECT_RESULT_SUCCESS(accountInfo.GetMiiImage(&size, nullptr, 0));

    // Create Thread for cancelling AuthorizePairing.
    nn::os::ThreadType cancelTestThread;
    nn::os::CreateThread(&cancelTestThread, StaticCancelTestThreadProc, &CancelTest::WaitTimeSpanForCancelAuthorizePairingTest, CancelTest::ThreadStack, CancelTest::ThreadStackSize, nn::os::DefaultThreadPriority);
    nn::os::SetThreadNamePointer(&cancelTestThread, CancelTest::CancelTestThreadName);

    // Start cancelTestThread and measure the time it took to cancel AuthorizePairing.
    nn::os::Tick    tickStart = nn::os::GetSystemTick();
    nn::os::StartThread(&cancelTestThread); // sleep $WaitTimeSpanForCancelAuthorizePairingTest millisecond and call 'CancelNetworkRequest' in the thread

    nn::Result result = nn::pctl::AuthorizePairing(&info);
    nn::os::Tick    tickElapsed = nn::os::GetSystemTick() - tickStart;
    nn::TimeSpan    timeSpanElapsed = nn::os::ConvertToTimeSpan(tickElapsed);

    // description=221 means 'ResultCanceled'
    NN_LOG("elapsed time:%lld(msec) module=%d description=%d\n", timeSpanElapsed.GetMilliSeconds(), result.GetModule(), result.GetDescription());
    EXPECT_TRUE(result.IsSuccess() || nn::pctl::ResultCanceled().Includes(result));

    // Join and destroy the thread
    nn::os::DestroyThread(&cancelTestThread);

    // Necessary to unlink pairing before this test exit.
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlinkPairing(true));
    nn::Result resultOfForceUnlinkPairing = nn::pctl::test::ForceUnlinkPairing(&PairingCodeInfo);
    NN_LOG("resultOfForceUnlinkPairing module=%d description=%d\n", resultOfForceUnlinkPairing.GetModule(), resultOfForceUnlinkPairing.GetDescription());
    // Moon server respond with a status of 404(not found) when the device is not linked.
    EXPECT_TRUE(resultOfForceUnlinkPairing.IsSuccess() || nn::http::ResultHttpStatusNotFound().Includes(resultOfForceUnlinkPairing));

    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultCanceled, result);
}

TEST_F(CancelTest, CancelRequestPairing)
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    nn::nsd::SetTd1EnvironmentForTest();
#else
    nn::nsd::EnvironmentIdentifier env;
    nn::nsd::GetEnvironmentIdentifier(&env);
    ASSERT_STREQ("td1", env.value);
#endif

    if (PairingCodeInfo.pairingCode[0] == '\0')
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::pctl::test::GetPairingInfo(&PairingCodeInfo, "ido_junichi+pctl01@exmx.nintendo.co.jp", "Password"));
    }

    nn::pctl::SetPinCode(nullptr);
    nn::pctl::ClearFreeCommunicationApplicationListForDebug();
    nn::pctl::DeletePairing();

    nn::pctl::PairingInfo info;

    // Create Thread for cancelling RequestPairing.
    nn::os::ThreadType cancelTestThread;
    nn::os::CreateThread(&cancelTestThread, StaticCancelTestThreadProc, &CancelTest::WaitTimeSpanForCancelRequestPairingTest, CancelTest::ThreadStack, CancelTest::ThreadStackSize, nn::os::DefaultThreadPriority);
    nn::os::SetThreadNamePointer(&cancelTestThread, CancelTest::CancelTestThreadName);

    // Start cancelTestThread and measure the time it took to cancel RequestPairing.
    nn::os::Tick    tickStart = nn::os::GetSystemTick();
    nn::os::StartThread(&cancelTestThread); // sleep WaitTimeSpanForCancelRequestPairingTest millisecond and call 'CancelNetworkRequest' in the thread.

    nn::Result result = nn::pctl::RequestPairing(&info, PairingCodeInfo.pairingCode);
    nn::os::Tick    tickElapsed = nn::os::GetSystemTick() - tickStart;
    nn::TimeSpan    timeSpanElapsed = nn::os::ConvertToTimeSpan(tickElapsed);

    // description=221 means 'ResultCanceled'.
    NN_LOG("elapsed time:%lld(msec) module=%d description=%d\n", timeSpanElapsed.GetMilliSeconds(), result.GetModule(), result.GetDescription());
    EXPECT_TRUE( result.IsSuccess() || nn::pctl::ResultCanceled().Includes(result) );

    // Join and destroy the thread.
    nn::os::DestroyThread(&cancelTestThread);

    // Necessary to unlink pairing before this test exit.
    NNT_EXPECT_RESULT_SUCCESS(nn::pctl::UnlinkPairing(true));
    nn::Result resultOfForceUnlinkPairing = nn::pctl::test::ForceUnlinkPairing(&PairingCodeInfo);
    NN_LOG("resultOfUnlink module=%d description=%d\n", resultOfForceUnlinkPairing.GetModule(), resultOfForceUnlinkPairing.GetDescription());
    // Moon server respond with a status of 404(not found) when the device is not linked.
    EXPECT_TRUE(resultOfForceUnlinkPairing.IsSuccess() || nn::http::ResultHttpStatusNotFound().Includes(resultOfForceUnlinkPairing));

    NNT_EXPECT_RESULT_FAILURE(nn::pctl::ResultCanceled, result);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    if (argc > 1 && (std::strcmp(argv[1], "--help") == 0 || std::strcmp(argv[1], "-h") == 0))
    {
        NN_LOG("Usage:\n" " testPctl_Cancel [no args]\n");
        return;
    }

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
