﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"
#include <nn/os/os_Config.h>

#include <nnc/nn_Macro.h>
#include <nnc/nn_Common.h>
#include <nnc/nn_Log.h>
#include <nnc/os.h>
#include <nnc/os/os_SdkThread.h>
#include <nnc/os/os_SystemEvent.h>
#include <nnc/os/os_InterruptEvent.h>
#include <nnc/os/os_MultipleWait.h>
#include <nnc/os/os_MessageQueue.h>
#include <nnc/os/os_NativeHandle.h>
#include <nnc/os/os_ThreadLocalStorage.h>
#include <nnc/os/os_Tick.h>
#include <nnc/os/os_MemoryFence.h>

//-----------------------------------------------------------------------------

#define NNT_OS_RETURN_FALSE_UNLESS(condition)                           \
                                do                                      \
                                {                                       \
                                    if (!(condition)) { return false; } \
                                } while (NNC_STATIC_CONDITION(false))

//-----------------------------------------------------------------------------

void nntosTestTypeInfoDumpForC(void)
{
    NNC_LOG("\n");
    NNC_LOG("C  :            ThreadType: size=%d align=%d\n", sizeof(nnosThreadType), NNC_ALIGNOF(nnosThreadType));
    NNC_LOG("C  :         SemaphoreType: size=%d align=%d\n", sizeof(nnosSemaphoreType), NNC_ALIGNOF(nnosSemaphoreType));
    NNC_LOG("C  :      MessageQueueType: size=%d align=%d\n", sizeof(nnosMessageQueueType), NNC_ALIGNOF(nnosMessageQueueType));
    NNC_LOG("C  :             EventType: size=%d align=%d\n", sizeof(nnosEventType), NNC_ALIGNOF(nnosEventType));
    NNC_LOG("C  :        TimerEventType: size=%d align=%d\n", sizeof(nnosTimerEventType), NNC_ALIGNOF(nnosTimerEventType));
    NNC_LOG("C  :    InterruptEventType: size=%d align=%d\n", sizeof(nnosInterruptEventType), NNC_ALIGNOF(nnosInterruptEventType));
    NNC_LOG("C  :       SystemEventType: size=%d align=%d\n", sizeof(nnosSystemEventType), NNC_ALIGNOF(nnosSystemEventType));
    NNC_LOG("C  :         MultiWaitType: size=%d align=%d\n", sizeof(nnosMultiWaitType), NNC_ALIGNOF(nnosMultiWaitType));
    NNC_LOG("C  :   MultiWaitHolderType: size=%d align=%d\n", sizeof(nnosMultiWaitHolderType), NNC_ALIGNOF(nnosMultiWaitHolderType));
    NNC_LOG("C  :             MutexType: size=%d align=%d\n", sizeof(nnosMutexType), NNC_ALIGNOF(nnosMutexType));
    NNC_LOG("C  : ConditionVariableType: size=%d align=%d\n", sizeof(nnosConditionVariableType), NNC_ALIGNOF(nnosConditionVariableType));
//    NNC_LOG("C  :           BarrierType: size=%d align=%d\n", sizeof(nnosBarrierType), NNC_ALIGNOF(nnosBarrierType));
    NNC_LOG("\n");
}

//-----------------------------------------------------------------------------

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    const nnosInterruptName g_InterruptName = NULL;
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #if defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6) || \
        defined(NN_BUILD_CONFIG_HARDWARE_SMMA53)   || \
        defined(NN_BUILD_CONFIG_HARDWARE_JUNO)
    const nnosInterruptName g_InterruptName = 32;       /* Not used */
    #else /* for jetson-tk1 or tk2 */
    const nnosInterruptName g_InterruptName = 32 + 22;  /* Unassigned */
    #endif
#endif

NNC_ALIGNAS(4096)    char g_Stack[4096];

void nntosThreadTestSubThread(void* arg)
{
    nnosThreadType* thread = (nnosThreadType*)arg;

    nnosSetThreadName(thread, NULL);

    if (thread == nnosGetCurrentThread())
    {
        nnosSetThreadNamePointer(thread, "SUCCEED");
    }
    else
    {
        nnosSetThreadNamePointer(thread, "FAIL");
    }
}


bool nntosThreadTestForC(void)
{
    nnosThreadType  thread;
    nnResult        result;
    int             pri;
    int             core;

    NNC_LOG("nnosThread 関連のテスト開始\n");

    // nnosCreateThread
    NNC_LOG("nnosCreateThread() で子スレッドを生成\n");
    result = nnosCreateThread(&thread, nntosThreadTestSubThread, &thread, g_Stack, sizeof(g_Stack), NN_OS_DEFAULT_THREAD_PRIORITY);
    NNT_OS_RETURN_FALSE_UNLESS( nnResultIsSuccess(result) );

    // nnosStartThread
    NNC_LOG("nnosStartThread() で子スレッドを起動\n");
    nnosStartThread(&thread);

    // nnosWaitThread
    NNC_LOG("nnosWaitThread() で子スレッドの終了を待機\n");
    nnosWaitThread(&thread);

    // nnosGetThreadNamePointer
    NNC_LOG("nnosGetThreadNamePointer() で子スレッドのスレッド名を出力\n");
    {
        const char* name = nnosGetThreadNamePointer(&thread);
        NNC_LOG("thead name = \x22%s\x22\n", name);
    }

    // nnosSleepThread
    NNC_LOG("nnosSleepThread() で 100msec 休止\n");
    nnosSleepThread(100 * 1000 * 1000 /* 100msec */);

    // nnosYieldThread
    NNC_LOG("nnosYieldThread() を発行\n");
    nnosYieldThread();

    // nnosChangeThreadPriority
    NNC_LOG("nnosChangeThreadPriority(3) を優先度を変更\n");
    nnosChangeThreadPriority(&thread, 3);

    // nnosGetThreadPriority
    NNC_LOG("nnosGetThreadPriority() で優先度を取得\n");
    pri = nnosGetThreadPriority(&thread);
    NNC_LOG("nnosGetThreadPriority() = %d\n", pri);

    // nnosGetThreadCurrentPriority
    pri = nnosGetThreadCurrentPriority(&thread);
    NNC_LOG("nnosGetThreadCurrentPriority() = %d\n", pri);

    // nnosGetCurrentCoreNumber
    core = nnosGetCurrentCoreNumber();
    NNC_LOG("nnosGetCurrentCoreNumber() = %d\n", core);

    // nnosDestroyThread
    NNC_LOG("nnosDestroyThread() で子スレッドを破棄\n");
    nnosDestroyThread(&thread);

    NNC_LOG("nnosThread 関連のテスト終了\n");
    return true;
}

//-----------------------------------------------------------------------------

bool nntosEventTestForC(void)
{
    nnosEventType   event;
    bool            result;

    NNC_LOG("nnosEvent 関連のテスト開始\n");

    // nnosInitializeEvent
    NNC_LOG("nnosInitializeEvent(ManualClear) でイベントを初期化\n");

    // AutoClear の列挙子を参照できているかのテスト
    nnosInitializeEvent(&event, false, nnosEventClearMode_AutoClear);
    nnosFinalizeEvent(&event);

    // ManualClear の列挙子を参照できているかのテスト
    nnosInitializeEvent(&event, true, nnosEventClearMode_ManualClear);

    // nnosWaitEvent
    NNC_LOG("nnosWaitEvent() で待機\n");
    nnosWaitEvent(&event);

    // nnosClearEvent
    NNC_LOG("nnosClearEvent() でクリア\n");
    nnosClearEvent(&event);

    // nnosTryWaitEvent で false を期待
    NNC_LOG("nnosTryWaitEvent() で false を期待\n");
    result = nnosTryWaitEvent(&event);
    NNT_OS_RETURN_FALSE_UNLESS( !result );

    // nnosTimedWaitEvent() で false を期待
    NNC_LOG("nnosTimedWaitEvent で false を期待（タイムアウト成立）\n");
    result = nnosTimedWaitEvent(&event, 100 * 1000 * 1000); // 100 msec
    NNT_OS_RETURN_FALSE_UNLESS( !result );

    // nnosSignalEvent
    NNC_LOG("nnosSignalEvent() でセット\n");
    nnosSignalEvent(&event);

    // nnosTryWaitEvent で true を期待
    NNC_LOG("nnosTryWaitEvent() で true を期待\n");
    result = nnosTryWaitEvent(&event);
    NNT_OS_RETURN_FALSE_UNLESS( result );

    // nnosTimedWaitEvent() で true を期待
    NNC_LOG("nnosTimedWaitEvent で true を期待\n");
    result = nnosTimedWaitEvent(&event, 100 * 1000 * 1000);   // 100 msec
    NNT_OS_RETURN_FALSE_UNLESS( result );

    // nnosFinalizeEvent
    NNC_LOG("nnosFinalizeEvent() でイベントを破棄\n");
    nnosFinalizeEvent(&event);

    NNC_LOG("nnosEvent 関連のテスト終了\n");
    return true;
}

//-----------------------------------------------------------------------------

bool nntosInterruptEventTestForC(void)
{
    nnosInterruptEventType  event;
    bool                    result;

    NNC_LOG("nnosInterruptEvent 関連のテスト開始\n");

    // nnosInitializeEvent
    NNC_LOG("nnosInitializeInterruptEvent(ManualClear) でイベントを初期化\n");

    // AutoClear の列挙子を参照できているかのテスト
    nnosInitializeInterruptEvent(&event, g_InterruptName, nnosEventClearMode_AutoClear);
    nnosFinalizeInterruptEvent(&event);

    // ManualClear の列挙子を参照できているかのテスト
    nnosInitializeInterruptEvent(&event, g_InterruptName, nnosEventClearMode_ManualClear);


    // nnosClearInterruptEvent
    NNC_LOG("nnosClearInterruptEvent() でクリア\n");
    nnosClearInterruptEvent(&event);

    // nnosTryWaitInterruptEvent で false を期待
    NNC_LOG("nnosTryWaitInterruptEvent() で false を期待\n");
    result = nnosTryWaitInterruptEvent(&event);
    NNT_OS_RETURN_FALSE_UNLESS( !result );
    if (result == true)
    {
        // ここの if 文の中には入って来ないが、
        // nnosWaitInterruptEvent() のリンクが正しく成功することを確認するため、
        // 最適化によってコードが排除されない場所に以下の記述をしておく。

        // nnosWaitInterruptEvent
        NNC_LOG("nnosWaitInterruptEvent() で待機\n");
        nnosWaitInterruptEvent(&event);
    }

    // nnosTimedWaitInterruptEvent() で false を期待
    NNC_LOG("nnosTimedWaitInterruptEvent で false を期待（タイムアウト成立）\n");
    result = nnosTimedWaitInterruptEvent(&event, 100 * 1000 * 1000); // 100 msec
    NNT_OS_RETURN_FALSE_UNLESS( !result );

    // nnosFinalizeInterruptEvent
    NNC_LOG("nnosFinalizeInterruptEvent() でイベントを破棄\n");
    nnosFinalizeInterruptEvent(&event);

    NNC_LOG("nnosInterruptEvent 関連のテスト終了\n");
    return true;
}

//-----------------------------------------------------------------------------

bool nntosSemaphoreTestForC(void)
{
    nnosSemaphoreType   semaphore;
    bool                result;
    int                 count;
    int                 i;

    NNC_LOG("nnosSemaphore 関連のテスト開始\n");

    // nnosInitializeSemaphore
    NNC_LOG("nnosInitializeSemaphore(current=1, max=5) でセマフォを初期化\n");
    nnosInitializeSemaphore(&semaphore, 1, 5);

    // nnosAcquireSemaphore
    NNC_LOG("nnosAcquireSemaphore() でセマフォを 1 つ獲得\n");
    nnosAcquireSemaphore(&semaphore);

    // nnosReleaseSemaphore
    NNC_LOG("nnosReleaseSemaphore() でセマフォを 1 つ返却\n");
    nnosReleaseSemaphore(&semaphore);

    // nnosReleaseSemaphoreCount
    NNC_LOG("nnosReleaseSemaphoreCount() でセマフォを 4 つ返却\n");
    nnosReleaseSemaphoreCount(&semaphore, 4);

    // nnosGetCurrentSemaphoreCount
    count = nnosGetCurrentSemaphoreCount(&semaphore);
    NNC_LOG("nnosSemaphore のカウント = %d\n", count);
    NNT_OS_RETURN_FALSE_UNLESS( count == 5 );

    // nnosTryAcquireSemaphore
    NNC_LOG("nnosTryAcquireSemaphore() を 4 回発行\n");
    for (i = 0; i < 4; ++i)
    {
        result = nnosTryAcquireSemaphore(&semaphore);
        NNT_OS_RETURN_FALSE_UNLESS( result );
    }

    // nnosTimedAcquireSemaphore
    NNC_LOG("nnosTimedAcquireSemaphore(1usec) で true を確認\n");
    result = nnosTimedAcquireSemaphore(&semaphore, 1000 /* nsec */);
    NNT_OS_RETURN_FALSE_UNLESS( result );

    // nnosGetCurrentSemaphoreCount
    count = nnosGetCurrentSemaphoreCount(&semaphore);
    NNC_LOG("nnosSemaphore のカウント = %d\n", count);
    NNT_OS_RETURN_FALSE_UNLESS( count == 0 );

    // nnosTryAcquireSemaphore
    NNC_LOG("nnosTryAcquireSemaphore() で false を確認\n");
    result = nnosTryAcquireSemaphore(&semaphore);
    NNT_OS_RETURN_FALSE_UNLESS( !result );

    // nnosTimedAcquireSemaphore
    NNC_LOG("nnosTimedAcquireSemaphore(1usec) で false を確認\n");
    result = nnosTimedAcquireSemaphore(&semaphore, 1000 /* nsec */);
    NNT_OS_RETURN_FALSE_UNLESS( !result );

    // nnosFinalizeSemaphore
    NNC_LOG("nnosFinalizeSemaphore() でセマフォを破棄\n");
    nnosFinalizeSemaphore(&semaphore);

    NNC_LOG("nnosSemaphore 関連のテスト終了\n");
    return true;
}

//-----------------------------------------------------------------------------

bool nntosTimerEventTestForC(void)
{
    nnosTimerEventType  timerEvent;
    bool                result;

    NNC_LOG("nnosTimerEvent 関連のテスト開始\n");

    // nnosInitializeTimerEvent
    NNC_LOG("nnosInitializeTimerEvent(ManualClear) でタイマーイベントを初期化\n");

    // AutoClear を参照できているかのテスト
    nnosInitializeTimerEvent(&timerEvent, nnosEventClearMode_AutoClear);
    nnosFinalizeTimerEvent(&timerEvent);

    // ManualClear を参照できているかのテスト
    nnosInitializeTimerEvent(&timerEvent, nnosEventClearMode_ManualClear);

    // nnosStartOneShotTimerEvent()
    NNC_LOG("nnosStartOneShotTimerEvent(10msec) でタイマー開始\n");
    nnosStartOneShotTimerEvent(&timerEvent, 10 * 1000 * 1000 /* nsec */);

    // nnosWaitTimerEvent
    NNC_LOG("nnosWaitTimerEvent() でタイマーを待機\n");
    nnosWaitTimerEvent(&timerEvent);

    // nnosClearTimerEvent
    NNC_LOG("nnosClearTimerEvent() でクリア\n");
    nnosClearTimerEvent(&timerEvent);

    // nnosStartPeriodicTimerEvent()
    NNC_LOG("nnosStartPeriodicTimerEvent(500msec, 5usec) でタイマー開始\n");
    nnosStartPeriodicTimerEvent(&timerEvent, 500 * 1000 * 1000 /* nsec */,
                                                      5 * 1000 /* nsec */);

    // nnosTryWaitTimerEvent
    NNC_LOG("nnosTryWaitTimerEvent() で false を期待\n");
    result = nnosTryWaitTimerEvent(&timerEvent);
    NNT_OS_RETURN_FALSE_UNLESS( !result );

    // nnosStopTimerEvent
    NNC_LOG("nnosStopTimerEvent() でタイマー停止\n");
    nnosStopTimerEvent(&timerEvent);

    // nnosSignalTimerEvent
    NNC_LOG("nnosSignalTimerEvent() でセット\n");
    nnosSignalTimerEvent(&timerEvent);

    // nnosTryWaitTimerEvent
    NNC_LOG("nnosTryWaitTimerEvent() で true を期待\n");
    result = nnosTryWaitTimerEvent(&timerEvent);
    NNT_OS_RETURN_FALSE_UNLESS( result );

    // nnosFinalizeTimerEvent
    NNC_LOG("nnosFinalizeTimerEvent() でタイマーイベントを破棄\n");
    nnosFinalizeTimerEvent(&timerEvent);


    NNC_LOG("nnosTimerEvent 関連のテスト終了\n");
    return true;
}

//-----------------------------------------------------------------------------

bool nntosMutexTestForC(void)
{
    nnosMutexType   mutex;
    bool            result;

    NNC_LOG("nnosMutex 関連のテスト開始\n");

    // nnosInitializeMutex
    NNC_LOG("nnosInitializeMutex() でミューテックスを初期化\n");
    nnosInitializeMutex(&mutex, true, 0);

    // nnosLockMutex()
    NNC_LOG("nnosLockMutex() でロック\n");
    nnosLockMutex(&mutex);

    // nnosTryLockMutex()
    NNC_LOG("nnosTryLockMutex() でロック（true を確認）\n");
    result = nnosTryLockMutex(&mutex);
    NNT_OS_RETURN_FALSE_UNLESS( result );

    // nnosUnlockMutex
    NNC_LOG("nnosUnlockMutex() でアンロックを 2 回\n");
    nnosUnlockMutex(&mutex);
    nnosUnlockMutex(&mutex);

    // nnosFinalizeMutex
    NNC_LOG("nnosFinalizeMutex() でミューテックスを破棄\n");
    nnosFinalizeMutex(&mutex);

    NNC_LOG("nnosMutex 関連のテスト終了\n");
    return true;
}

//-----------------------------------------------------------------------------

nnosMutexType             g_Mutex = NN_OS_MUTEX_INITIALIZER(false);
nnosConditionVariableType g_Cond  = NN_OS_CONDITION_VARIABLE_INITIALIZER();

bool            g_CondFlag;
nnosEventType   g_Event;

void nntosConditionVariableTestForCSubThread(void* arg)
{
    nnosConditionVariableType* cond = (nnosConditionVariableType*)arg;

    NNC_LOG("子スレッドから nnosSignalConditionVariable() でシグナル通知\n");
    nnosLockMutex(&g_Mutex);
    g_CondFlag = true;
    nnosSignalConditionVariable(cond);
    nnosUnlockMutex(&g_Mutex);

    // 一旦親スレッドへ制御を移すため、イベント待ちに入る。
    nnosWaitEvent(&g_Event);

    NNC_LOG("子スレッドから nnosBroadcastConditionVariable() でシグナル通知\n");
    nnosLockMutex(&g_Mutex);
    g_CondFlag = true;
    nnosBroadcastConditionVariable(cond);
    nnosUnlockMutex(&g_Mutex);
}

bool nntosConditionVariableTestForCImpl(nnosConditionVariableType* pCond)
{
    nnosThreadType  thread;
    nnResult        result;

    g_CondFlag  = false;

    // 子スレッドと同期用のイベントを初期化しておく
    nnosInitializeEvent(&g_Event, false, nnosEventClearMode_ManualClear);

    NNC_LOG("nnosConditionVariable 関連のテスト開始\n");

    // 動作確認用の子スレッドを生成
    NNC_LOG("子スレッドを生成（優先度低）\n");
    result = nnosCreateThread(&thread, nntosConditionVariableTestForCSubThread, pCond, g_Stack, sizeof(g_Stack), NN_OS_LOWEST_THREAD_PRIORITY);
    NNT_OS_RETURN_FALSE_UNLESS( nnResultIsSuccess(result) );

    // nnosLockMutex()
    NNC_LOG("nnosLockMutex() でロック\n");
    nnosLockMutex(&g_Mutex);

    // 子スレッドを起動
    NNC_LOG("nnosStartThread() で子スレッドを起動\n");
    nnosStartThread(&thread);

    // nnosWaitConditionVariable
    NNC_LOG("nnosWaitConditionVariable() で条件変数を待機\n");
    while (g_CondFlag == false)
    {
        nnosWaitConditionVariable(pCond, &g_Mutex);
    }
    g_CondFlag = false;

    // 子スレッドの動作を再開する
    nnosSignalEvent(&g_Event);

    // nnosTimedWaitConditionVariable
    NNC_LOG("nnosTimedWaitConditionVariable(10sec) で条件変数を待機\n");
    while (g_CondFlag == false)
    {
        nnosConditionVariableStatus ret = nnosTimedWaitConditionVariable(pCond, &g_Mutex, 10ll * 1000 * 1000 * 1000 /*nsec*/);
        if (ret == nnosConditionVariableStatus_Timeout)
        {
            nnosUnlockMutex(&g_Mutex);
            return false;
        }
    }

    // nnosUnlockMutex()
    NNC_LOG("nnosUnlockMutex() でアンロック\n");
    nnosUnlockMutex(&g_Mutex);

    // 子スレッドの終了待ち
    NNC_LOG("子スレッドの終了待ち\n");
    nnosWaitThread(&thread);
    nnosDestroyThread(&thread);

    NNC_LOG("nnosConditionVariable 関連のテスト終了\n");

    nnosFinalizeEvent(&g_Event);
    return true;
}


bool nntosConditionVariableTestForC(void)
{
    nnosConditionVariableType   cond;
    bool result;

    // nnosInitializeConditionVariable
    NNC_LOG("nnosInitializeConditionVariable() で条件変数を初期化\n");
    nnosInitializeConditionVariable(&cond);

    result = nntosConditionVariableTestForCImpl(&cond);

    // nnosFinalizeConditionVariable
    NNC_LOG("nnosFinalizeConditionVariable() で条件変数を破棄\n");
    nnosFinalizeConditionVariable(&cond);

    if (!result)
    {
        return false;
    }

    // NN_OS_CONDITION_VARIABLE_INITIALIZER で初期化された条件変数のテスト
    NNC_LOG("静的に初期化された条件変数オブジェクトのテスト\n");
    return nntosConditionVariableTestForCImpl(&g_Cond);
}


//-----------------------------------------------------------------------------

bool nntosSystemEventTestForC(void)
{
    nnosSystemEventType systemEvent;
    nnosSystemEventType systemEvent2;
    nnosNativeHandle    readableHandle;
    nnosNativeHandle    writableHandle;
    nnosNativeHandle    invalidReadableHandle;
    nnosNativeHandle    invalidWritableHandle;
    bool                result;

    // Attach, detach 系のテスト
    NNC_LOG("nnosSystemEvent の attach / detach テスト\n");
    nnosCreateSystemEvent(&systemEvent, nnosEventClearMode_ManualClear, true);
    readableHandle = nnosDetachReadableHandleOfSystemEvent(&systemEvent);
    writableHandle = nnosDetachWritableHandleOfSystemEvent(&systemEvent);
    if (readableHandle == NN_OS_INVALID_NATIVE_HANDLE || writableHandle == NN_OS_INVALID_NATIVE_HANDLE)
    {
        NNC_LOG("readableHandle=0x%x\n", readableHandle);
        NNC_LOG("writableHandle=0x%x\n", writableHandle);
        return false;
    }
    invalidReadableHandle = nnosGetReadableHandleOfSystemEvent(&systemEvent);
    invalidWritableHandle = nnosGetWritableHandleOfSystemEvent(&systemEvent);
    if (invalidReadableHandle != NN_OS_INVALID_NATIVE_HANDLE || invalidWritableHandle != NN_OS_INVALID_NATIVE_HANDLE)
    {
        NNC_LOG("invalidReadableHandle=0x%x\n", invalidReadableHandle);
        NNC_LOG("invalidWritableHandle=0x%x\n", invalidWritableHandle);
        return false;
    }

    nnosAttachSystemEvent(&systemEvent2, readableHandle, 0, writableHandle, 0, nnosEventClearMode_AutoClear);
    nnosDestroySystemEvent(&systemEvent2);

    nnosAttachReadableHandleToSystemEvent(&systemEvent2, readableHandle, 0, nnosEventClearMode_AutoClear);
    nnosDestroySystemEvent(&systemEvent2);

    nnosAttachWritableHandleToSystemEvent(&systemEvent2, writableHandle, 0, nnosEventClearMode_AutoClear);
    nnosDestroySystemEvent(&systemEvent2);

    nnosDestroySystemEvent(&systemEvent);

    // デタッチした NativeHandle のクローズ
    nnosCloseNativeHandle(readableHandle);
    nnosCloseNativeHandle(writableHandle);

    // Signal, wait テスト用に再度システムイベントを作成
    NNC_LOG("nnosSystemEvent の signal / wait テスト\n");
    nnosCreateSystemEvent(&systemEvent, nnosEventClearMode_ManualClear, true);

    nnosSignalSystemEvent(&systemEvent);
    nnosWaitSystemEvent(&systemEvent);
    result = nnosTryWaitSystemEvent(&systemEvent);
    if (!result)
    {
        NNC_LOG("nnosTryWaitSystemEvent failed.\n");
        return false;
    }

    result = nnosTimedWaitSystemEvent(&systemEvent, 1000000LL); // 1ms
    if (!result)
    {
        NNC_LOG("nnosTimedWaitSystemEvent failed.\n");
        return false;
    }

    nnosClearSystemEvent(&systemEvent);
    nnosDestroySystemEvent(&systemEvent);

    return true;
}

//-----------------------------------------------------------------------------

bool nntosMessageQueueTestForC(void)
{
    nnosMessageQueueType    messageQueue;
    uintptr_t               messageQueueBuffer[8];
    uintptr_t               data;

    nnosInitializeMessageQueue(&messageQueue, messageQueueBuffer, 8);

    // Send and Jam
    NNC_LOG("nnosMessageQueue の Send / Jam テスト\n");
    nnosSendMessageQueue(&messageQueue, 0x11111111);
    nnosTrySendMessageQueue(&messageQueue, 0x22222222);
    nnosTimedSendMessageQueue(&messageQueue, 0x33333333, 1000000LL); // 1ms
    nnosJamMessageQueue(&messageQueue, 0x44444444);
    nnosTryJamMessageQueue(&messageQueue, 0x55555555);
    nnosTimedJamMessageQueue(&messageQueue, 0x66666666, 1000000LL); // 1ms

    // この時点でメッセージキューの中身は以下のようになるはず
    // 0x66666666 <- 先頭
    // 0x55555555
    // 0x44444444
    // 0x11111111
    // 0x22222222
    // 0x33333333 <- 末尾

    // Receive and Peek
    NNC_LOG("nnosMessageQueue の Receive / Peek テスト\n");
    nnosReceiveMessageQueue(&data, &messageQueue);
    if (data != 0x66666666)
    {
        NNC_LOG("nnosReceiveMessageQueue: data=0x%x (should be 0x66666666)\n", data);
        return false;
    }
    nnosTryReceiveMessageQueue(&data, &messageQueue);
    if (data != 0x55555555)
    {
        NNC_LOG("nnosTryReceiveMessageQueue: data=0x%x (should be 0x55555555)\n", data);
        return false;
    }
    nnosTimedReceiveMessageQueue(&data, &messageQueue, 1000000LL); // 1ms
    if (data != 0x44444444)
    {
        NNC_LOG("nnosTimedReceiveMessageQueue: data=0x%x (should be 0x44444444)\n", data);
        return false;
    }
    nnosPeekMessageQueue(&data, &messageQueue);
    if (data != 0x11111111)
    {
        NNC_LOG("nnosPeekMessageQueue: data=0x%x (should be 0x11111111)\n", data);
        return false;
    }
    nnosTryPeekMessageQueue(&data, &messageQueue);
    if (data != 0x11111111)
    {
        NNC_LOG("nnosTryPeekMessageQueue: data=0x%x (should be 0x11111111)\n", data);
        return false;
    }
    nnosTimedPeekMessageQueue(&data, &messageQueue, 1000000LL); // 1ms
    if (data != 0x11111111)
    {
        NNC_LOG("nnosTimedPeekMessageQueue: data=0x%x (should be 0x11111111)\n", data);
        return false;
    }

    nnosFinalizeMessageQueue(&messageQueue);

    return true;
}

//-----------------------------------------------------------------------------

bool nntosMultipleWaitTestForC(void)
{
    nnosMultiWaitType           multiWait;
    nnosMultiWaitHolderType     holder[7];
    nnosEventType               event;
    nnosSemaphoreType           semaphore;
    nnosTimerEventType          timerEvent;
    nnosSystemEventType         systemEvent;
    nnosMessageQueueType        messageQueue;
    nnosThreadType              thread;
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nnosInterruptEventType      interruptEvent;
#endif
    nnosMultiWaitHolderType*    signaledHolder;
    uintptr_t                   value;
    nnResult                    result;
    int                         i;
    int                         numHolders = 6;
    uintptr_t                   messageQueueBuffer[4];

    // 最初は非シグナル状態のイベントを用意
    nnosInitializeEvent(&event, true, nnosEventClearMode_ManualClear);

    // 初期カウント 0 のセマフォを用意
    nnosInitializeSemaphore(&semaphore, 0, 1);

    // タイマーイベントを用意
    nnosInitializeTimerEvent(&timerEvent, nnosEventClearMode_AutoClear);

    // システムイベントを用意
    nnosCreateSystemEvent(&systemEvent, nnosEventClearMode_AutoClear, false);

    // メッセージキューを用意
    nnosInitializeMessageQueue(&messageQueue, messageQueueBuffer, 4);

    // スレッドを用意
    result = nnosCreateThread(&thread, nntosThreadTestSubThread, &thread, g_Stack, sizeof(g_Stack), NN_OS_DEFAULT_THREAD_PRIORITY);
    NNT_OS_RETURN_FALSE_UNLESS( nnResultIsSuccess(result) );

    // Horizon でのみテストする
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // 割り込みイベントを用意
    nnosInitializeInterruptEvent(&interruptEvent, g_InterruptName, nnosEventClearMode_AutoClear);
#endif

    // 多重待ちオブジェクトリストを構築
    nnosInitializeMultiWait( &multiWait );

    // イベントをリンク
    nnosInitializeMultiWaitHolderForEvent( &holder[0], &event );
    nnosLinkMultiWaitHolder( &multiWait, &holder[0] );
    nnosSetMultiWaitHolderUserData( &holder[0], 0x11111111 );

    // セマフォをリンク
    nnosInitializeMultiWaitHolderForSemaphore( &holder[1], &semaphore );
    nnosLinkMultiWaitHolder( &multiWait, &holder[1] );
    nnosSetMultiWaitHolderUserData( &holder[1], 0x22222222 );

    // タイマーイベントをリンク
    nnosInitializeMultiWaitHolderForTimerEvent( &holder[2], &timerEvent );
    nnosLinkMultiWaitHolder( &multiWait, &holder[2] );
    nnosSetMultiWaitHolderUserData( &holder[2], 0x33333333 );

    // システムイベントをリンク
    nnosInitializeMultiWaitHolderForSystemEvent( &holder[3], &systemEvent );
    nnosLinkMultiWaitHolder( &multiWait, &holder[3] );
    nnosSetMultiWaitHolderUserData( &holder[3], 0x44444444 );

    // メッセージキューをリンク
    nnosInitializeMultiWaitHolderForMessageQueue( &holder[4], &messageQueue, nnosMessageQueueWaitType_WaitForNotEmpty );
    nnosLinkMultiWaitHolder( &multiWait, &holder[4] );
    nnosSetMultiWaitHolderUserData( &holder[4], 0x55555555 );

    // スレッドをリンク
    nnosInitializeMultiWaitHolderForThread( &holder[5], &thread );
    nnosLinkMultiWaitHolder( &multiWait, &holder[5] );
    nnosSetMultiWaitHolderUserData( &holder[5], 0x66666666 );

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // 割り込みイベントをリンク
    ++numHolders;
    nnosInitializeMultiWaitHolderForInterruptEvent( &holder[6], &interruptEvent );
    nnosLinkMultiWaitHolder( &multiWait, &holder[6] );
    nnosSetMultiWaitHolderUserData( &holder[6], 0x77777777 );
#endif

    // event で多重待ち解除する
    {
        NNC_LOG("WaitAny( {event, semaphore, timerEvent, thread, (interruptEvent)} ) expect=true\n");

        // WaitAny を発行（すぐに戻ってくる）
        signaledHolder = nnosWaitAny( &multiWait );

        // シグナルが成立している Holder オブジェクトが正しいか？
        if (signaledHolder != &holder[0])
        {
            NNC_LOG("nnosWaitAny failed. signaledHolder != &holder[0].\n");
            return false;
        }

        // ユーザデータ値が正しいか？
        value = nnosGetMultiWaitHolderUserData(signaledHolder);
        NNC_LOG("SignaledHolder's UserData=0x%08x\n", value);
        if (value != 0x11111111)
        {
            return false;
        }

        // TryWaitAny を発行（すぐに戻ってくる）
        signaledHolder = nnosTryWaitAny( &multiWait );

        // シグナルが成立している Holder オブジェクトが正しいか？
        if (signaledHolder != &holder[0])
        {
            NNC_LOG("nnosTryWaitAny failed. signaledHolder != &holder[0].\n");
            return false;
        }

        // TryWaitAny を発行（すぐに戻ってくる）
        signaledHolder = nnosTimedWaitAny( &multiWait, 1000000 );

        // シグナルが成立している Holder オブジェクトが正しいか？
        if (signaledHolder != &holder[0])
        {
            NNC_LOG("nnosTimedWaitAny failed. signaledHolder != &holder[0].\n");
            return false;
        }
    }

    // Holder をリストから外す＆ファイナライズ
    for (i=0; i<numHolders; ++i)
    {
        nnosUnlinkMultiWaitHolder( &holder[i] );
        nnosFinalizeMultiWaitHolder( &holder[i] );
    }

    // ファイナライズ
    nnosFinalizeMultiWait( &multiWait );

    nnosFinalizeEvent(&event);
    nnosFinalizeSemaphore(&semaphore);
    nnosFinalizeTimerEvent(&timerEvent);
    nnosDestroySystemEvent(&systemEvent);
    nnosFinalizeMessageQueue(&messageQueue);
    nnosDestroyThread(&thread);
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    nnosFinalizeInterruptEvent(&interruptEvent);
#endif

    return true;
}

//-----------------------------------------------------------------------------

volatile bool g_TlsDestructorFlag;

void nntosTlsDestructor(uintptr_t value)
{
    NNC_LOG("nntosTlsDestructor is called.\n");
    if (value != 0xbeef1234)
    {
        NNC_LOG("TLS value is not 0xbeef1234! (=0x%x)\n", value);
        return;
    }
    g_TlsDestructorFlag = true;
}

void nntosThreadLocalStorageTestForCSubThread(void* arg)
{
    nnosTlsSlot slot = *(nnosTlsSlot*)arg;
    nnosSetTlsValue(slot, 0xbeef1234);
}

bool nntosThreadLocalStorageTestForC(void)
{
    nnosThreadType thread;
    nnosTlsSlot    slot;
    nnResult       result;
    int            initialUsedTlsCount;
    int            usedTlsCount;
    uintptr_t      value;

    NNC_LOG("nnosTlsSlot 関連のテスト開始\n");

    g_TlsDestructorFlag = false;

    // アプリ用の確保済み TLS の数をカウント
    NNC_LOG("アプリ用の確保済み TLS の数をカウント\n");
    initialUsedTlsCount = nnosGetUsedTlsSlotCount();

    // TLS を確保
    NNC_LOG("TLS を確保\n");
    result = nnosAllocateTlsSlot(&slot, nntosTlsDestructor);
    NNT_OS_RETURN_FALSE_UNLESS( nnResultIsSuccess(result) );

    // アプリ用の確保済み TLS の数が減っていないことを確認
    NNC_LOG("アプリ用の確保済み TLS の数をカウント\n");
    usedTlsCount = nnosGetUsedTlsSlotCount();
    NNT_OS_RETURN_FALSE_UNLESS( usedTlsCount == initialUsedTlsCount );

    // TLS への読み書き
    NNC_LOG("TLS への読み書き\n");
    nnosSetTlsValue(slot, 0xbeefbeef);
    value = nnosGetTlsValue(slot);
    NNT_OS_RETURN_FALSE_UNLESS( value == 0xbeefbeef);

    // スレッド終了時に TLS のデストラクタ関数が呼ばれるか確認
    result = nnosCreateThread(&thread, nntosThreadLocalStorageTestForCSubThread, (void*)&slot, g_Stack, sizeof(g_Stack), NN_OS_LOWEST_THREAD_PRIORITY);
    nnosStartThread(&thread);
    nnosWaitThread(&thread);

    // TLS のデストラクタ関数が呼ばれたかどうか確認
    NNC_LOG("TLS のデストラクタ関数が呼ばれたかどうか確認\n");
    NNT_OS_RETURN_FALSE_UNLESS( g_TlsDestructorFlag );

    // 解放
    NNC_LOG("解放\n");
    nnosFreeTlsSlot(slot);

    // アプリ用の確保済み TLS の数をカウント
    NNC_LOG("アプリ用の確保済み TLS の数をカウント\n");
    usedTlsCount = nnosGetUsedTlsSlotCount();
    NNT_OS_RETURN_FALSE_UNLESS( usedTlsCount == initialUsedTlsCount );

    NNC_LOG("nnosTlsSlot 関連のテスト終了\n");
    return true;
}

//-----------------------------------------------------------------------------

bool nntosTickTestForC(void)
{
    nnosTick start;
    nnosTick end;
    int64_t  freq;
    int64_t  duration;
    nnosTick durationTick;

    freq = nnosGetSystemTickFrequency();
    NNC_LOG("system tick frequency = %lld\n", freq);

    start = nnosGetSystemTick();
    NNC_LOG("start tick = %lld\n", start);

    end = nnosGetSystemTick();
    NNC_LOG("end tick = %lld\n", end);

    duration = nnosConvertTickToNanoSeconds(end - start);
    durationTick = nnosConvertNanoSecondsToTick(duration);
    NNC_LOG("duration = %lld ns (%lld tick)\n", duration, durationTick);

    NNT_OS_RETURN_FALSE_UNLESS( (end - start) == durationTick );

    return true;
}

//-----------------------------------------------------------------------------

bool nntosMemoryFenceTestForC(void)
{
    NNC_LOG("nnosMemoryFence 関連のテスト開始\n");

    nnosFenceMemoryStoreStore();
    nnosFenceMemoryStoreLoad();
    nnosFenceMemoryStoreAny();
    nnosFenceMemoryLoadStore();
    nnosFenceMemoryLoadLoad();
    nnosFenceMemoryLoadAny();
    nnosFenceMemoryAnyStore();
    nnosFenceMemoryAnyLoad();
    nnosFenceMemoryAnyAny();

    NNC_LOG("nnosMemoryFence 関連のテスト終了\n");

    return true;
}

//-----------------------------------------------------------------------------
