﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>

#include <nn/init.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_VirtualAddressMemory.h>
#include <nn/util/util_TinyMt.h>

#include <nnt.h>

#include "test_VirtualAddressMemoryCommon.h"

// 64bit のみ対応
#if !(defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) || defined(NN_BUILD_CONFIG_CPU_X64))
    #error "This program supports only 64bit architecture."
#endif

//---------------------------------------------------------------------------
//  VirtualAddressMemory 機能のストレステスト
//---------------------------------------------------------------------------

namespace
{
#if defined(NNT_OS_APPLET)
    // Applet region (~450MB) - heap size - usage of other processes on develop (~90MB) - code/data/bss - margin
    static const size_t MaxAllocatedMemory = 96 * 1024 * 1024UL;
    const size_t HeapSize = 128 * 1024 * 1024UL;
#else
    static const size_t MaxAllocatedMemory = 1024 * 1024 * 1024;
    const size_t HeapSize = 256 * 1024 * 1024UL;
#endif
    uint8_t g_Heap[HeapSize];

    nnt::os::VirtualAddressMemory::CommandProcessor g_Processor;

    // 共通の正常系テストパターン
    void SmokeTest()
    {
        // 1GB のリージョンで 4KB の割り当てを繰り返して性能測定
        nnt::os::VirtualAddressMemory::MemoryPagePerformanceTest(MaxAllocatedMemory, 4 * 1024);

        // 1GB のリージョンで 2MB の割り当てを繰り返して性能測定
        nnt::os::VirtualAddressMemory::MemoryPagePerformanceTest(MaxAllocatedMemory, 2 * 1024 * 1024);

        // 2 MB のリージョン割り当てを 1024 回繰り返して性能測定
        nnt::os::VirtualAddressMemory::AddressRegionPerformanceTest(2 * 1024 * 1024, 1024);
    }
}

namespace nnt { namespace os { namespace VirtualAddressMemory {

TEST(VirtualAddressMemoryStressBasic, SmokeTest)
{
    SmokeTest();

    ConfirmResourceUsage(16 * 1024 * 1024);
}

TEST(VirtualAddressMemoryStressBasic, ExcessAddressRegionApi)
{
    std::vector<uintptr_t> regionAddresses;
    size_t regionSize = 1024 * 1024;
    size_t size1GB = 1024 * 1024 * 1024UL;
    size_t apiCallCount = 128 * size1GB / regionSize; // 128GB の仮想アドレス空間は確保できないと仮定

    // 過剰にリージョンの確保を試行し、エラーを起こす
    NN_LOG("Excess Allocate Address Region\n");
    bool failed = false;
    for (int i = 0; i < apiCallCount; i++)
    {
        uintptr_t address;
        auto result = nn::os::AllocateAddressRegion(&address, regionSize);

        if (result.IsSuccess())
        {
            regionAddresses.push_back(address);
        }
        else
        {
            failed = true;
            break;
        }
    }
    ASSERT_TRUE(failed);

    // 過剰にリージョンの解放を試行し、エラーを起こす
    NN_LOG("Excess Free Address Region\n");
    for (auto address : regionAddresses)
    {
        auto result = nn::os::FreeAddressRegion(address);
        ASSERT_TRUE(result.IsSuccess());
    }
    for (int i = 0; i < apiCallCount - regionAddresses.size(); i++)
    {
        int index = i % regionAddresses.size();
        auto result = nn::os::FreeAddressRegion(regionAddresses[index]);
        ASSERT_TRUE(result.IsFailure());
    }

    // SmokeTest を問題なく実行できることを確認
    SmokeTest();

    ConfirmResourceUsage(16 * 1024 * 1024);
}

TEST(VirtualAddressMemoryStressBasic, ExcessMemoryPageApi)
{
    uintptr_t regionAddress;
    size_t regionSize = MaxAllocatedMemory;
    size_t pageSize = 1024 * 1024UL;
    size_t apiCallCount = 16 * regionSize / pageSize; // 正常系で呼べる最大値の16倍

    // リージョンを確保する
    nn::os::AllocateAddressRegion(&regionAddress, regionSize);

    // 過剰にページの確保を試行し、エラーを起こす
    NN_LOG("Excess Allocate Memory Pages\n");
    for (int i = 0; i < apiCallCount; i++)
    {
        auto result = nn::os::AllocateMemoryPages(regionAddress + i * pageSize, pageSize);

        if (i < regionSize / pageSize)
        {
            ASSERT_TRUE(result.IsSuccess());
        }
        else
        {
            ASSERT_TRUE(result.IsFailure());
        }
    }

    // 過剰にページの解放を試行し、エラーを起こす
    NN_LOG("Excess Free Memory Pages\n");
    for (int i = 0; i < apiCallCount; i++)
    {
        auto result = nn::os::FreeMemoryPages(regionAddress + i * pageSize, pageSize);

        if (i < regionSize / pageSize)
        {
            ASSERT_TRUE(result.IsSuccess());
        }
        else
        {
            ASSERT_TRUE(result.IsFailure());
        }
    }

    // SmokeTest を問題なく実行できることを確認
    SmokeTest();

    // リージョンを解放する
    nn::os::FreeAddressRegion(regionAddress);

    ConfirmResourceUsage(16 * 1024 * 1024);
}

// 小さいページをたくさん Allocate / Free する
TEST(VirtualAddressMemoryStressBasic, ManySmallPages)
{
    nn::os::MemoryInfo memInfo;
    nn::os::QueryMemoryInfo(&memInfo);

    const size_t regionSize = MaxAllocatedMemory;
    ASSERT_LE(regionSize, memInfo.totalAvailableMemorySize - memInfo.totalUsedMemorySize);

    const size_t repeatCount = 10;

    // たくさん Allocate / Free するページのサイズ
    const size_t pageSize = 4096;

    // 最初のページのみ Allocate / Free の対象にせず、データ壊れが無いかの検証に使う
    const size_t firstPageSize = 16 * 1024 * 1024UL;

    g_Processor.ProcessAllocateAddressRegion(0, regionSize);
    g_Processor.ProcessAllocateMemoryPages(0, 0, firstPageSize);
    g_Processor.ProcessFillMemory(0, 0, firstPageSize);

    // 小さいページをたくさん Allocate / Free
    for (int i = 0; i < repeatCount; i++)
    {
        CommandList commands;

        const size_t loopCount = (regionSize - firstPageSize) / pageSize;

        for (size_t j = 0; j < loopCount; j++)
        {
            commands.AddAllocatePagesCommand(0, pageSize * j + firstPageSize, pageSize);
        }

        for (size_t j = 0; j < loopCount; j++)
        {
            commands.AddFreePagesCommand(0, pageSize * j + firstPageSize, pageSize);
        }

        g_Processor.Process(commands.begin(), commands.end());
        auto processingTime = g_Processor.GetProcessingTime();

        NN_LOG("Loop %d: %ld msec (%d commands).\n", i,
            processingTime.GetMilliSeconds(),
            commands.size());
    }

    // 最初のページを使ってメモリ化けの確認
    g_Processor.ProcessVerifyMemory(0, 0, firstPageSize);
    g_Processor.ProcessFreeMemoryPages(0, 0, firstPageSize);
    g_Processor.ProcessFreeAddressRegion(0);
    EXPECT_TRUE(g_Processor.IsVerificationPassed());

    ConfirmResourceUsage(16 * 1024 * 1024);
}

// 多くのリージョンで、Allocate や Free を繰り返す
TEST(VirtualAddressMemoryStressBasic, ManyRegion)
{
    nn::os::MemoryInfo memInfo;
    nn::os::QueryMemoryInfo(&memInfo);

    const size_t virtualAddressMemorySize = MaxAllocatedMemory;
    ASSERT_LE(virtualAddressMemorySize, memInfo.totalAvailableMemorySize - memInfo.totalUsedMemorySize);

    const size_t firstPageSize = 4 * 1024;
    const size_t secondPageSize = 4 * 1024;
    const size_t regionSize = firstPageSize + secondPageSize;
    const size_t regionCount = virtualAddressMemorySize / regionSize;

    const size_t repeatCount = 10;

    // 多くのリージョンを作成
    for (int regionId = 0; regionId < regionCount; regionId++)
    {
        g_Processor.ProcessAllocateAddressRegion(regionId, regionSize);
    }

    // 各リージョンでページを確保・解放
    for (int i = 0; i < repeatCount; i++)
    {
        CommandList commands;

        for (int regionId = 0; regionId < regionCount; regionId++)
        {
            commands.AddAllocatePagesCommand(regionId, 0, firstPageSize);
            commands.AddFillCommand(regionId, 0, firstPageSize);
            commands.AddAllocatePagesCommand(regionId, firstPageSize, secondPageSize);
        }

        for (int regionId = 0; regionId < regionCount; regionId++)
        {
            commands.AddFreePagesCommand(regionId,  firstPageSize, secondPageSize);
            commands.AddVerifyCommand(regionId,  0, firstPageSize);
            commands.AddFreePagesCommand(regionId,  0, firstPageSize);
        }

        g_Processor.Process(commands.begin(), commands.end());
        auto processingTime = g_Processor.GetProcessingTime();

        EXPECT_TRUE(g_Processor.IsVerificationPassed());

        NN_LOG("Loop %d: %ld msec (%d commands).\n", i,
            processingTime.GetMilliSeconds(),
            commands.size());
    }

    // リージョンを解放
    for (int regionId = 0; regionId < regionCount; regionId++)
    {
        g_Processor.ProcessFreeAddressRegion(regionId);
    }

    // TODO: リソース消費が 16MB を超える原因の調査
    ConfirmResourceUsage(64 * 1024 * 1024);
}

// 巨大なページを確保してから、部分的な Free -> Allocate を繰り返す
TEST(VirtualAddressMemoryStressBasic, PartialFree)
{
    nn::os::MemoryInfo memInfo;
    nn::os::QueryMemoryInfo(&memInfo);

    const size_t regionSize = MaxAllocatedMemory;
    ASSERT_LE(regionSize, memInfo.totalAvailableMemorySize - memInfo.totalUsedMemorySize);

    const size_t pageSizes[] = {
                  4 * 1024,
                  8 * 1024,
                 12 * 1024,
                 16 * 1024,
                 64 * 1024,
                256 * 1024,
           1 * 1024 * 1024,
           2 * 1024 * 1024,
           3 * 1024 * 1024,
           4 * 1024 * 1024,
          16 * 1024 * 1024,
          64 * 1024 * 1024,
         256 * 1024 * 1024,
    };
    const size_t offsetStepSize = 777 * 4 * 1024; // 中途半端な数

    // 巨大なページの確保
    g_Processor.ProcessAllocateAddressRegion(0, regionSize);
    g_Processor.ProcessAllocateMemoryPages(0, 0, regionSize);

    for (auto pageSize : pageSizes)
    {
        if (pageSize > regionSize)
        {
            continue;
        }

        CommandList commands;

        for (size_t offset = 0; offset <= regionSize - pageSize; offset += offsetStepSize)
        {
            commands.AddFreePagesCommand(0, offset, pageSize);
            commands.AddAllocatePagesCommand(0, offset, pageSize);
        }

        g_Processor.Process(commands.begin(), commands.end());
        auto processingTime = g_Processor.GetProcessingTime();

        NN_LOG("Page size %zu: %ld msec (%d commands).\n", pageSize,
            processingTime.GetMilliSeconds(),
            commands.size());
    }

    // 巨大なページの解放
    g_Processor.ProcessFreeMemoryPages(0, 0, regionSize);
    g_Processor.ProcessFreeAddressRegion(0);

    // TODO: リソース消費が 16MB を超える原因の調査
    ConfirmResourceUsage(128 * 1024 * 1024);
}

}}} // namespace nnt::os::VirtualAddressMemory

extern "C" void nninitStartup()
{
    ::nn::init::InitializeAllocator(reinterpret_cast<void*>(g_Heap), HeapSize);
}
