﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <map>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_VirtualAddressMemory.h>
#include <nn/util/util_TinyMt.h>

#include <nnt.h>

#define NNT_OS_DETAIL_LOG(...)
//#define NNT_OS_DETAIL_LOG NN_LOG

namespace nnt { namespace os { namespace VirtualAddressMemory {

//---------------------------------------------------------------------------
// VirtualAddressMemory 機能のテストに共通のコード
//---------------------------------------------------------------------------

// CommandProcessor が解釈するコマンドの種類
enum CommandType
{
    CommandType_AllocateAddressRegion,
    CommandType_FreeAddressRegion,
    CommandType_AllocateMemoryPages,
    CommandType_FreeMemoryPages,
    CommandType_FillMemory,
    CommandType_VerifyMemory,
    CommandType_Sync,
    CommandType_Sleep,
};

// CommandProcessor が解釈するコマンド
struct Command
{
    CommandType type;
    int regionId;
    size_t offset;
    size_t size;
    nn::os::BarrierType* pBarrier;
    nn::TimeSpan timeSpan;
};

// 仮想アドレスメモリ管理用のリソース消費量を表示し、maxResourceUsage 以下であることを確認します。
void ConfirmResourceUsage(size_t maxResourceUsage);

// AddressRegion の Allocate / Free を繰り返して、パフォーマンスを表示します。
void AddressRegionPerformanceTest(size_t size, size_t loopCount);

// AddressRegion から MemoryPage の Allocate / Free を繰り返して、パフォーマンスを表示します。
void MemoryPagePerformanceTest(size_t regionSize, size_t pageSize);

// コマンドを格納するリスト
class CommandList : public std::vector<Command>
{
public:
    void AddAllocateRegionCommand(int regionId, size_t size);

    void AddFreeRegionCommand(int regionId);

    void AddAllocatePagesCommand(int regionId, size_t offset, size_t size);

    void AddFreePagesCommand(int regionId, size_t offset, size_t size);

    void AddFillCommand(int regionId, size_t offset, size_t size);

    void AddVerifyCommand(int regionId, size_t offset, size_t size);

    void AddSyncCommand(nn::os::BarrierType* pBarrier);

    void AddSleepCommand(nn::TimeSpan timeSpan);
};

// 仮想アドレス空間上のアドレス領域の確保・解放などのコマンドを処理する、テスト専用の処理系
class CommandProcessor
{
public:
    CommandProcessor()
    {
        m_Mt.Initialize(0);
        ClearCommand();
    }

    void ClearCommand()
    {
        m_IsVerificationPassed = true;
    }

    nn::TimeSpan GetProcessingTime()
    {
        return m_ProcessingTime;
    }

    bool IsVerificationPassed()
    {
        return m_IsVerificationPassed;
    }

    void LoadRegionInfo(CommandProcessor& other);

    void Process(Command& command);

    void ProcessAllocateAddressRegion(int regionId, size_t size);

    void ProcessFreeAddressRegion(int regionId);

    void ProcessAllocateMemoryPages(int regionId, size_t offset, size_t size);

    void ProcessFreeMemoryPages(int regionId, size_t offsett, size_t size);

    void ProcessFillMemory(int regionId, size_t offset, size_t size);

    void ProcessVerifyMemory(int regionId, size_t offset, size_t size);

    void ProcessSync(nn::os::BarrierType* pBarrier);

    void ProcessSleep(nn::TimeSpan timeSpan);

    template <class InputIterator>
    void Process(InputIterator first, InputIterator last)
    {
        auto start = nn::os::GetSystemTick();
        for (auto iter = first; iter != last; iter++)
        {
            Process(*iter);
        }
        auto end = nn::os::GetSystemTick();

        m_ProcessingTime = nn::os::ConvertToTimeSpan(end - start);
    }
private:

    // 指定された領域をランダム値で埋める
    // このとき、値の総和が 0 になるようにする
    void Fill(uintptr_t address, size_t size);

    // 指定された領域が Fill() を実行した時点から書き換わっていなければ true を返す
    // すなわち、値の総和が 0 であることを確認する
    bool Verify(uintptr_t address, size_t size);

private:
    // Region ID -> Address のマップ
    std::map<size_t, uintptr_t> m_RegionAddressMap;

    nn::util::TinyMt m_Mt;

    bool m_IsVerificationPassed;

    nn::TimeSpan m_ProcessingTime;
};

}}} // namespace nnt::os::VirtualAddressMemory
