﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/os.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/init.h>
#include "../Common/test_Helper.h"
#include "../Common/test_MemoryAccess.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

#include <random>

namespace nnt { namespace os { namespace TransferMemory {

namespace {
    size_t RoundUp(size_t size, size_t align)
    {
        return (size + align - 1) & ~(align - 1);
    }
}

//---------------------------------------------------------------------------
//  Transfer memory 機能のテスト
//---------------------------------------------------------------------------

const size_t trmemSize = 1 * 1024 * 1024;

TEST(TransferMemoryTest, BasicUsage)
{
    // 個別のテスト毎に、テストに必要な量のヒープサイズを指定しておく
    auto result = nn::os::SetMemoryHeapSize( 16 * 1024 * 1024 );
    EXPECT_TRUE( result.IsSuccess() );

    // Transfer memory にする領域をヒープから取得
    uintptr_t   address;
    NNT_OS_LOG("aligned size = %d\n", RoundUp(trmemSize, nn::os::MemoryBlockUnitSize));
    result = nn::os::AllocateMemoryBlock( &address, RoundUp(trmemSize, nn::os::MemoryBlockUnitSize) );
    EXPECT_TRUE( result.IsSuccess() );

    //
    // Client を想定したテスト
    //

    // Transfer memory オブジェクトＡの生成
    NNT_OS_LOG(NN_TEXT("Transfer memory Ａの生成\n"));
    nn::os::TransferMemoryType    trmem1;
    result = nn::os::CreateTransferMemory(&trmem1, reinterpret_cast<void*>(address), trmemSize, nn::os::MemoryPermission_None);
    ASSERT_TRUE(result.IsSuccess());

    NNT_OS_LOG(NN_TEXT("Transfer memory Ａ: address=0x%08x, size=%d\n"), address, trmemSize);

    // TORIAEZU:
    // Transfer memory にした領域にアクセスできない確認は別途 Death test でする
    // …が、現状実機ては Death test できないので保留

    // Transfer memory オブジェクトからハンドルを切り離す
    auto handle = nn::os::DetachTransferMemory(&trmem1);
    NNT_OS_LOG("handle=0x%08x\n", handle);
    ASSERT_NE(handle, nn::os::NativeHandle(0U));

    //
    // Server を想定したテスト（ハンドルを IPC 経由で受け取ったと仮定）
    //

    // Transfer memory オブジェクトＢの初期化
    NNT_OS_LOG(NN_TEXT("Transfer memory ＡのハンドルからTransfer memory Ｂを初期化\n"));
    nn::os::TransferMemoryType    trmem2;
    nn::os::AttachTransferMemory(&trmem2, trmemSize, handle, true);

    // Transfer memory Ｂをマップ
    NNT_OS_LOG(NN_TEXT("Transfer memory Ｂをマップ： \n"));
    void* adrs2;
    result = nn::os::MapTransferMemory(&adrs2, &trmem2, nn::os::MemoryPermission_None);
    ASSERT_TRUE(result.IsSuccess());
    NNT_OS_LOG("address2= 0x%p\n", adrs2);
    ASSERT_NE(adrs2, nullptr);
    ASSERT_NE(address, reinterpret_cast<uintptr_t>(adrs2));
    ASSERT_EQ(0U, reinterpret_cast<uintptr_t>(adrs2) % nn::os::MemoryPageSize);

    // マップした領域に書き込めることを確認
    NNT_OS_LOG(NN_TEXT("マップした領域への書き込みテスト\n"));
    uint8_t* ptr = reinterpret_cast<uint8_t*>(adrs2);
    for (size_t i=0; i<trmemSize; i++)
    {
        ptr[i] = i % 64;
    }
    for (size_t i=0; i<trmemSize; i++)
    {
        EXPECT_EQ(i % 64, ptr[i]);
    }

    // Transfer memory をアンマップ
    NNT_OS_LOG(NN_TEXT("Transfer memory をアンマップ\n"));
    nn::os::UnmapTransferMemory(&trmem2);

    // アンマップは何回でも実行できる（計３回）
    nn::os::UnmapTransferMemory(&trmem2);
    nn::os::UnmapTransferMemory(&trmem2);

    // 再マップ
    NNT_OS_LOG(NN_TEXT("Transfer memory Ｂを再マップ： \n"));
    result = nn::os::MapTransferMemory(&adrs2, &trmem2, nn::os::MemoryPermission_None);
    ASSERT_TRUE(result.IsSuccess());
    NNT_OS_LOG("address2= 0x%p\n", adrs2);
    ASSERT_NE(adrs2, nullptr);
    ASSERT_NE(address, reinterpret_cast<uintptr_t>(adrs2));
    ASSERT_EQ(0U, reinterpret_cast<uintptr_t>(adrs2) % nn::os::MemoryPageSize);

    // 再マップした領域に書き込めることを確認
    NNT_OS_LOG(NN_TEXT("再マップした領域への書き込みテスト\n"));
    ptr = reinterpret_cast<uint8_t*>(adrs2);
    for (size_t i=0; i<trmemSize; i++)
    {
        ptr[i] = i % 64;
    }
    for (size_t i=0; i<trmemSize; i++)
    {
        EXPECT_EQ(i % 64, ptr[i]);
    }

    // 再アンマップ
    NNT_OS_LOG(NN_TEXT("Transfer memory を再アンマップ\n"));
    nn::os::UnmapTransferMemory(&trmem2);

    // Transfer memory を破棄
    NNT_OS_LOG(NN_TEXT("Transfer memory を破棄\n"));
    nn::os::DestroyTransferMemory(&trmem2);

    // メモリを返却
    nn::os::FreeMemoryBlock(address, nn::os::MemoryBlockUnitSize);

    // ヒープを戻しておく
    nn::os::SetMemoryHeapSize( 0 );
}

TEST(TransferMemoryTest, CreateFailure)
{
    // エラーが返るのはハンドル数上限、Transfer memory 数上限
}

// Win32 では空間が足りずに失敗する場合がある
#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) || defined(NN_BUILD_CONFIG_CPU_X64)
TEST(TransferMemoryTest, MapTwice)
{
    // 大きめのサイズ２つ
    size_t trmemSize1 = 512 * 1024 * 1024;  // 512MB
    size_t trmemSize2 = 192 * 1024 * 1024;  // 192MB

    // 個別のテスト毎に、テストに必要な量のヒープサイズを指定しておく
    auto result = nn::os::SetMemoryHeapSize( RoundUp(trmemSize1, nn::os::MemoryBlockUnitSize) + RoundUp(trmemSize2, nn::os::MemoryBlockUnitSize) );
    if (result.IsSuccess())
    {
        // ヒープが確保できる環境でのみテストする

        // Transfer memory にする領域をヒープから取得
        uintptr_t   address1;
        uintptr_t   address2;
        result = nn::os::AllocateMemoryBlock( &address1, RoundUp(trmemSize1, nn::os::MemoryBlockUnitSize) );
        ASSERT_TRUE( result.IsSuccess() );
        result = nn::os::AllocateMemoryBlock( &address2, RoundUp(trmemSize2, nn::os::MemoryBlockUnitSize) );
        ASSERT_TRUE( result.IsSuccess() );

        // Transfer memory オブジェクトＡの生成
        NNT_OS_LOG(NN_TEXT("Transfer memory Ａの生成\n"));
        void* adrs1;
        nn::os::TransferMemoryType    trmem1;
        result = nn::os::CreateTransferMemory(&trmem1, reinterpret_cast<void*>(address1), trmemSize1, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());

        // ギリギリ Map できるはず
        NNT_OS_LOG(NN_TEXT("Map できるはず\n"));
        result = nn::os::MapTransferMemory(&adrs1, &trmem1, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());

        NNT_OS_LOG("TRMem1: orig_address=0x%p mapped_address=0x%p\n", address1, adrs1);

        // Transfer memory オブジェクトＢの生成
        NNT_OS_LOG(NN_TEXT("Transfer memory Ｂの生成\n"));
        void* adrs2;
        nn::os::TransferMemoryType    trmem2;
        result = nn::os::CreateTransferMemory(&trmem2, reinterpret_cast<void*>(address2), trmemSize2, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());

        // ギリギリ Map できるはず
        NNT_OS_LOG(NN_TEXT("Map できるはず\n"));
        result = nn::os::MapTransferMemory(&adrs2, &trmem2, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());

        NNT_OS_LOG("TRMem2: orig_address=0x%p mapped_address=0x%p\n", address2, adrs2);

        // 後始末
        nn::os::UnmapTransferMemory(&trmem1);
        nn::os::UnmapTransferMemory(&trmem2);
        nn::os::DestroyTransferMemory(&trmem1);
        nn::os::DestroyTransferMemory(&trmem2);
        nn::os::FreeMemoryBlock(address1, RoundUp(trmemSize1, nn::os::MemoryBlockUnitSize));
        nn::os::FreeMemoryBlock(address2, RoundUp(trmemSize2, nn::os::MemoryBlockUnitSize));
        nn::os::SetMemoryHeapSize( 0 );
    }
    else
    {
        NNT_OS_LOG(NN_TEXT("1GB のヒープが確保できなかったためテストせず\n"));
    }
}
#endif

// Win32 では空間が足りずに失敗する場合がある
#if defined(NN_OS_CPU_ARM_AARCH64_ARMV8A) || defined(NN_BUILD_CONFIG_CPU_X64)
TEST(TransferMemoryTest, BoundaryValueTest)
{
    // 確保できるはずのサイズ
    size_t bigTrmemSize = 341 * 1024 * 1024;  // 256MB

    // 個別のテスト毎に、テストに必要な量のヒープサイズを指定しておく
    auto result = nn::os::SetMemoryHeapSize( RoundUp(bigTrmemSize, nn::os::MemoryBlockUnitSize) );
    if (result.IsSuccess())
    {
        // 1GB のヒープが確保できる環境でのみテストする

        // Transfer memory にする領域をヒープから取得
        uintptr_t   address;
        result = nn::os::AllocateMemoryBlock( &address, RoundUp(bigTrmemSize, nn::os::MemoryBlockUnitSize) );
        EXPECT_TRUE( result.IsSuccess() );

        // Transfer memory オブジェクトＡの生成
        NNT_OS_LOG(NN_TEXT("Transfer memory Ａの生成\n"));
        void* adrs1;
        nn::os::TransferMemoryType    trmem1;
        result = nn::os::CreateTransferMemory(&trmem1, reinterpret_cast<void*>(address), bigTrmemSize, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());

        // ギリギリ Map できるはず
        NNT_OS_LOG(NN_TEXT("Map できるはず\n"));
        result = nn::os::MapTransferMemory(&adrs1, &trmem1, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());

        // 後始末
        nn::os::UnmapTransferMemory(&trmem1);
        nn::os::DestroyTransferMemory(&trmem1);
        nn::os::FreeMemoryBlock(address, RoundUp(bigTrmemSize, nn::os::MemoryBlockUnitSize));
        nn::os::SetMemoryHeapSize( 0 );
    }
    else
    {
        NNT_OS_LOG(NN_TEXT("1GB のヒープが確保できなかったためテストせず\n"));
    }
}
#endif

// Win32 だと Map 時に適切なエラーを返せない
// LP64 時は仮想アドレス空間が 4GB になるため、空間が確保できてしまうためテストできない
#if defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)
TEST(TransferMemoryTest, MapFailureByOutOfAddressSpace)
{
    size_t bigTrmemSize = 1024 * 1024 * 1024;  // 1GB

    // 個別のテスト毎に、テストに必要な量のヒープサイズを指定しておく
    auto result = nn::os::SetMemoryHeapSize( RoundUp(bigTrmemSize, nn::os::MemoryBlockUnitSize) );
    if (result.IsSuccess())
    {
        // 1GB のヒープが確保できる環境でのみテストする

        // Transfer memory にする領域をヒープから取得
        uintptr_t   address;
        result = nn::os::AllocateMemoryBlock( &address, RoundUp(bigTrmemSize, nn::os::MemoryBlockUnitSize) );
        EXPECT_TRUE( result.IsSuccess() );

        // Transfer memory オブジェクトＡの生成
        NNT_OS_LOG(NN_TEXT("Transfer memory Ａの生成\n"));
        void* adrs1;
        nn::os::TransferMemoryType    trmem1;
        result = nn::os::CreateTransferMemory(&trmem1, reinterpret_cast<void*>(address), bigTrmemSize, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());

        // Map 時、アドレス空間が足りない場合
        NNT_OS_LOG(NN_TEXT("Map 時、アドレス空間が足りない場合\n"));
        result = nn::os::MapTransferMemory(&adrs1, &trmem1, nn::os::MemoryPermission_None);
        EXPECT_TRUE(nn::os::ResultOutOfAddressSpace::Includes(result));

        // 後始末
        nn::os::DestroyTransferMemory(&trmem1);
        nn::os::FreeMemoryBlock(address, RoundUp(bigTrmemSize, nn::os::MemoryBlockUnitSize));
        nn::os::SetMemoryHeapSize( 0 );
    }
    else
    {
        NNT_OS_LOG(NN_TEXT("1GB のヒープが確保できなかったためテストせず\n"));
    }
}

TEST(TransferMemoryTest, MapFailure)
{
    // 個別のテスト毎に、テストに必要な量のヒープサイズを指定しておく
    auto result = nn::os::SetMemoryHeapSize( 16 * 1024 * 1024 );
    EXPECT_TRUE( result.IsSuccess() );

    // Transfer memory にする領域をヒープから取得
    uintptr_t   address;
    result = nn::os::AllocateMemoryBlock( &address, RoundUp(trmemSize, nn::os::MemoryBlockUnitSize) );
    EXPECT_TRUE( result.IsSuccess() );

    //
    // Client を想定したテスト
    //

    // Transfer memory オブジェクトＡの生成
    NNT_OS_LOG(NN_TEXT("Transfer memory Ａの生成\n"));
    nn::os::TransferMemoryType    trmem1;
    result = nn::os::CreateTransferMemory(&trmem1, reinterpret_cast<void*>(address), trmemSize, nn::os::MemoryPermission_None);
    ASSERT_TRUE(result.IsSuccess());

    // Transfer memory オブジェクトからハンドルを切り離す
    auto handle = nn::os::DetachTransferMemory(&trmem1);
    NNT_OS_LOG(" handle=0x%08x\n", handle);
    ASSERT_NE(handle, nn::os::NativeHandle(0U));

    //
    // Server を想定したテスト（ハンドルを IPC 経由で受け取ったと仮定）
    //

    nn::os::TransferMemoryType    trmem2;
    nn::os::TransferMemoryType    trmem3;
    void* adrs2;
    void* adrs3;

    // 誤ったハンドルをアタッチした場合
    NNT_OS_LOG(NN_TEXT("誤ったハンドルをアタッチした場合\n"));
    auto dummyHandle = nn::os::NativeHandle(0x12345678U);
    nn::os::AttachTransferMemory(&trmem2, trmemSize, dummyHandle, true);
    result = nn::os::MapTransferMemory(&adrs2, &trmem2, nn::os::MemoryPermission_None);
    EXPECT_TRUE(nn::os::ResultInvalidHandle::Includes(result));

    // サイズが一致しない場合
    NNT_OS_LOG(NN_TEXT("サイズが一致しない場合\n"));
    nn::os::AttachTransferMemory(&trmem2, trmemSize * 2, handle, true);
    result = nn::os::MapTransferMemory(&adrs2, &trmem2, nn::os::MemoryPermission_None);
    EXPECT_TRUE(nn::os::ResultInvalidTransferMemorySize::Includes(result));

    // owner permission が一致しない場合
    NNT_OS_LOG(NN_TEXT("owner permission が一致しない場合\n"));
    nn::os::AttachTransferMemory(&trmem2, trmemSize, handle, true);
    result = nn::os::MapTransferMemory(&adrs2, &trmem2, nn::os::MemoryPermission_ReadWrite);
    EXPECT_TRUE(nn::os::ResultInvalidTransferMemoryState::Includes(result));

    // 一旦マップ成功させる
    nn::os::AttachTransferMemory(&trmem2, trmemSize, handle, true);
    result = nn::os::MapTransferMemory(&adrs2, &trmem2, nn::os::MemoryPermission_None);

    // 既にマップ済みの場合
    NNT_OS_LOG(NN_TEXT("既にマップ済みの場合\n"));
    nn::os::AttachTransferMemory(&trmem3, trmemSize, handle, true);
    result = nn::os::MapTransferMemory(&adrs3, &trmem3, nn::os::MemoryPermission_None);
    EXPECT_TRUE(nn::os::ResultInvalidTransferMemoryState::Includes(result));

    // アドレス空間が足りない場合
    // …は、別関数でテストする

    // Transfer memory をアンマップ
    NNT_OS_LOG(NN_TEXT("Transfer memory をアンマップ\n"));
    nn::os::UnmapTransferMemory(&trmem2);

    // Transfer memory を破棄
    NNT_OS_LOG(NN_TEXT("Transfer memory を破棄\n"));
    nn::os::DestroyTransferMemory(&trmem2);

    // メモリを返却
    nn::os::FreeMemoryBlock(address, nn::os::MemoryBlockUnitSize);

    // ヒープを戻しておく
    nn::os::SetMemoryHeapSize( 0 );
}
#endif

TEST(TransferMemoryClassTest, BasicUsage)
{
    // 個別のテスト毎に、テストに必要な量のヒープサイズを指定しておく
    auto result = nn::os::SetMemoryHeapSize( 16 * 1024 * 1024 );
    EXPECT_TRUE( result.IsSuccess() );

    // Transfer memory にする領域をヒープから取得
    uintptr_t   address;
    result = nn::os::AllocateMemoryBlock( &address, RoundUp(trmemSize, nn::os::MemoryBlockUnitSize) );
    EXPECT_TRUE( result.IsSuccess() );

    //
    // Client を想定したテスト
    //
    {
        // Transfer memory オブジェクトＡの生成
        NNT_OS_LOG(NN_TEXT("Transfer memory Ａの生成\n"));
        nn::os::TransferMemory        trmem1(reinterpret_cast<void*>(address), trmemSize, nn::os::MemoryPermission_None);

        NNT_OS_LOG(NN_TEXT("Transfer memory Ａ: address=0x%08x, size=%d\n"), address, trmemSize);

        // TORIAEZU:
        // Transfer memory にした領域にアクセスできない確認は別途 Death test でする
        // …が、現状実機ては Death test できないので保留

        // Transfer memory オブジェクトからハンドルを切り離す
        auto handle = trmem1.Detach();
        NNT_OS_LOG("handle=0x%08x\n", handle);
        ASSERT_NE(handle, nn::os::NativeHandle(0U));

        //
        // Server を想定したテスト（ハンドルを IPC 経由で受け取ったと仮定）
        //

        // Transfer memory オブジェクトＢの初期化
        NNT_OS_LOG(NN_TEXT("Transfer memory ＡのハンドルからTransfer memory Ｂを初期化\n"));
        nn::os::TransferMemory        trmem2;
        trmem2.Attach(trmemSize, handle, true);

        // Transfer memory Ｂをマップ
        NNT_OS_LOG(NN_TEXT("Transfer memory Ｂをマップ： \n"));
        void* adrs2;
        result = trmem2.Map(&adrs2, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());
        NNT_OS_LOG("address2= 0x%p\n", adrs2);
        ASSERT_NE(adrs2, nullptr);
        ASSERT_NE(address, reinterpret_cast<uintptr_t>(adrs2));
        ASSERT_EQ(0U, reinterpret_cast<uintptr_t>(adrs2) % nn::os::MemoryPageSize);

        // マップした領域に書き込めることを確認
        NNT_OS_LOG(NN_TEXT("マップした領域への書き込みテスト\n"));
        uint8_t* ptr = reinterpret_cast<uint8_t*>(adrs2);
        for (size_t i=0; i<trmemSize; i++)
        {
            ptr[i] = i % 64;
        }
        for (size_t i=0; i<trmemSize; i++)
        {
            EXPECT_EQ(i % 64, ptr[i]);
        }

        // Transfer memory をアンマップ
        NNT_OS_LOG(NN_TEXT("Transfer memory をアンマップ\n"));
        trmem2.Unmap();

        // 再マップ
        NNT_OS_LOG(NN_TEXT("Transfer memory Ｂを再マップ： \n"));
        result = trmem2.Map(&adrs2, nn::os::MemoryPermission_None);
        ASSERT_TRUE(result.IsSuccess());
        NNT_OS_LOG("address2= 0x%p\n", adrs2);
        ASSERT_NE(adrs2, nullptr);
        ASSERT_NE(address, reinterpret_cast<uintptr_t>(adrs2));
        ASSERT_EQ(0U, reinterpret_cast<uintptr_t>(adrs2) % nn::os::MemoryPageSize);

        // 再マップした領域に書き込めることを確認
        NNT_OS_LOG(NN_TEXT("再マップした領域への書き込みテスト\n"));
        ptr = reinterpret_cast<uint8_t*>(adrs2);
        for (size_t i=0; i<trmemSize; i++)
        {
            ptr[i] = i % 64;
        }
        for (size_t i=0; i<trmemSize; i++)
        {
            EXPECT_EQ(i % 64, ptr[i]);
        }

        // 再アンマップ
        NNT_OS_LOG(NN_TEXT("Transfer memory をデストラクタ時に再アンマップ\n"));
        // 直後に trmem2 のデストラクタが走るときに自動アンマップされる。
        // ここでの明示的な Unmap() は行わない。
        //trmem2.Unmap();
    }

    // メモリを返却
    nn::os::FreeMemoryBlock(address, nn::os::MemoryBlockUnitSize);

    // ヒープを戻しておく
    nn::os::SetMemoryHeapSize( 0 );
}


//---------------------------------------------------------------------------

}}} // namespace nnt::os::TransferMemory

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of TransferMemory APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of TransferMemory APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
