﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  時間管理機能のテスト
//---------------------------------------------------------------------------

// windows.h で min/max が定義されるのを抑制する
#ifndef NOMINMAX
#define NOMINMAX
#endif

#include "../Common/test_Pragma.h"

#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/nn_SdkText.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include <nn/os/os_Tick.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace tickAndTimeSpan {

//---------------------------------------------------------------------------
// 静的オブジェクトのコンストラクタからも各 API が正しい値を返すか

class GlobalTestClass
{
public:
    GlobalTestClass()   NN_NOEXCEPT;

    int64_t GetTick() const NN_NOEXCEPT
    {
        return m_Tick;
    }

    int64_t GetTickFrequency() const NN_NOEXCEPT
    {
        return m_TickFrequency;
    }

    int64_t GetTickFrequencyConverted() const NN_NOEXCEPT
    {
        return m_TickFrequencyConverted;
    }

private:
    int64_t m_Tick;
    int64_t m_TickFrequency;
    int64_t m_TickFrequencyConverted;
};

// コンストラクタ
GlobalTestClass::GlobalTestClass()
NN_NOEXCEPT
{
    m_Tick          = nn::os::GetSystemTick().GetInt64Value();
    m_TickFrequency = nn::os::GetSystemTickFrequency();

    m_TickFrequencyConverted = nn::os::ConvertToTick( nn::os::Tick( nn::os::GetSystemTickFrequency() ).ToTimeSpan() ).GetInt64Value();
}

GlobalTestClass g_GlobalTestClass;

// テストコードはこちら
TEST(TickApiInStaticInitializer, test_TickApiInStaticInitializer)
{
    NNT_OS_LOG(NN_TEXT("静的オブジェクトのコンストラクタからも Tick の API が正しい値を返すか\n"));

    int64_t tick  = g_GlobalTestClass.GetTick();
    NNT_OS_LOG(" Tick          = %lld", tick);
    CheckBool( tick > 0 );

    int64_t tick2 = g_GlobalTestClass.GetTickFrequency();
    NNT_OS_LOG(" TickFrequency = %lld", tick2);
    CheckBool( tick2 > 0 );

    NNT_OS_LOG(NN_TEXT("一度 TimeSpan に変換した TickFrequency 値が元の変換前の値に等しいか？\n"));
    int64_t tick3 = g_GlobalTestClass.GetTickFrequencyConverted();
    NNT_OS_LOG(" TickFrequencyConverted = %lld", tick2);
    CheckBool( tick2 == tick3 );
}

//---------------------------------------------------------------------------
// ConvertToTimeSpan() および ConvertToTick の精度の検証

static void
TestConvertToTick(int64_t tick, nn::TimeSpan span)
{
    int64_t ts = span.GetNanoSeconds();

    if (ts == 0)
    {
        return;
    }

    auto    one   = nn::TimeSpan::FromNanoSeconds( 1 );
    int64_t tick1 = nn::os::ConvertToTick( span - one ).GetInt64Value();
    int64_t tick2 = nn::os::ConvertToTick( span       ).GetInt64Value();
    int64_t tick3 = nn::os::ConvertToTick( span + one ).GetInt64Value();

    NNT_OS_LOG("  TimeSpan= %20lld -> Tick= %20lld", ts - 1, tick1);
    CheckBool( tick2 >= tick1 );
    NNT_OS_LOG("  TimeSpan= %20lld -> Tick= %20lld", ts,   tick2);
    CheckBool( tick  == tick2 );
    NNT_OS_LOG("  TimeSpan= %20lld -> Tick= %20lld", ts + 1, tick3);
    CheckBool( tick3 >= tick2 );

    NNT_OS_LOG("\n");
}

static void
TestConvertToTimeSpan(int64_t tick)
{
    auto ts1 = nn::os::ConvertToTimeSpan( nn::os::Tick(tick - 1) );
    auto ts2 = nn::os::ConvertToTimeSpan( nn::os::Tick(tick)     );
    auto ts3 = nn::os::ConvertToTimeSpan( nn::os::Tick(tick + 1) );

    NNT_OS_LOG("Tick= %20lld -> TimeSpan= %20lld\n", tick - 1, ts1.GetNanoSeconds());
    NNT_OS_LOG("Tick= %20lld -> TimeSpan= %20lld", tick,   ts2.GetNanoSeconds());
    CheckBool( ts2 >= ts1 );
    NNT_OS_LOG("Tick= %20lld -> TimeSpan= %20lld", tick + 1, ts3.GetNanoSeconds());
    CheckBool( ts3 >= ts2 );
    NNT_OS_LOG("\n");

    TestConvertToTick( tick - 1, ts1 );
    TestConvertToTick( tick,   ts2 );
    TestConvertToTick( tick + 1, ts3 );
}

TEST(ConvertToTimeSpan, test_ConvertToTimeSpan)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    int64_t tick     = nn::os::GetSystemTick().GetInt64Value();
    int64_t tickFreq = nn::os::GetSystemTickFrequency();
    NNT_OS_LOG("TickFreq = %20lld\n\n", tickFreq);

    // 境界条件も含めて変換精度を検査
    TestConvertToTimeSpan( 1 );
    TestConvertToTimeSpan( tickFreq / 2 );
    TestConvertToTimeSpan( tickFreq     );
    TestConvertToTimeSpan( tickFreq * 2 );
    TestConvertToTimeSpan( tick );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

TEST(ConvertToTimeSpan, test_ConvertToTimeSpanBoundaryValue)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("tick = {min, -292年, -1, 0, 1, 292年, max} のとき nn::os::ConvertToTimeSpan( tick ) の大小関係が保存されることを確認する\n"));

    // オーバーフローするギリギリの値と、それが少なくとも 292 年よりも大きな値であることを確認する
    int64_t tickMin     = nn::os::ConvertToTick( nn::TimeSpan::FromNanoSeconds( std::numeric_limits<int64_t>::min() ) ).GetInt64Value() + 1;
    int64_t tickMax     = nn::os::ConvertToTick( nn::TimeSpan::FromNanoSeconds( std::numeric_limits<int64_t>::max() ) ).GetInt64Value() - 1;
    int64_t years292    = nn::os::ConvertToTick( nn::TimeSpan::FromDays( 365LL * 292LL ) ).GetInt64Value();

    int64_t tsMin       = nn::os::ConvertToTimeSpan( nn::os::Tick(  tickMin ) ).GetNanoSeconds();
    int64_t tsMinus292y = nn::os::ConvertToTimeSpan( nn::os::Tick( -years292 ) ).GetNanoSeconds();
    int64_t tsMinusOne  = nn::os::ConvertToTimeSpan( nn::os::Tick( -1 ) ).GetNanoSeconds();
    int64_t tsZero      = nn::os::ConvertToTimeSpan( nn::os::Tick(  0 ) ).GetNanoSeconds();
    int64_t tsPlusOne   = nn::os::ConvertToTimeSpan( nn::os::Tick(  1 ) ).GetNanoSeconds();
    int64_t tsPlus292y  = nn::os::ConvertToTimeSpan( nn::os::Tick(  years292 ) ).GetNanoSeconds();
    int64_t tsMax       = nn::os::ConvertToTimeSpan( nn::os::Tick(  tickMax ) ).GetNanoSeconds();

    CheckBool( tsMin       < tsMinus292y );
    CheckBool( tsMinus292y < tsMinusOne );
    CheckBool( tsMinusOne  < tsZero );
    CheckBool( tsZero      < tsPlusOne );
    CheckBool( tsPlusOne   < tsPlus292y );
    CheckBool( tsPlus292y  < tsMax );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

TEST(ConvertToTick, test_ConvertToTickBoundaryValue)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("timeSpan = {min, -292年, -1, 0, 1, 292年, max} のとき nn::os::ConvertToTick( timeSpan ) の大小関係が保存されることを確認する\n"));

    // TimeSpan の最小・最大値と、それが少なくとも 292 年よりも大きな値であることを確認する
    int64_t tickMin       = nn::os::ConvertToTick( nn::TimeSpan::FromNanoSeconds(  std::numeric_limits<int64_t>::min() ) ).GetInt64Value();
    int64_t tickMinus292y = nn::os::ConvertToTick( nn::TimeSpan::FromDays( -365LL * 292LL ) ).GetInt64Value();
    int64_t tickMinusOne  = nn::os::ConvertToTick( nn::TimeSpan::FromNanoSeconds(  -1 ) ).GetInt64Value();
    int64_t tickZero      = nn::os::ConvertToTick( nn::TimeSpan::FromNanoSeconds(   0 ) ).GetInt64Value();
    int64_t tickPlusOne   = nn::os::ConvertToTick( nn::TimeSpan::FromNanoSeconds(   1 ) ).GetInt64Value();
    int64_t tickPlus292y  = nn::os::ConvertToTick( nn::TimeSpan::FromDays(  365LL * 292LL ) ).GetInt64Value();
    int64_t tickMax       = nn::os::ConvertToTick( nn::TimeSpan::FromNanoSeconds(  std::numeric_limits<int64_t>::max() ) ).GetInt64Value();

    CheckBool( tickMin       < tickMinus292y );
    CheckBool( tickMinus292y < tickMinusOne );
    CheckBool( tickMinusOne  < tickZero );
    CheckBool( tickZero      < tickPlusOne );
    CheckBool( tickPlusOne   < tickPlus292y );
    CheckBool( tickPlus292y  < tickMax );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

static void
TestLinearMappingOfConvertToTick(int64_t timeSpan)
{
    nn::TimeSpan plusTimeSpan  = nn::TimeSpan::FromNanoSeconds(  timeSpan );
    nn::TimeSpan minusTimeSpan = nn::TimeSpan::FromNanoSeconds( -timeSpan );
    int64_t      tick1         = - nn::os::ConvertToTick( plusTimeSpan ).GetInt64Value();
    int64_t      tick2         =   nn::os::ConvertToTick( minusTimeSpan ).GetInt64Value();

    NNT_OS_LOG("-nn::os::ConvertToTick(  %lldns ) = %lldtick\n", timeSpan, tick1);
    NNT_OS_LOG(" nn::os::ConvertToTick( -%lldns ) = %lldtick\n", timeSpan, tick2);

    CheckBool( tick1 == tick2 );
}

TEST(CheckLinearMapping, test_LinearMappingOfConvertToTick)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("-nn::os::ConvertToTick( timespan ) == nn::os::ConvertToTick( -timespan ) であることを確認する\n"));

    TestLinearMappingOfConvertToTick(0LL);
    TestLinearMappingOfConvertToTick(1LL);
    TestLinearMappingOfConvertToTick(1000LL);
    TestLinearMappingOfConvertToTick(1000LL * 1000LL);
    TestLinearMappingOfConvertToTick(1000LL * 1000LL * 1000LL);
    TestLinearMappingOfConvertToTick(1000LL * 1000LL * 1000LL * 60);
    TestLinearMappingOfConvertToTick(1000LL * 1000LL * 1000LL * 60 * 60);
    TestLinearMappingOfConvertToTick(1000LL * 1000LL * 1000LL * 60 * 60 * 24);

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

static void
TestLinearMappingOfConvertToTimeSpan(int64_t tick)
{
    int64_t ts1  = - nn::os::ConvertToTimeSpan( nn::os::Tick(  tick ) ).GetNanoSeconds();
    int64_t ts2  =   nn::os::ConvertToTimeSpan( nn::os::Tick( -tick ) ).GetNanoSeconds();

    NNT_OS_LOG("-nn::os::ConvertToTimeSpan(  %lldtick ) = %lldns\n", tick, ts1);
    NNT_OS_LOG(" nn::os::ConvertToTimeSpan( -%lldtick ) = %lldns\n", tick, ts2);

    CheckBool( ts1 == ts2 );
}

TEST(CheckLinearMapping, test_LinearMappingOfConvertToTimeSpan)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    NNT_OS_LOG(NN_TEXT("-nn::os::ConvertToTimeSpan( tick ) == nn::os::ConvertToTimeSpan( -tick ) であることを確認する\n"));

    TestLinearMappingOfConvertToTimeSpan(0LL);
    TestLinearMappingOfConvertToTimeSpan(1LL);
    TestLinearMappingOfConvertToTimeSpan(1000LL);
    TestLinearMappingOfConvertToTimeSpan(1000LL * 1000LL);
    TestLinearMappingOfConvertToTimeSpan(1000LL * 1000LL * 1000LL);
    TestLinearMappingOfConvertToTimeSpan(1000LL * 1000LL * 1000LL * 60);
    TestLinearMappingOfConvertToTimeSpan(1000LL * 1000LL * 1000LL * 60 * 60);
    TestLinearMappingOfConvertToTimeSpan(1000LL * 1000LL * 1000LL * 60 * 60 * 24);

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

}}} // namespace nnt::os::tickAndTimeSpan

//---------------------------------------------------------------------------
// TimeSpan クラスメソッドの精度の検証
//  チューニング実装されている以下のメソッドについて境界値検証を行なう
//  - GetDays()
//  - GetHours()
//  - GetMinutes()
//  - GetSeconds()
//  - GetMilliSeconds()
//  - GetMicroSeconds()
//  - GetNanoSeconds()

static void
CompareGetValues(const char *str, int64_t expect, int64_t value)
{
    NN_UNUSED(str);
    NNT_OS_LOG("%7s: expect=%20lld Get()=%20lld", str, expect, value);
    CheckBool( expect == value );
}

static void
TestTimeSpanGetMethod(int64_t timeValue)
{
    auto timeSpan = nn::TimeSpan::FromNanoSeconds( timeValue );

    int64_t nsecs   = timeSpan.GetNanoSeconds();
    int64_t usecs   = nsecs / 1000;
    int64_t msecs   = nsecs / 1000 / 1000;
    int64_t seconds = nsecs / 1000 / 1000 / 1000;
    int64_t minutes = nsecs / 1000 / 1000 / 1000 / 60;
    int64_t hours   = nsecs / 1000 / 1000 / 1000 / 60 / 60;
    int64_t days    = nsecs / 1000 / 1000 / 1000 / 60 / 60 / 24;

    CompareGetValues("  nsecs", nsecs,   timeSpan.GetNanoSeconds()  );
    CompareGetValues("  usecs", usecs,   timeSpan.GetMicroSeconds() );
    CompareGetValues("  msecs", msecs,   timeSpan.GetMilliSeconds() );
    CompareGetValues("seconds", seconds, timeSpan.GetSeconds()      );
    CompareGetValues("minutes", minutes, timeSpan.GetMinutes()      );
    CompareGetValues("  hours", hours,   timeSpan.GetHours()        );
    CompareGetValues("   days", days,    timeSpan.GetDays()         );
    NNT_OS_LOG("\n");
}


TEST(TestTimeSpanGetMethod, test_TimeSpanGetMethod)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // ビット幅ごとの数値表現における境界条件の変換精度の検査
    for (int i=0; i<64; i++)
    {
        TestTimeSpanGetMethod( (0x1LL << i) - 1 );
        TestTimeSpanGetMethod(  0x1LL << i );
    }

    // 時間単位における境界条件の変換精度の検査
    TestTimeSpanGetMethod( 1000LL - 1 );
    TestTimeSpanGetMethod( 1000LL     );
    TestTimeSpanGetMethod( 1000LL + 1 );

    TestTimeSpanGetMethod( 1000LL * 1000LL - 1 );
    TestTimeSpanGetMethod( 1000LL * 1000LL     );
    TestTimeSpanGetMethod( 1000LL * 1000LL + 1 );

    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL - 1 );
    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL     );
    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL + 1 );

    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60 - 1 );
    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60     );
    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60 + 1 );

    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60 * 60 - 1 );
    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60 * 60     );
    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60 * 60 + 1 );

    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60 * 60 * 24 - 1 );
    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60 * 60 * 24     );
    TestTimeSpanGetMethod( 1000LL * 1000LL * 1000LL * 60 * 60 * 24 + 1 );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_CHRONO)
//---------------------------------------------------------------------------
// TimeSpan クラスのコンストラクタに std::chrono::duration を渡す検証
//  typedef された duration でインスタンスを生成し、その生の値を比較する

template<typename _Rep, typename _Period>
static void CompareRawCount(const char *str, const std::chrono::duration<_Rep, _Period>& duration, int64_t value)
{
    NN_UNUSED(str);
    int64_t expect = duration.count();

    NNT_OS_LOG("%7s: expect=%20lld value()=%20lld", str, expect, value);
    CheckBool( expect == value );
}

static void
TestTimeSpanFromChrono(int64_t timeValue)
{
    int64_t nsecs   = timeValue;
    int64_t usecs   = timeValue / 1000;
    int64_t msecs   = timeValue / 1000 / 1000;
    int64_t seconds = timeValue / 1000 / 1000 / 1000;
    int64_t minutes = timeValue / 1000 / 1000 / 1000 / 60;
    int64_t hours   = timeValue / 1000 / 1000 / 1000 / 60 / 60;

    std::chrono::nanoseconds  chrono_nsecs   = std::chrono::nanoseconds(nsecs);
    std::chrono::microseconds chrono_usecs   = std::chrono::microseconds(usecs);
    std::chrono::milliseconds chrono_msecs   = std::chrono::milliseconds(msecs);
    std::chrono::seconds      chrono_seconds = std::chrono::seconds(seconds);
    std::chrono::minutes      chrono_minutes = std::chrono::minutes(minutes);
    std::chrono::hours        chrono_hours   = std::chrono::hours(hours);

    // implicit なコンストラクタを呼ぶ
    nn::TimeSpan timeSpan_nsecs     = chrono_nsecs;
    nn::TimeSpan timeSpan_usecs     = chrono_usecs;
    nn::TimeSpan timeSpan_msecs     = chrono_msecs;
    nn::TimeSpan timeSpan_seconds   = chrono_seconds;
    nn::TimeSpan timeSpan_minutes   = chrono_minutes;
    nn::TimeSpan timeSpan_hours     = chrono_hours;

    CompareRawCount("  nsecs", chrono_nsecs, timeSpan_nsecs.GetNanoSeconds() );
    CompareRawCount("  usecs", chrono_usecs, timeSpan_usecs.GetMicroSeconds() );
    CompareRawCount("  msecs", chrono_msecs, timeSpan_msecs.GetMilliSeconds() );
    CompareRawCount("seconds", chrono_seconds, timeSpan_seconds.GetSeconds() );
    CompareRawCount("minutes", chrono_minutes, timeSpan_minutes.GetMinutes() );
    CompareRawCount("  hours", chrono_hours, timeSpan_hours.GetHours() );
    NNT_OS_LOG("\n");
}


TEST(TestTimeSpanConstructorFromChrono, test_TimeSpanConstructorFromChrono)
{
    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // ビット幅ごとの数値表現における境界条件の変換精度の検査
    for (int i=0; i<64; i++)
    {
        TestTimeSpanFromChrono( (0x1LL << i) - 1 );
        TestTimeSpanFromChrono(  0x1LL << i );
    }

    // 時間単位における境界条件の変換精度の検査
    TestTimeSpanFromChrono( 1000LL - 1 );
    TestTimeSpanFromChrono( 1000LL     );
    TestTimeSpanFromChrono( 1000LL + 1 );

    TestTimeSpanFromChrono( 1000LL * 1000LL - 1 );
    TestTimeSpanFromChrono( 1000LL * 1000LL     );
    TestTimeSpanFromChrono( 1000LL * 1000LL + 1 );

    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL - 1 );
    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL     );
    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL + 1 );

    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60 - 1 );
    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60     );
    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60 + 1 );

    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60 * 60 - 1 );
    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60 * 60     );
    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60 * 60 + 1 );

    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60 * 60 * 24 - 1 );
    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60 * 60 * 24     );
    TestTimeSpanFromChrono( 1000LL * 1000LL * 1000LL * 60 * 60 * 24 + 1 );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}
#endif  /* NN_BUILD_CONFIG_COMPILER_SUPPORTS_STD_CHRONO */

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of TickAndTimeSpan APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of TickAndTimeSpan APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
