﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_FormatString.h>
#include <nn/os.h>
#include <nn/os/os_SharedMemory.h>
#include "../Common/test_Helper.h"
#include "../Common/test_MemoryAccess.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace sharedMemory {

//---------------------------------------------------------------------------
//  共有メモリ機能のテスト
//---------------------------------------------------------------------------

const size_t shmemSize = 1 * 1024 * 1024;

//---------------------------------------------------------------------------
// 新規に作成した共有メモリオブジェクトＡから共有メモリハンドルを取得し、
// もう１つの共有メモリオブジェクトＢを初期化する。
// それぞれの共有メモリを適当なアドレスにマップし、
// 空間Ａのメモリの内容を書き換えた後、空間Ｂのメモリと空間Ａのメモリを比較し、
// 同じ内容が見えているかを確認する。
//
void DumpSharedMemory(void* address1, void* address2)
{
    uint8_t* p1 = reinterpret_cast<uint8_t*>(address1);
    uint8_t* p2 = reinterpret_cast<uint8_t*>(address2);

    for (int i=0; i<16; ++i)
    {
        char buf[128];
        nn::util::SNPrintf(buf, sizeof(buf),
                "0x%08p: %02x %02x %02x %02x %02x %02x %02x %02x |"
                "0x%08p: %02x %02x %02x %02x %02x %02x %02x %02x\n",
                p1, p1[0], p1[1], p1[2], p1[3], p1[4], p1[5], p1[6], p1[7],
                p2, p2[0], p2[1], p2[2], p2[3], p2[4], p2[5], p2[6], p2[7]);

        NNT_OS_LOG(buf);
        p1 += 8;
        p2 += 8;
    }
}


//---------------------------------------------------------------------------
//  nn::os::MapSharedMemory( アドレス指定不可 ) 版のテスト
//
TEST(SharedMemoryTest, test_SharedMemoryTest)
{
    CLEAR_GOOGLE_TEST();

    {
        // 共有メモリオブジェクトＡの生成と初期化
        SEQ_SET(100);
        NNT_OS_LOG(NN_TEXT("共有メモリＡの生成"));
        nn::os::SharedMemoryType    shmem1;
        nn::Result result = nn::os::CreateSharedMemory(&shmem1, shmemSize,
                                        nn::os::MemoryPermission_ReadWrite,
                                        nn::os::MemoryPermission_ReadWrite);
        auto handle = nn::os::GetSharedMemoryHandle(&shmem1);
        NNT_OS_LOG(" handle=0x%08x", handle);
        CheckBool( result.IsSuccess() );

        // 共有メモリオブジェクトＢの初期化
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡのハンドルを使って共有メモリＢを初期化"));
        nn::os::SharedMemoryType    shmem2;
        nn::os::AttachSharedMemory(&shmem2, shmemSize, handle, false);
        CheckBool( true );

        // 共有メモリＡをマップ
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡをマップ： "));
        void* adrs1 = nn::os::MapSharedMemory(&shmem1, nn::os::MemoryPermission_ReadWrite);
        NNT_OS_LOG("address1= 0x%p\n", adrs1);
        if (adrs1 == NULL)
        {
            FAIL();
        }

        // 共有メモリＢをマップ
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＢをマップ： "));
        void* adrs2 = nn::os::MapSharedMemory(&shmem2, nn::os::MemoryPermission_ReadWrite);
        NNT_OS_LOG("address2= 0x%p\n", adrs2);
        if ((adrs2 == NULL) || (adrs1 == adrs2))
        {
            FAIL();
        }

        // 共有メモリＡのアドレスとサイズをチェック
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡのアドレスとサイズをチェック"));
        auto gotAddress1 = nn::os::GetSharedMemoryAddress(&shmem1);
        auto gotSize1    = nn::os::GetSharedMemorySize(&shmem1);
        CheckBool( (gotAddress1 == adrs1) && (gotSize1 == shmemSize) );

        // 共有メモリＢのアドレスとサイズをチェック
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＢのアドレスとサイズをチェック"));
        auto gotAddress2 = nn::os::GetSharedMemoryAddress(&shmem2);
        auto gotSize2    = nn::os::GetSharedMemorySize(&shmem2);
        CheckBool( (gotAddress2 == adrs2) && (gotSize2 == shmemSize) );

        // マップ直後の共有メモリの内容をダンプ
        DumpSharedMemory(adrs1, adrs2);

        // 共有メモリＡ の内容をビット反転して 共有メモリＢ にコピー
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡの内容をビット反転して共有メモリＢにコピー"));
        uint64_t* src = reinterpret_cast<uint64_t*>(adrs1);
        uint64_t* dst = reinterpret_cast<uint64_t*>(adrs2);

        for (size_t count = 0; count < shmemSize; count += sizeof(uint64_t))
        {
            *dst = ~(*src);
            ++dst;
            ++src;
        }
        CheckBool( true );

        // 共有メモリＡ の内容と 共有メモリＢ の内容が一致しているか確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡの内容と共有メモリＢの内容が一致しているか"));
        src = reinterpret_cast<uint64_t*>(adrs1);
        dst = reinterpret_cast<uint64_t*>(adrs2);
        for (size_t count = 0; count < shmemSize; count += sizeof(uint64_t))
        {
            if (*src != *dst)
            {
                NNT_OS_LOG("\n");
                NNT_OS_LOG(NN_TEXT("不一致： address1=0x%p  address2=0x%p\n"), src, dst);
                FAIL();
            }
            ++dst;
            ++src;
        }
        CheckBool( true );

        // 書換え直後の共有メモリの内容をダンプ
        DumpSharedMemory(adrs1, adrs2);

        // 共有メモリをアンマップ
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("両方の共有メモリをアンマップ"));
        nn::os::UnmapSharedMemory(&shmem1);
        nn::os::UnmapSharedMemory(&shmem2);

        // アンマップは何回でも実行できる
        // shmem1 は複数回、shmem2 は１回だけアンマップを行なう
        nn::os::UnmapSharedMemory(&shmem1);

        CheckBool( true );

        // 共有メモリを破棄
        //
        // shmem1 は直接 Create したものなので Destroy にてハンドルがシステム
        // に返却されるが、shmem2 は Attach 時に false を指定しているので
        // Destroy 時にはハンドルは何もされない。
        //
        // また、このテストでの Destroy の順番は shmem2 → shmem1 の順である
        // 必要がある。shmem2 は false で Attach されたものなので、shmem1 が
        // 先にハンドルの管理を終了してしまうのはロジックとして正しくない。
        // 複数プロセスの場合は、両プロセスがそれぞれ共有メモリハンドルの
        // 管理責任があるので、Destroy の順番は順不同で問題ない。

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("両方の共有メモリを破棄"));
        nn::os::DestroySharedMemory(&shmem2);
        nn::os::DestroySharedMemory(&shmem1);
        CheckBool( true );
    }

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  nn::os::SharedMemory クラス( アドレス指定可能 Map なし ) 版のテスト
//
TEST(SharedMemoryClassTest, test_SharedMemoryTest)
{
    CLEAR_GOOGLE_TEST();

    {
        // 共有メモリオブジェクトＡの生成と初期化
        SEQ_SET(100);
        NNT_OS_LOG(NN_TEXT("共有メモリＡの生成"));
        nn::os::SharedMemory    shmem1(shmemSize,
                                        nn::os::MemoryPermission_ReadWrite,
                                        nn::os::MemoryPermission_ReadWrite);
        auto handle = shmem1.GetHandle();
        NNT_OS_LOG(" handle=0x%08x", handle);

        // 共有メモリオブジェクトＢの初期化
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡのハンドルを使って共有メモリＢを初期化"));
        nn::os::SharedMemory    shmem2;
        shmem2.Attach(shmemSize, handle, false);
        CheckBool( true );

        // 共有メモリＡをマップ
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡをマップ： "));
        void* adrs1 = shmem1.Map(nn::os::MemoryPermission_ReadWrite);
        NNT_OS_LOG("address1= 0x%p\n", adrs1);
        if (adrs1 == NULL)
        {
            FAIL();
        }

        // 共有メモリＢをマップ
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＢをマップ： "));
        void* adrs2 = shmem2.Map(nn::os::MemoryPermission_ReadWrite);
        NNT_OS_LOG("address2= 0x%p\n", adrs2);
        if ((adrs2 == NULL) || (adrs1 == adrs2))
        {
            FAIL();
        }

        // 共有メモリＡのアドレスとサイズをチェック
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡのアドレスとサイズをチェック"));
        auto gotAddress1 = shmem1.GetMappedAddress();
        auto gotSize1    = shmem1.GetSize();
        CheckBool( (gotAddress1 == adrs1) && (gotSize1 == shmemSize) );

        // 共有メモリＢのアドレスとサイズをチェック
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＢのアドレスとサイズをチェック"));
        auto gotAddress2 = shmem2.GetMappedAddress();
        auto gotSize2    = shmem2.GetSize();
        CheckBool( (gotAddress2 == adrs2) && (gotSize2 == shmemSize) );

        // マップ直後の共有メモリの内容をダンプ
        DumpSharedMemory(adrs1, adrs2);

        // 共有メモリＡ の内容をビット反転して 共有メモリＢ にコピー
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡの内容をビット反転して共有メモリＢにコピー"));
        uint64_t* src = reinterpret_cast<uint64_t*>(adrs1);
        uint64_t* dst = reinterpret_cast<uint64_t*>(adrs2);

        for (size_t count = 0; count < shmemSize; count += sizeof(uint64_t))
        {
            *dst = ~(*src);
            ++dst;
            ++src;
        }
        CheckBool( true );

        // 共有メモリＡ の内容と 共有メモリＢ の内容が一致しているか確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("共有メモリＡの内容と共有メモリＢの内容が一致しているか"));
        src = reinterpret_cast<uint64_t*>(adrs1);
        dst = reinterpret_cast<uint64_t*>(adrs2);
        for (size_t count = 0; count < shmemSize; count += sizeof(uint64_t))
        {
            if (*src != *dst)
            {
                NNT_OS_LOG("\n");
                NNT_OS_LOG(NN_TEXT("不一致： address1=0x%p  address2=0x%p\n"), src, dst);
                FAIL();
            }
            ++dst;
            ++src;
        }
        CheckBool( true );

        // 書換え直後の共有メモリの内容をダンプ
        DumpSharedMemory(adrs1, adrs2);

#if 1
        // SharedMemory 食らうのデストラクタで自動的にアンマップされるため、
        // ここではあえて明示的な Unmap() を行なわない。

        // 共有メモリをアンマップ
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("両方の共有メモリをアンマップ"));
        shmem1.Unmap();
        shmem2.Unmap();
        CheckBool( true );
#endif

        // 共有メモリを破棄（デストラクタで破棄）
        //
        // shmem1 は直接 Create したものなので Destroy にてハンドルがシステム
        // に返却されるが、shmem2 は Attach 時に false を指定しているので
        // Destroy 時にはハンドルは何もされない。
        //
        // また、このテストでの Destroy の順番は shmem2 → shmem1 の順である
        // 必要がある。shmem2 は false で Attach されたものなので、shmem1 が
        // 先にハンドルの管理を終了してしまうのはロジックとして正しくない。
        // 複数プロセスの場合は、両プロセスがそれぞれ共有メモリハンドルの
        // 管理責任があるので、Destroy の順番は順不同で問題ない。

        SEQ_NONE();
        CheckBool( true );
    }

    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------

}}} // namespace nnt::os::sharedMemory

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of SharedMemory APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of SharedMemory APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
