﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  MessageQueue 専用のテストヘルパー
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_MessageQueueHelper.h"


//---------------------------------------------------------------------------

namespace nnt { namespace os { namespace messageQueue {

int                 g_seq;
uintptr_t           g_MessageQueueBuffer[ messageQueueSize ];
MessageQueueHelper  g_MqHelper;

NN_ALIGNAS( 4096 ) char g_StackOfThread1[ 16384 ];
NN_ALIGNAS( 4096 ) char g_StackOfThread2[ 16384 ];

//---------------------------------------------------------------------------
// スレッド１の生成
void MessageQueueHelper::CreateThread1( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Create Thread1:");
    nn::Result err = nn::os::CreateThread(
                        &m_Thread1,                     // Thread オブジェクト
                        &ThreadFunc1,                   // エントリ関数
                        reinterpret_cast<void*>(0x1111),// 引数
                        g_StackOfThread1,               // スタック
                        16384,                          // スタックサイズ
                        priThread1);                    // 優先度

    CheckBool( err.IsSuccess() );
}

// スレッド２の生成
void MessageQueueHelper::CreateThread2( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Create Thread2:");
    nn::Result err = nn::os::CreateThread(
                      &m_Thread2,                     // Thread オブジェクト
                      &ThreadFunc2,                   // エントリ関数
                      reinterpret_cast<void*>(0x2222),// 引数
                      g_StackOfThread2,               // スタック
                      16384,                          // スタックサイズ
                      priThread2);                    // 優先度

    CheckBool( err.IsSuccess() );
}

//---------------------------------------------------------------------------
// スレッド１の破棄
void MessageQueueHelper::DestroyThread1( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Destroy Thread1:");
    nn::os::DestroyThread( &m_Thread1 );

    CheckBool( true );
}

// スレッド２の破棄
void MessageQueueHelper::DestroyThread2( TestThreadAction state )
{
    if (state == THR_NOTUSE)
    {
        return;
    }

    SEQ_NONE();
    NNT_OS_LOG("Destroy Thread2:");
    nn::os::DestroyThread( &m_Thread2 );

    CheckBool( true );
}


//---------------------------------------------------------------------------
// MessageQueue オブジェクトの Initialize
void MessageQueueHelper::InitializeMessage()
{
    SEQ_NONE();
    NNT_OS_LOG("Initialize MessageQueue max=%d:", messageQueueSize);
    nn::os::InitializeMessageQueue(
                        &m_MessageQueue,        // MessageQueue オブジェクト
                        g_MessageQueueBuffer,   // MessageQueue バッファ
                        messageQueueSize);      // MessageQueue サイズ

    CheckBool( true );
}

//---------------------------------------------------------------------------
// MessageQueue オブジェクトのクリーンナップ（待ちスレッドを全解除）
void MessageQueueHelper::Cleanup()
{
    uintptr_t   data;

    // Cleanup 後の状態へ
    m_isCleanuped = true;

    // メッセージキューを操作して、WAITING したままのスレッドを起こす
    for (;;)
    {
        bool cancelThread1 = (GetThreadExpectState1() == THR_STATE_NONE) ||
                             (m_Thread1._state == nn::os::ThreadType::State_Exited);
        bool cancelThread2 = (GetThreadExpectState2() == THR_STATE_NONE) ||
                             (m_Thread2._state == nn::os::ThreadType::State_Exited);

        if ( cancelThread1 && cancelThread2 )
        {
            break;
        }

        // ダミーでメッセージキューに送信
        for (int i=0; i<messageQueueSize; i++)
        {
            nn::os::TrySendMessageQueue( &m_MessageQueue, 0 );
        }
        testSleep( NNT_CALIBRATE_RATE() * 2 );

        // ダミーでメッセージキューから受信
        for (int i=0; i<messageQueueSize; i++)
        {
            nn::os::TryReceiveMessageQueue( &data, &m_MessageQueue );
        }
        testSleep( NNT_CALIBRATE_RATE() * 2 );
    }

    // ２つのスレッドが本当に終了するまで待つ
    if (GetThreadExpectState1() != THR_STATE_NONE)
    {
        nn::os::WaitThread( &m_Thread1 );
    }
    if (GetThreadExpectState2() != THR_STATE_NONE)
    {
        nn::os::WaitThread( &m_Thread2 );
    }
}

//---------------------------------------------------------------------------
// MessageQueue オブジェクトの Finalize
void MessageQueueHelper::FinalizeMessage()
{
    SEQ_NONE();
    NNT_OS_LOG("Finalize MessageQueue:");
    nn::os::FinalizeMessageQueue( &m_MessageQueue );

    CheckBool( true );
}

//---------------------------------------------------------------------------
// MessageQueue オブジェクト状態の構築
void MessageQueueHelper::ConstructMessageQueue( int mq )
{
    // メッセージキューの初期状態を構築
    SEQ_NONE();
    NNT_OS_LOG("Construct initial MessageQueue: num=%d", mq);

    // メッセージキューにデータをためる
    for (int i = 1; i <= mq; ++i)
    {
        uintptr_t   data = (i << 24) | (i << 16) | (i << 8) | i;
        SendMessageQueue( &m_MessageQueue, data );
    }
    CheckBool( true );
}

//---------------------------------------------------------------------------
//  スレッド実行の開始
void MessageQueueHelper::StartThread1( TestThreadAction state )
{
    SEQ_NONE();
    if (state != THR_NOTUSE)
    {
        NNT_OS_LOG("Start Thread1:");
        nn::os::StartThread( &m_Thread1 );     // Thread オブジェクト
        CheckBool( true );
    }
    else
    {
        NNT_OS_LOG("Thread1 is not used\n");
    }
}

void MessageQueueHelper::StartThread2( TestThreadAction state )
{
    SEQ_NONE();
    if (state != THR_NOTUSE)
    {
        NNT_OS_LOG("Start Thread2:");
        nn::os::StartThread( &m_Thread2 );     // Thread オブジェクト
        CheckBool( true );
    }
    else
    {
        NNT_OS_LOG("Thread2 is not used\n");
    }
}

//---------------------------------------------------------------------------
// メッセージキューの内部状態のチェック
void    MessageQueueHelper::MessageQueueCheck( int mqex )
{
    CheckParam( GetMessageQueueCount(), mqex );
}

//---------------------------------------------------------------------------
// スレッド状態のチェック
void    MessageQueueHelper::ThreadStateCheck(
                                    const nn::os::ThreadType& pThread,
                                    TestThreadAction          action,
                                    TestThreadLastState       expect,
                                    bool                      apiResult,
                                    bool                      apiExpect )
{
    switch (pThread._state)
    {
        case nn::os::ThreadType::State_Initialized:
            {
                NNT_OS_LOG(" INITIALIZED");
            }
            break;

        case nn::os::ThreadType::State_Started:
            {
                NNT_OS_LOG(" STARTED");
            }
            break;

        case nn::os::ThreadType::State_Exited:
            {
                if (( action == THR_WAIT_TIMED_SEND) ||
                    ( action == THR_WAIT_TIMED_JAM) ||
                    ( action == THR_WAIT_TIMED_RECV) ||
                    ( action == THR_WAIT_TIMED_PEEK))
                {
                    NNT_OS_LOG(" TIMEDOUT->EXITED");
                }
                else
                {
                    NNT_OS_LOG(" EXITED");
                }
            }
            break;

        case nn::os::ThreadType::State_NotInitialized:
            {
                NNT_OS_LOG(" NotInitialized");
            }
            break;

        default:
            {
                NNT_OS_LOG(" UNKNOWN");
            }
            break;
    }


    switch (expect)
    {
        // スレッド未使用
        case THR_STATE_NONE:
            {
                CheckBool( true );
            }
            break;

        // EXITED
        case THR_STATE_EXITED:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Exited );
                NNT_OS_LOG("(-----): API return value:");
                CheckParam( apiResult, apiExpect );
            }
            break;

        // WAITING（待ちを継続）
        case THR_STATE_WAITING:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Started );
            }
            break;

        // タイムアウトを検知して終了
        case THR_STATE_TIMEDOUT:
            {
                CheckParam( pThread._state, nn::os::ThreadType::State_Exited );
                NNT_OS_LOG("(-----): API return value:");
                CheckParam( apiResult, apiExpect );
            }
            break;

        // 想定外のエラー（テスト不具合）
        default:
            {
                NN_ASSERT( false );
            }
            break;
    }
}


//---------------------------------------------------------------------------
//  OS-API 発行
//---------------------------------------------------------------------------

bool    MessageQueueHelper::CallMessageQueueApi(
                                        TestThreadAction    action,
                                        int                 seq,
                                        TestThreadLastState expectState,
                                        uintptr_t           expectValue)
{
    const uintptr_t value   = valueAdd;
    bool            result  = true;
    auto            timeout = nn::TimeSpan::FromNanoSeconds( NNT_CALIBRATE_RATE() * (expectState == THR_STATE_TIMEDOUT ? 10 : 10000 ) );

    NN_UNUSED( expectValue );

    SEQ_CHECK( seq );
    NNT_OS_LOG("Call ");

    switch (action)
    {
        // NOTUSE 系（スレッドを使用しない）
        case THR_NOTUSE:
            {
                NN_ASSERT( false );
            }
            break;

        // SendMessage() 待ち状態（無期限）
        case THR_WAIT_SEND:
            {
                NNT_OS_LOG("SendMessageQueue() goes WAIT\n");
                nn::os::SendMessageQueue( &m_MessageQueue, value );
                result = true;
            }
            break;

        // TimedSendMessage() 待ち状態（時限あり）
        case THR_WAIT_TIMED_SEND:
            {
                NNT_OS_LOG("TimedSendMessageQueue() goes WAIT\n");
                result = nn::os::TimedSendMessageQueue( &m_MessageQueue, value, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TimedSendMessage() 待ち状態（負のタイムアウト値）
        case THR_WAIT_TIMED_SEND_MINUS:
            {
                NNT_OS_LOG("TimedSendMessageQueue() goes WAIT (timeout=-1nsec)\n");
                result = nn::os::TimedSendMessageQueue( &m_MessageQueue, value, nn::TimeSpan::FromNanoSeconds( -1 ) );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // JamMessage() 待ち状態（無期限）
        case THR_WAIT_JAM:
            {
                NNT_OS_LOG("JamMessageQueue() goes WAIT\n");
                nn::os::JamMessageQueue( &m_MessageQueue, value );
                result = true;
            }
            break;

        // TimedJamMessage() 待ち状態（時限あり）
        case THR_WAIT_TIMED_JAM:
            {
                NNT_OS_LOG("TimedJamMessageQueue() goes WAIT\n");
                result = nn::os::TimedJamMessageQueue( &m_MessageQueue, value, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TimedJamMessage() 待ち状態（負のタイムアウト値）
        case THR_WAIT_TIMED_JAM_MINUS:
            {
                NNT_OS_LOG("TimedJamMessageQueue() goes WAIT (timeout=-1nsec)\n");
                result = nn::os::TimedJamMessageQueue( &m_MessageQueue, value, nn::TimeSpan::FromNanoSeconds( -1 ) );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // ReceiveMessage() 待ち状態（無期限）
        case THR_WAIT_RECV:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("ReceiveMessageQueue() goes WAIT\n");
                nn::os::ReceiveMessageQueue( &data, &m_MessageQueue );
                result = true;
            }
            break;

        // TimedReceiveMessage() 待ち状態（時限あり）
        case THR_WAIT_TIMED_RECV:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TimedReceiveMessageQueue() goes WAIT\n");
                result = nn::os::TimedReceiveMessageQueue( &data, &m_MessageQueue, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TimedReceiveMessage() 待ち状態（負のタイムアウト値）
        case THR_WAIT_TIMED_RECV_MINUS:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TimedReceiveMessageQueue() goes WAIT (timeout=-1nsec)\n");
                result = nn::os::TimedReceiveMessageQueue( &data, &m_MessageQueue, nn::TimeSpan::FromNanoSeconds( -1 ) );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // PeekMessage() 待ち状態（無期限）
        case THR_WAIT_PEEK:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("PeekMessageQueue() goes WAIT\n");
                nn::os::PeekMessageQueue( &data, &m_MessageQueue );
                result = true;
            }
            break;

        // TimedPeekMessage() 待ち状態（時限あり）
        case THR_WAIT_TIMED_PEEK:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TimedPeekMessageQueue() goes WAIT\n");
                result = nn::os::TimedPeekMessageQueue( &data, &m_MessageQueue, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TimedPeekMessage() 待ち状態（負のタイムアウト値）
        case THR_WAIT_TIMED_PEEK_MINUS:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TimedPeekMessageQueue() goes WAIT (timeout=-1nsec)\n");
                result = nn::os::TimedPeekMessageQueue( &data, &m_MessageQueue, nn::TimeSpan::FromNanoSeconds( -1 ) );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;


        // SendMessageQueue() の発行
        case THR_CALL_SEND:
            {
                NNT_OS_LOG("SendMessageQueue()\n");
                nn::os::SendMessageQueue( &m_MessageQueue, value );
                result = true;
            }
            break;

        // TrySendMessageQueue() の発行
        case THR_CALL_TRY_SEND:
            {
                NNT_OS_LOG("TrySendMessageQueue()\n");
                result = nn::os::TrySendMessageQueue( &m_MessageQueue, value );
            }
            break;

        // TimedSendMessageQueue() の発行
        case THR_CALL_TIMED_SEND:
            {
                NNT_OS_LOG("TimedSendMessageQueue()\n");
                result = nn::os::TimedSendMessageQueue( &m_MessageQueue, value, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TimedSendMessageQueue() の発行（負のタイムアウト値）
        case THR_CALL_TIMED_SEND_MINUS:
            {
                NNT_OS_LOG("TimedSendMessageQueue() (timeout=-1nsec)\n");
                result = nn::os::TimedSendMessageQueue( &m_MessageQueue, value, nn::TimeSpan::FromNanoSeconds( -1 ) );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // JamMessageQueue() の発行
        case THR_CALL_JAM:
            {
                NNT_OS_LOG("JamMessageQueue()\n");
                nn::os::JamMessageQueue( &m_MessageQueue, value );
                result = true;
            }
            break;

        // TryJamMessageQueue() の発行
        case THR_CALL_TRY_JAM:
            {
                NNT_OS_LOG("TryJamMessageQueue()\n");
                result = nn::os::TryJamMessageQueue( &m_MessageQueue, value );
            }
            break;

        // JamMessageQueue() の発行
        case THR_CALL_TIMED_JAM:
            {
                NNT_OS_LOG("TimedJamMessageQueue()\n");
                result = nn::os::TimedJamMessageQueue( &m_MessageQueue, value, timeout );
            }
            break;

        // JamMessageQueue() の発行
        case THR_CALL_TIMED_JAM_MINUS:
            {
                NNT_OS_LOG("TimedJamMessageQueue() (timeout=-1nsec)\n");
                result = nn::os::TimedJamMessageQueue( &m_MessageQueue, value, nn::TimeSpan::FromNanoSeconds( -1 ) );

                // THR_CALL_TIMED_JAM に以下のコードが無かったが意図したもの？　本当は必要かも？
                //
                // タイムアウトによる待ち解除なら、その旨を返す
                //if ( result == false )
                //{
                //    return result;
                //}
            }
            break;

        // TimedReceiveMessageQueue() の発行
        case THR_CALL_RECV:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("ReceiveMessageQueue()\n");
                nn::os::ReceiveMessageQueue( &data, &m_MessageQueue );
                result = true;

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TryReceiveMessageQueue() の発行
        case THR_CALL_TRY_RECV:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TryReceiveMessageQueue()\n");
                result = nn::os::TryReceiveMessageQueue( &data, &m_MessageQueue );
            }
            break;

        // TimedReceiveMessageQueue() の発行
        case THR_CALL_TIMED_RECV:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TimedReceiveMessageQueue()\n");
                result = nn::os::TimedReceiveMessageQueue( &data, &m_MessageQueue, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TimedReceiveMessageQueue() の発行（負のタイムアウト値）
        case THR_CALL_TIMED_RECV_MINUS:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TimedReceiveMessageQueue() (timeout=-1nsec)\n");
                result = nn::os::TimedReceiveMessageQueue( &data, &m_MessageQueue, nn::TimeSpan::FromNanoSeconds( -1 ) );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // PeekMessageQueue() の発行
        case THR_CALL_PEEK:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("PeekMessageQueue()\n");
                nn::os::PeekMessageQueue( &data, &m_MessageQueue );
                result = true;
            }
            break;

        // TryPeekMessageQueue() の発行
        case THR_CALL_TRY_PEEK:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TryPeekMessageQueue()\n");
                result = nn::os::TryPeekMessageQueue( &data, &m_MessageQueue );
            }
            break;

        // TimedPeekMessageQueue() の発行
        case THR_CALL_TIMED_PEEK:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TimedPeekMessageQueue()\n");
                result = nn::os::TimedPeekMessageQueue( &data, &m_MessageQueue, timeout );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // TimedPeekMessageQueue() の発行（負のタイムアウト値）
        case THR_CALL_TIMED_PEEK_MINUS:
            {
                uintptr_t   data = 0;
                NNT_OS_LOG("TimedPeekMessageQueue() (timeout=-1nsec)\n");
                result = nn::os::TimedPeekMessageQueue( &data, &m_MessageQueue, nn::TimeSpan::FromNanoSeconds( -1 ) );

                // タイムアウトによる待ち解除なら、その旨を返す
                if ( result == false )
                {
                    return result;
                }
            }
            break;

        // 想定外のエラー（テスト不具合）
        default:
            {
                NN_ASSERT( false );
            }
            break;
    }

    return result;
}   // NOLINT(readability/fn_size)

//---------------------------------------------------------------------------
//  テスト用スレッド関数の本体
//---------------------------------------------------------------------------
void    ThreadFunc1(void* exinf)
{
    TestThreadAction    action      = g_MqHelper.GetThreadAction1();
    TestThreadLastState expectState = g_MqHelper.GetThreadExpectState1();
    bool                expect      = g_MqHelper.GetThreadExpect1();
    bool                result;

    NN_UNUSED( exinf );

    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_START );

    // スレッド起動
    SEQ_CHECK( g_seq + 1 );
    NNT_OS_LOG("Begin of thread1:");
    CheckBool( true );

    // Thread1 から API を発行
    g_MqHelper.SetApiResult1( false );
    result = g_MqHelper.CallMessageQueueApi( action, g_seq + 2, expectState,
                                             g_MqHelper.GetExpectValue1() );
    g_MqHelper.SetApiResult1( result );

    // Cleanuped ならスレッド終了（THR_STATE_WAITING だった場合など）
    if ( g_MqHelper.IsCleanuped() )
    {
        NNT_OS_LOG("(-----): Thread1: Canceled\n");
        return;
    }

    SEQ_SYNC_ACK( NNT_POINT_THREAD1_CALLED );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD1_CALLED );

    // スレッドの事後状態別の後処理
    switch (expectState)
    {
        // API から普通に戻ってきた場合
        case THR_STATE_EXITED:
            {
                SEQ_CHECK( g_seq + 5 );
                NNT_OS_LOG("Thread1: return from API:");
                CheckParam( result, expect );
            }
            break;

        case THR_STATE_TIMEDOUT:
            break;

        // 想定外のエラー（テスト不具合）
        case THR_STATE_WAITING:
        default:
            CheckBool( false );
            break;
    }

    // スレッド終了
    SEQ_CHECK( expectState == THR_STATE_TIMEDOUT ? g_seq + 9 : g_seq + 6 );
    NNT_OS_LOG("End of thread1:");
    CheckBool( true );

    SEQ_SYNC_ACK( NNT_POINT_THREAD1_END );
}


void    ThreadFunc2(void* exinf)
{
    TestThreadAction    action      = g_MqHelper.GetThreadAction2();
    TestThreadLastState expectState = g_MqHelper.GetThreadExpectState2();
    bool                expect      = g_MqHelper.GetThreadExpect2();
    bool                result;

    NN_UNUSED( exinf );

    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_START );

    // スレッド起動
    SEQ_CHECK( g_seq + 3 );
    NNT_OS_LOG("Begin of thread2:");
    CheckBool( true );

    // Thread2 から API を発行
    g_MqHelper.SetApiResult2( false );
    result = g_MqHelper.CallMessageQueueApi( action, g_seq + 4, expectState,
                                             g_MqHelper.GetExpectValue2() );
    g_MqHelper.SetApiResult2( result );

    // Cleanuped ならスレッド終了（THR_STATE_WAITING だった場合など）
    if ( g_MqHelper.IsCleanuped() )
    {
        NNT_OS_LOG("(-----): Thread2: Canceled\n");
        return;
    }

    SEQ_SYNC_ACK( NNT_POINT_THREAD2_CALLED );
    SEQ_SYNC_WAIT( NNT_POINT_THREAD2_CALLED );

    // スレッドの事後状態別の後処理
    switch (expectState)
    {
        // API から普通に戻ってきた場合
        case THR_STATE_EXITED:
            {
                SEQ_CHECK( g_seq + 7 );
                NNT_OS_LOG("Thread2: return from API:");
                CheckParam( result, expect );
            }
            break;

        case THR_STATE_TIMEDOUT:
            break;

        // 想定外のエラー（テスト不具合）
        case THR_STATE_WAITING:
        default:
            CheckBool( false );
            break;
    }

    // スレッド終了
    SEQ_CHECK( expectState == THR_STATE_TIMEDOUT? g_seq + 10 : g_seq + 8 );
    NNT_OS_LOG("End of thread2:");
    CheckBool( true );

    SEQ_SYNC_ACK( NNT_POINT_THREAD2_END );
}

//---------------------------------------------------------------------------
// Receive/Get 系で受信時の期待値を設定
//---------------------------------------------------------------------------
void    MessageQueueHelper::SetExpectValue( uintptr_t value1, uintptr_t value2 )
{
    m_expectValue1 = value1;
    m_expectValue2 = value2;
}

//---------------------------------------------------------------------------

int    MessageQueueHelper::GetLastSeq( TestThreadLastState th1ex,
                                       TestThreadLastState th2ex )
{
    static const int lastSeqTable[THR_STATE_MAX][THR_STATE_MAX] =
    {
    //  th2: NONE,WAITING, EXITED, TIMEDOUT
        {       0,      0,      0,      0,      },  // NONE
        {       2,      4,      8,      10,     },  // WAITING
        {       6,      6,      8,      10,     },  // EXITED
        {       9,      9,      9,      10,     },  // TIMEDOUT
    };                                              // th1

    return g_seq + lastSeqTable[th1ex][th2ex];
}

//---------------------------------------------------------------------------
//  doTest() 関数
//---------------------------------------------------------------------------

void    MessageQueueHelper::doTest( int                 mq,
                                    TestThreadAction    th1,
                                    TestThreadAction    th2,
                                    bool                rslt1,
                                    bool                rslt2,
                                    int                 mqex,
                                    TestThreadLastState th1ex,
                                    TestThreadLastState th2ex,
                                    const char*         functionName,
                                    int                 line)
{
    // シーケンス番号を次の 100番台 へ進める
    g_seq = g_Sequence.Get();
    g_seq = ((g_seq + 100) / 100) * 100;

    // 個別のテスト結果をクリア
    CLEAR_GOOGLE_TEST();

    // テスト用 API の発行元情報を退避
    g_Sequence.SetFunctionName( functionName );
    g_Sequence.SetLineNumber( line );

    // Cleanup 前の状態へ
    m_isCleanuped = false;
    g_MqHelper.SetApiResult1( true );
    g_MqHelper.SetApiResult2( true );

    // スレッドの生成
    g_seq = (g_seq / 100) * 100;
    SEQ_SET( g_seq );
    NNT_OS_LOG("Start Test\n");
    CreateThread1( th1 );
    CreateThread2( th2 );

    // メッセージキューオブジェクトの Initialize
    InitializeMessage();

    // メッセージキューの状態を構築
    ConstructMessageQueue( mq );

    // スレッドの状態を構築
    SetThreadInfo1( th1, th1ex, rslt1 );
    SetThreadInfo2( th2, th2ex, rslt2 );

    // 順番にテストシーケンスを指示する
    SEQ_SYNC_INIT();

    StartThread1( th1 );
    StartThread2( th2 );
    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_START );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_CALLED );

    if (th2ex != THR_STATE_NONE)
    {
        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_START );
        SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_CALLED );
    }

    SEQ_SYNC_SIGNAL( NNT_POINT_THREAD1_CALLED );
    SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD1_END );

    if (th2ex != THR_STATE_NONE)
    {
        SEQ_SYNC_SIGNAL( NNT_POINT_THREAD2_CALLED );
        SEQ_SYNC_DELAY( 20, NNT_POINT_THREAD2_END );
    }

    // タイムアウト系の API は、もう片方のスレッドで同期が成立しない場合、
    // この時点で既にタイムアウトが成立し、対象スレッドは
    // THR_TIMEDOUT_EXITED 状態という形で既に Exited になっている。

    // スレッド１の状態を検査
    SEQ_WAIT_UNTIL( GetLastSeq(th1ex, th2ex), g_seq + 11 );
    NNT_OS_LOG("Thread1 state check:");
    ThreadStateCheck( m_Thread1, th1, th1ex, GetApiResult1(), rslt1 );

    // スレッド２の状態を検査
    SEQ_NONE();
    NNT_OS_LOG("Thread2 state check:");
    ThreadStateCheck( m_Thread2, th2, th2ex, GetApiResult2(), rslt2 );

    // メッセージキューの状態を検査
    SEQ_NONE();
    NNT_OS_LOG("MessageQueue state check:");
    MessageQueueCheck( mqex );

    // メッセージキュー状態を初期状態へ
    SEQ_NONE();
    NNT_OS_LOG("Cleanup MessageQueue and Threads\n");
    Cleanup();

    // メッセージキューオブジェクトの Finalize
    FinalizeMessage();

    // スレッドの破棄
    DestroyThread1( th1 );
    DestroyThread2( th2 );

    // 個別のテスト結果で GoogleTest に判定を通知
    JUDGE_GOOGLE_TEST();
}

}}} // namespace nnt::os:messageQueue

