﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  MessageQueue 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <nn/os.h>
#include <nn/nn_SdkText.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_MessageQueueHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace messageQueue {

namespace {
    const int maxCount = messageQueueSize;
};

//---------------------------------------------------------------------------
// 単独の SendMessageQueue() テスト

TEST(SendMessageQueue, test_SendMessage1)
{
    // SendMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_SEND,              // Thread1 は SendMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(SendMessageQueue, test_SendMessage2)
{
    // SendMessage 境界条件２(num=max-1)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_CALL_SEND,              // Thread1 は SendMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(SendMessageQueue, test_SendMessage3)
{
    // SendMessage 送信待ち(num=max)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_SEND,              // Thread1 は SendMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING);         // Thread1 は送信待ち
}

TEST(SendMessageQueue, test_SendMessage4)
{
    // SendMessage 送信待ち(num=max-1、Thread1 は通過、Thread2 は待機)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount - 1),       // MessageQueue は MAX 個
            THR_CALL_SEND,              // Thread1 は SendMessageQueue()
            THR_WAIT_SEND,              // Thread2 は SendMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_WAITING);         // Thread2 は送信待ち
}

TEST(SendMessageQueue, test_SendMessage5)
{
    // SendMessage 送信待ち(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_SEND,              // Thread1 は SendMessageQueue()
            THR_WAIT_SEND,              // Thread2 は SendMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING,          // Thread1 は送信待ち
            THR_STATE_WAITING);         // Thread2 は送信待ち
}

//---------------------------------------------------------------------------
// 単独の TrySendMessageQueue() テスト

TEST(TrySendMessageQueue, test_TrySendMessage1)
{
    // TrySendMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TRY_SEND,          // Thread1 は TrySendMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TrySendMessageQueue, test_TrySendMessage2)
{
    // TrySendMessage 境界条件２(num=max-1)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_CALL_TRY_SEND,          // Thread1 は TrySendMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TrySendMessageQueue, test_TrySendMessage3)
{
    // TrySendMessage 送信ポーリング(num=max)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TRY_SEND,          // Thread1 は TrySendMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TrySendMessageQueue, test_TrySendMessage4)
{
    // TrySendMessage 送信ポーリング(num=max-1、Thread1 は通過、Thread2 は待機)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount - 1),       // MessageQueue は MAX 個
            THR_CALL_TRY_SEND,          // Thread1 は TrySendMessageQueue()
            THR_CALL_TRY_SEND,          // Thread2 は TrySendMessageQueue()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TrySendMessageQueue, test_TrySendMessage5)
{
    // TrySendMessage 送信ポーリング(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TRY_SEND,          // Thread1 は TrySendMessageQueue()
            THR_CALL_TRY_SEND,          // Thread2 は TrySendMessageQueue()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
// 単独の TimedSendMessageQueue() テスト

TEST(TimedSendMessageQueue, test_TimedSendMessage1)
{
    // TimedSendMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedSendMessageQueue, test_TimedSendMessage2)
{
    // TimedSendMessage 境界条件２(num=max-1)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_CALL_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedSendMessageQueue, test_TimedSendMessage3)
{
    // TimedSendMessage 送信待ち(num=max、タイムアウト成立)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND,        // Thread1 は SendMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedSendMessageQueue, test_TimedSendMessage4)
{
    // TimedSendMessage 送信待ち(num=max-1、Thread1 は通過、Thread2 はTMOUT)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount - 1),       // MessageQueue は MAX 個
            THR_CALL_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            THR_WAIT_TIMED_SEND,        // Thread2 は TimedSendMessageQueue()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

TEST(TimedSendMessageQueue, test_TimedSendMessage5)
{
    // TimedSendMessage 送信待ち(num=max、Thread1/Thread2 共にTMOUT)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            THR_WAIT_TIMED_SEND,        // Thread2 は TimedSendMessageQueue()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedSendMessageQueue() テスト（負のタイムアウト値）

TEST(TimedSendMessageQueueWithMinusTimeout, test_TimedSendMessageWithMinusTimeout1)
{
    // TimedSendMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TIMED_SEND_MINUS,  // Thread1 は TimedSendMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedSendMessageQueueWithMinusTimeout, test_TimedSendMessageWithMinusTimeout2)
{
    // TimedSendMessage 送信待ち(num=max、タイムアウト成立)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND_MINUS,  // Thread1 は SendMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(TimedSendMessageQueueWithMinusTimeoutDeathTest, test_TimedSendMessageWithMinusTimeoutDeathTest)
{
    // TimedSendMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
                THR_CALL_TIMED_SEND_MINUS,  // Thread1 は TimedSendMessageQueue()
                true,                       // Thread1 での API 返値
                MQ_NUM(1),                  // MessageQueue は 1 個
                THR_STATE_EXITED)           // Thread1 は終了
    , "");
}


#endif

//---------------------------------------------------------------------------
// 単独の JamMessageQueue() テスト

TEST(JamMessageQueue, test_JamMessage1)
{
    // JamMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_JAM,               // Thread1 は JamMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(JamMessageQueue, test_JamMessage2)
{
    // JamMessage 境界条件２(num=max-1)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_CALL_JAM,               // Thread1 は JamMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(JamMessageQueue, test_JamMessage3)
{
    // JamMessage 送信待ち(num=max)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_JAM,               // Thread1 は JamMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING);         // Thread1 は送信待ち
}

TEST(JamMessageQueue, test_JamMessage4)
{
    // JamMessage 送信待ち(num=max-1、Thread1 は通過、Thread2 は待機)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount - 1),       // MessageQueue は MAX 個
            THR_CALL_JAM,               // Thread1 は JamMessageQueue()
            THR_WAIT_JAM,               // Thread2 は JamMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_WAITING);         // Thread2 は送信待ち
}

TEST(JamMessageQueue, test_JamMessage5)
{
    // JamMessage 送信待ち(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_JAM,               // Thread1 は JamMessageQueue()
            THR_WAIT_JAM,               // Thread2 は JamMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING,          // Thread1 は送信待ち
            THR_STATE_WAITING);         // Thread2 は送信待ち
}

//---------------------------------------------------------------------------
// 単独の TryJamMessageQueue() テスト

TEST(TryJamMessageQueue, test_TryJamMessage1)
{
    // TryJamMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TRY_JAM,           // Thread1 は TryJamMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryJamMessageQueue, test_TryJamMessage2)
{
    // TryJamMessage 境界条件２(num=max-1)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_CALL_TRY_JAM,           // Thread1 は TryJamMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryJamMessageQueue, test_TryJamMessage3)
{
    // TryJamMessage 送信ポーリング(num=max)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TRY_JAM,           // Thread1 は TryJamMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryJamMessageQueue, test_TryJamMessage4)
{
    // TryJamMessage 送信ポーリング(num=max-1、Thread1 は通過、Thread2 は待機)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount - 1),       // MessageQueue は MAX 個
            THR_CALL_TRY_JAM,           // Thread1 は TryJamMessageQueue()
            THR_CALL_TRY_JAM,           // Thread2 は TryJamMessageQueue()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TryJamMessageQueue, test_TryJamMessage5)
{
    // TryJamMessage 送信ポーリング(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TRY_JAM,           // Thread1 は TryJamMessageQueue()
            THR_CALL_TRY_JAM,           // Thread2 は TryJamMessageQueue()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
// 単独の TimedJamMessageQueue() テスト

TEST(TimedJamMessageQueue, test_TimedJamMessage1)
{
    // TimedJamMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TIMED_JAM,         // Thread1 は TimedJamMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedJamMessageQueue, test_TimedJamMessage2)
{
    // TimedJamMessage 境界条件２(num=max-1)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_CALL_TIMED_JAM,         // Thread1 は TimedJamMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedJamMessageQueue, test_TimedJamMessage3)
{
    // TimedJamMessage 送信待ち(num=max、タイムアウト成立)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM,         // Thread1 は JamMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedJamMessageQueue, test_TimedJamMessage4)
{
    // TimedJamMessage 送信待ち(num=max-1、Thread1 は通過、Thread2 はTMOUT)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount - 1),       // MessageQueue は MAX 個
            THR_CALL_TIMED_JAM,         // Thread1 は TimedJamMessageQueue()
            THR_WAIT_TIMED_JAM,         // Thread2 は TimedJamMessageQueue()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

TEST(TimedJamMessageQueue, test_TimedJamMessage5)
{
    // TimedJamMessage 送信待ち(num=max、Thread1/Thread2 共にTMOUT)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM,         // Thread1 は TimedJamMessageQueue()
            THR_WAIT_TIMED_JAM,         // Thread2 は TimedJamMessageQueue()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedJamMessageQueue() テスト（負のタイムアウト値）

TEST(TimedJamMessageQueueWithMinusTimeout, test_TimedJamMessageWithMinusTimeout1)
{
    // TimedJamMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TIMED_JAM_MINUS,   // Thread1 は TimedJamMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedJamMessageQueueWithMinusTimeout, test_TimedJamMessageWithMinusTimeout2)
{
    // TimedJamMessage 送信待ち(num=max、タイムアウト成立)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM_MINUS,   // Thread1 は JamMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(TimedJamMessageQueueWithMinusTimeoutDeathTest, test_TimedJamMessageWithMinusTimeoutDeathTest)
{
    // TimedJamMessage 境界条件１(num=0)
    setExpectValue( 0, 0 );
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
                THR_CALL_TIMED_JAM_MINUS,   // Thread1 は TimedJamMessageQueue()
                true,                       // Thread1 での API 返値
                MQ_NUM(1),                  // MessageQueue は 1 個
                THR_STATE_EXITED)           // Thread1 は終了
    , "");
}


#endif

//---------------------------------------------------------------------------
// 単独の ReceiveMessageQueue() テスト

TEST(ReceiveMessageQueue, test_ReceiveMessage1)
{
    // ReceiveMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_RECV,              // Thread1 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ReceiveMessageQueue, test_ReceiveMessage2)
{
    // ReceiveMessage 境界条件２(num=1)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_RECV,              // Thread1 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(ReceiveMessageQueue, test_ReceiveMessage3)
{
    // ReceiveMessage 受信待ち(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_RECV,              // Thread1 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_WAITING);         // Thread1 は受信待ち
}

TEST(ReceiveMessageQueue, test_ReceiveMessage4)
{
    // ReceiveMessage 受信待ち(num=1、Thread1 は通過、Thread2 は待機)
    setExpectValue( valueIndex0, 0 );
    doTest2(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_RECV,              // Thread1 は ReceiveMessageQueue()
            THR_WAIT_RECV,              // Thread2 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_WAITING);         // Thread2 は受信待ち
}

TEST(ReceiveMessageQueue, test_ReceiveMessage5)
{
    // ReceiveMessage 受信待ち(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_RECV,              // Thread1 は ReceiveMessageQueue()
            THR_WAIT_RECV,              // Thread2 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_WAITING,          // Thread1 は受信待ち
            THR_STATE_WAITING);         // Thread2 は受信待ち
}

//---------------------------------------------------------------------------
// 単独の TryReceiveMessageQueue() テスト

TEST(TryReceiveMessageQueue, test_TryReceiveMessage1)
{
    // TryReceiveMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TRY_RECV,          // Thread1 は TryReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryReceiveMessageQueue, test_TryReceiveMessage2)
{
    // TryReceiveMessage 境界条件２(num=1)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_TRY_RECV,          // Thread1 は TryReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryReceiveMessageQueue, test_TryReceiveMessage3)
{
    // TryReceiveMessage 受信ポーリング(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TRY_RECV,          // Thread1 は TryReceiveMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryReceiveMessageQueue, test_TryReceiveMessage4)
{
    // TryReceiveMessage 受信ポーリング(num=1、Thread1 は通過、Thread2 は待機)
    setExpectValue( valueIndex0, 0 );
    doTest2(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_TRY_RECV,          // Thread1 は TryReceiveMessageQueue()
            THR_CALL_TRY_RECV,          // Thread2 は TryReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TryReceiveMessageQueue, test_TryReceiveMessage5)
{
    // TryReceiveMessage 受信ポーリング(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TRY_RECV,          // Thread1 は TryReceiveMessageQueue()
            THR_CALL_TRY_RECV,          // Thread2 は TryReceiveMessageQueue()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
// 単独の TimedReceiveMessageQueue() テスト

TEST(TimedReceiveMessageQueue, test_TimedReceiveMessage1)
{
    // TimedReceiveMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TIMED_RECV,        // Thread1 は TimedReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedReceiveMessageQueue, test_TimedReceiveMessage2)
{
    // TimedReceiveMessage 境界条件２(num=1)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_TIMED_RECV,        // Thread1 は TimedReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedReceiveMessageQueue, test_TimedReceiveMessage3)
{
    // TimedReceiveMessage 受信待ち(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV,        // Thread1 は TimedReceiveMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedReceiveMessageQueue, test_TimedReceiveMessage4)
{
    // TimedReceiveMessage 受信待ち(num=1、Thread1 は通過、Thread2 は待機)
    setExpectValue( valueIndex0, 0 );
    doTest2(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_TIMED_RECV,        // Thread1 は TimedReceiveMessageQueue()
            THR_WAIT_TIMED_RECV,        // Thread2 は TimedReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

TEST(TimedReceiveMessageQueue, test_TimedReceiveMessage5)
{
    // TimedReceiveMessage 受信待ち(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV,        // Thread1 は TimedReceiveMessageQueue()
            THR_WAIT_TIMED_RECV,        // Thread2 は TimedReceiveMessageQueue()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedReceiveMessageQueue() テスト（負のタイムアウト値）

TEST(TimedReceiveMessageQueueWithMinusTimeout, test_TimedReceiveMessageWithMinusTimeout1)
{
    // TimedReceiveMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TIMED_RECV_MINUS,  // Thread1 は TimedReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedReceiveMessageQueueWithMinusTimeout, test_TimedReceiveMessageWithMinusTimeout2)
{
    // TimedReceiveMessage 受信待ち(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV_MINUS,  // Thread1 は TimedReceiveMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(TimedReceiveMessageQueueWithMinusTimeoutDeathTest, test_TimedReceiveMessageWithMinusTimeoutDeathTest)
{
    // TimedReceiveMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
                THR_CALL_TIMED_RECV_MINUS,  // Thread1 は TimedReceiveMessageQueue()
                true,                       // Thread1 での API 返値
                MQ_NUM(maxCount - 1),       // MessageQueue は MAX-1 個
                THR_STATE_EXITED)           // Thread1 は終了
    , "");
}


#endif

//---------------------------------------------------------------------------
// 単独の PeekMessageQueue() テスト

TEST(PeekMessageQueue, test_PeekMessage1)
{
    // PeekMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_PEEK,              // Thread1 は PeekMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個（減らない）
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(PeekMessageQueue, test_PeekMessage2)
{
    // PeekMessage 境界条件２(num=1)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_PEEK,              // Thread1 は PeekMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個（減らない）
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(PeekMessageQueue, test_PeekMessage3)
{
    // PeekMessage 取得待ち(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_PEEK,              // Thread1 は PeekMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_WAITING);         // Thread1 は取得待ち
}

TEST(PeekMessageQueue, test_PeekMessage4)
{
    // PeekMessage 取得待ち(num=1、Thread1/2 は共に通過）
    setExpectValue( valueIndex0, valueIndex0 );
    doTest2(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_PEEK,              // Thread1 は PeekMessageQueue()
            THR_CALL_PEEK,              // Thread2 は PeekMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個（減らない）
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(PeekMessageQueue, test_PeekMessage5)
{
    // PeekMessage 取得待ち(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_PEEK,              // Thread1 は PeekMessageQueue()
            THR_WAIT_PEEK,              // Thread2 は PeekMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_WAITING,          // Thread1 は取得待ち
            THR_STATE_WAITING);         // Thread2 は取得待ち
}

//---------------------------------------------------------------------------
// 単独の TryPeekMessageQueue() テスト

TEST(TryPeekMessageQueue, test_TryPeekMessage1)
{
    // TryPeekMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TRY_PEEK,          // Thread1 は TryPeekMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個（減らない）
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryPeekMessageQueue, test_TryPeekMessage2)
{
    // TryPeekMessage 境界条件２(num=1)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_TRY_PEEK,          // Thread1 は TryPeekMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個（減らない）
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryPeekMessageQueue, test_TryPeekMessage3)
{
    // TryPeekMessage 取得ポーリング(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TRY_PEEK,          // Thread1 は TryPeekMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TryPeekMessageQueue, test_TryPeekMessage4)
{
    // TryPeekMessage 取得ポーリング(num=1、Thread1/2 は共に通過）
    setExpectValue( valueIndex0, valueIndex0 );
    doTest2(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_TRY_PEEK,          // Thread1 は TryPeekMessageQueue()
            THR_CALL_TRY_PEEK,          // Thread2 は TryPeekMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個（減らない）
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(TryPeekMessageQueue, test_TryPeekMessage5)
{
    // TryPeekMessage 取得ポーリング(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_CALL_TRY_PEEK,          // Thread1 は TryPeekMessageQueue()
            THR_CALL_TRY_PEEK,          // Thread2 は TryPeekMessageQueue()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
// 単独の TimedPeekMessageQueue() テスト

TEST(TimedPeekMessageQueue, test_TimedPeekMessage1)
{
    // TimedPeekMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TIMED_PEEK,        // Thread1 は TimedPeekMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個（減らない）
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedPeekMessageQueue, test_TimedPeekMessage2)
{
    // TimedPeekMessage 境界条件２(num=1)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_TIMED_PEEK,        // Thread1 は TimedPeekMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個（減らない）
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedPeekMessageQueue, test_TimedPeekMessage3)
{
    // TimedPeekMessage 取得待ち(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK,        // Thread1 は TimedPeekMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

TEST(TimedPeekMessageQueue, test_TimedPeekMessage4)
{
    // TimedPeekMessage 取得待ち(num=1、Thread1/2 は共に通過）
    setExpectValue( valueIndex0, valueIndex0 );
    doTest2(MQ_NUM(1),                  // MessageQueue は 1 個
            THR_CALL_TIMED_PEEK,        // Thread1 は TimedPeekMessageQueue()
            THR_CALL_TIMED_PEEK,        // Thread2 は TimedPeekMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(1),                  // MessageQueue は 1 個（減らない）
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(TimedPeekMessageQueue, test_TimedPeekMessage5)
{
    // TimedPeekMessage 取得待ち(num=max、スレッド２つ)
    setExpectValue( 0, 0 );
    doTest2(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK,        // Thread1 は TimedPeekMessageQueue()
            THR_WAIT_TIMED_PEEK,        // Thread2 は TimedPeekMessageQueue()
            false,                      // Thread1 での API 返値
            false,                      // Thread2 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_TIMEDOUT,         // Thread1 はタイムアウト
            THR_STATE_TIMEDOUT);        // Thread2 はタイムアウト
}

// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedPeekMessageQueue() テスト（負のタイムアウト値）

TEST(TimedPeekMessageQueueWithMinusTimeout, test_TimedPeekMessageWithMinusTimeout1)
{
    // TimedPeekMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_CALL_TIMED_PEEK_MINUS,  // Thread1 は TimedPeekMessageQueue()
            true,                       // Thread1 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個（減らない）
            THR_STATE_EXITED);          // Thread1 は終了
}

TEST(TimedPeekMessageQueueWithMinusTimeout, test_TimedPeekMessageWithMinusTimeout2)
{
    // TimedPeekMessage 取得待ち(num=0)
    setExpectValue( 0, 0 );
    doTest1(MQ_NUM(0),                  // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK_MINUS,  // Thread1 は TimedPeekMessageQueue()
            false,                      // Thread1 での API 返値
            MQ_NUM(0),                  // MessageQueue は 0 個
            THR_STATE_TIMEDOUT);        // Thread1 はタイムアウト
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(TimedPeekMessageQueueWithMinusTimeoutDeathTest, test_TimedPeekMessageWithMinusTimeoutDeathTest)
{
    // TimedPeekMessage 境界条件１(num=max)
    setExpectValue( valueIndex0, 0 );
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(MQ_NUM(maxCount),           // MessageQueue は MAX 個
                THR_CALL_TIMED_PEEK_MINUS,  // Thread1 は TimedPeekMessageQueue()
                true,                       // Thread1 での API 返値
                MQ_NUM(maxCount),           // MessageQueue は MAX 個（減らない）
                THR_STATE_EXITED)           // Thread1 は終了
    , "");
}


#endif

//---------------------------------------------------------------------------
// Send 待機中に Receive する（Try/Timed 含む）

TEST(CombinedMessageQueueTest1, test_SendAndReceive)
{
    // Thread1 で Send()、Thread2 で Receive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_SEND,              // Thread1 は SendMessageQueue()
            THR_CALL_RECV,              // Thread2 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest1, test_SendAndTryReceive)
{
    // Thread1 で Send()、Thread2 で TryReceive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_SEND,              // Thread1 は SendMessageQueue()
            THR_CALL_TRY_RECV,          // Thread2 は TryReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest1, test_SendAndTimedReceive)
{
    // Thread1 で Send()、Thread2 で TimedReceive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_SEND,              // Thread1 は SendMessageQueue()
            THR_CALL_TIMED_RECV,        // Thread2 は TimedReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest1, test_TimedSendAndReceive)
{
    // Thread1 で TimedSend()、Thread2 で Receive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            THR_CALL_RECV,              // Thread2 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest1, test_TimedSendAndTryReceive)
{
    // Thread1 で TimedSend()、Thread2 で TryReceive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            THR_CALL_TRY_RECV,          // Thread2 は TryReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest1, test_TimedSendAndTimedReceive)
{
    // Thread1 で TimedSend()、Thread2 で TimedReceive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            THR_CALL_TIMED_RECV,        // Thread2 は TimedReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

//---------------------------------------------------------------------------
// Send 待機中に Peek する（Try/Timed 含む）

TEST(CombinedMessageQueueTest2, test_SendAndPeek)
{
    // Thread1 で Send()、Thread2 で Peek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_SEND,              // Thread1 は SendMessageQueue()
            THR_CALL_PEEK,              // Thread2 は PeekMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING,          // Thread1 は待機状態
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(CombinedMessageQueueTest2, test_SendAndTryPeek)
{
    // Thread1 で Send()、Thread2 で TryPeek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_SEND,              // Thread1 は SendMessageQueue()
            THR_CALL_TRY_PEEK,          // Thread2 は TryPeekMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING,          // Thread1 は待機状態
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(CombinedMessageQueueTest2, test_SendAndTimedPeek)
{
    // Thread1 で Send()、Thread2 で TimedPeek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_SEND,              // Thread1 は SendMessageQueue()
            THR_CALL_TIMED_PEEK,        // Thread2 は TimedPeekMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING,          // Thread1 は待機状態
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(CombinedMessageQueueTest2, test_TimedSendAndPeek)
{
    // Thread1 で TimedSend()、Thread2 で Peek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            THR_CALL_PEEK,              // Thread2 は PeekMessageQueue()
            false,                      // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING,          // Thread1 はタイムアウトだけど長時間
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(CombinedMessageQueueTest2, test_TimedSendAndTryPeek)
{
    // Thread1 で TimedSend()、Thread2 で TryPeek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            THR_CALL_TRY_PEEK,          // Thread2 は TryPeekMessageQueue()
            false,                      // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING,          // Thread1 はタイムアウトだけど長時間
            THR_STATE_EXITED);          // Thread2 は終了
}

TEST(CombinedMessageQueueTest2, test_TimedSendAndTimedPeek)
{
    // Thread1 で TimedSend()、Thread2 で TimedPeek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_SEND,        // Thread1 は TimedSendMessageQueue()
            THR_CALL_TIMED_PEEK,        // Thread2 は TimedPeekMessageQueue()
            false,                      // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_WAITING,          // Thread1 はタイムアウトだけど長時間
            THR_STATE_EXITED);          // Thread2 は終了
}

//---------------------------------------------------------------------------
// Jam 待機中に Receive する（Try/Timed 含む）

TEST(CombinedMessageQueueTest3, test_JamAndReceive)
{
    // Thread1 で Jam()、Thread2 で Receive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_JAM,               // Thread1 は JamMessageQueue()
            THR_CALL_RECV,              // Thread2 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest3, test_JamAndTryReceive)
{
    // Thread1 で Jam()、Thread2 で TryReceive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_JAM,               // Thread1 は JamMessageQueue()
            THR_CALL_TRY_RECV,          // Thread2 は TryReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest3, test_JamAndTimedReceive)
{
    // Thread1 で Jam()、Thread2 で TimedReceive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_JAM,               // Thread1 は JamMessageQueue()
            THR_CALL_TIMED_RECV,        // Thread2 は TimedReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest3, test_TimedJamAndReceive)
{
    // Thread1 で TimedJam()、Thread2 で Receive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM,         // Thread1 は TimedJamMessageQueue()
            THR_CALL_RECV,              // Thread2 は ReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest3, test_TimedJamAndTryReceive)
{
    // Thread1 で TimedJam()、Thread2 で TryReceive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM,         // Thread1 は TimedJamMessageQueue()
            THR_CALL_TRY_RECV,          // Thread2 は TryReceiveMessageQueue()
            true,                       // Thread1 での API 返値
            true,                       // Thread2 での API 返値
            MQ_NUM(maxCount),           // MessageQueue は MAX 個
            THR_STATE_EXITED,           // Thread1 は終了
            THR_STATE_EXITED);          // Thread2 も終了
}

TEST(CombinedMessageQueueTest3, test_TimedJamAndTimedReceive)
{
    // Thread1 で TimedJam()、Thread2 で TimedReceive()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM,     // Thread1 は TimedJamMessageQueue()
            THR_CALL_TIMED_RECV,    // Thread2 は TimedReceiveMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

//---------------------------------------------------------------------------
// Jam 待機中に Peek する（Try/Timed 含む）

TEST(CombinedMessageQueueTest4, test_JamAndPeek)
{
    // Thread1 で Jam()、Thread2 で Peek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_WAIT_JAM,           // Thread1 は JamMessageQueue()
            THR_CALL_PEEK,          // Thread2 は PeekMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_STATE_WAITING,      // Thread1 は待機状態
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedMessageQueueTest4, test_JamAndTryPeek)
{
    // Thread1 で Jam()、Thread2 で TryPeek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_WAIT_JAM,           // Thread1 は JamMessageQueue()
            THR_CALL_TRY_PEEK,      // Thread2 は TryPeekMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_STATE_WAITING,      // Thread1 は待機状態
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedMessageQueueTest4, test_JamAndTimedPeek)
{
    // Thread1 で Jam()、Thread2 で TimedPeek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_WAIT_JAM,           // Thread1 は JamMessageQueue()
            THR_CALL_TIMED_PEEK,    // Thread2 は TimedPeekMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_STATE_WAITING,      // Thread1 は待機状態
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedMessageQueueTest4, test_TimedJamAndPeek)
{
    // Thread1 で TimedJam()、Thread2 で Peek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM,     // Thread1 は TimedJamMessageQueue()
            THR_CALL_PEEK,          // Thread2 は PeekMessageQueue()
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_STATE_WAITING,      // Thread1 はタイムアウトだけど長時間
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedMessageQueueTest4, test_TimedJamAndTryPeek)
{
    // Thread1 で TimedJam()、Thread2 で TryPeek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM,     // Thread1 は TimedJamMessageQueue()
            THR_CALL_TRY_PEEK,      // Thread2 は TryPeekMessageQueue()
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_STATE_WAITING,      // Thread1 はタイムアウトだけど長時間
            THR_STATE_EXITED);      // Thread2 は終了
}

TEST(CombinedMessageQueueTest4, test_TimedJamAndTimedPeek)
{
    // Thread1 で TimedJam()、Thread2 で TimedPeek()
    setExpectValue( 0, valueIndex0 );
    doTest2(MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_WAIT_TIMED_JAM,     // Thread1 は TimedJamMessageQueue()
            THR_CALL_TIMED_PEEK,    // Thread2 は TimedPeekMessageQueue()
            false,                  // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(maxCount),       // MessageQueue は MAX 個
            THR_STATE_WAITING,      // Thread1 はタイムアウトだけど長時間
            THR_STATE_EXITED);      // Thread2 は終了
}

//---------------------------------------------------------------------------
// Receive 待機中に Send する（Try/Timed 含む）

TEST(CombinedMessageQueueTest5, test_ReceiveAndSend)
{
    // Thread1 で Receive()、Thread2 で Send()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_RECV,          // Thread1 は ReceiveMessageQueue()
            THR_CALL_SEND,          // Thread2 は SendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest5, test_ReceiveAndTrySend)
{
    // Thread1 で Receive()、Thread2 で TrySend()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_RECV,          // Thread1 は ReceiveMessageQueue()
            THR_CALL_TRY_SEND,      // Thread2 は TrySendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest5, test_ReceiveAndTimedSend)
{
    // Thread1 で Receive()、Thread2 で TimedSend()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_RECV,          // Thread1 は ReceiveMessageQueue()
            THR_CALL_TIMED_SEND,    // Thread2 は TimedSendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest5, test_TimedReceiveAndSend)
{
    // Thread1 で TimedReceive()、Thread2 で Send()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV,    // Thread1 は TimedReceiveMessageQueue()
            THR_CALL_SEND,          // Thread2 は SendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest5, test_TimedReceiveAndTrySend)
{
    // Thread1 で TimedReceive()、Thread2 で TrySend()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV,    // Thread1 は TimedReceiveMessageQueue()
            THR_CALL_TRY_SEND,      // Thread2 は TrySendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest5, test_TimedReceiveAndTimedSend)
{
    // Thread1 で TimedReceive()、Thread2 で TimedSend()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV,    // Thread1 は TimedReceiveMessageQueue()
            THR_CALL_TIMED_SEND,    // Thread2 は TimedSendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

//---------------------------------------------------------------------------
// Receive 待機中に Jam する（Try/Timed 含む）

TEST(CombinedMessageQueueTest6, test_ReceiveAndJam)
{
    // Thread1 で Receive()、Thread2 で Jam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_RECV,          // Thread1 は ReceiveMessageQueue()
            THR_CALL_JAM,           // Thread2 は JamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest6, test_ReceiveAndTryJam)
{
    // Thread1 で Receive()、Thread2 で TryJam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_RECV,          // Thread1 は ReceiveMessageQueue()
            THR_CALL_TRY_JAM,       // Thread2 は TryJamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest6, test_ReceiveAndTimedJam)
{
    // Thread1 で Receive()、Thread2 で TimedJam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_RECV,          // Thread1 は ReceiveMessageQueue()
            THR_CALL_TIMED_JAM,     // Thread2 は TimedJamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest6, test_TimdeReceiveAndJam)
{
    // Thread1 で TimedReceive()、Thread2 で Jam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV,    // Thread1 は TimedReceiveMessageQueue()
            THR_CALL_JAM,           // Thread2 は JamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest6, test_TimdeReceiveAndTryJam)
{
    // Thread1 で TimedReceive()、Thread2 で TryJam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV,    // Thread1 は TimedReceiveMessageQueue()
            THR_CALL_TRY_JAM,       // Thread2 は TryJamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest6, test_TimedReceiveAndTimedJam)
{
    // Thread1 で TimedReceive()、Thread2 で TimedJam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_RECV,    // Thread1 は TimedReceiveMessageQueue()
            THR_CALL_TIMED_JAM,     // Thread2 は TimedJamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(0),              // MessageQueue は 0 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

//---------------------------------------------------------------------------
// Peek 待機中に Send する（Try/Timed 含む）

TEST(CombinedMessageQueueTest7, test_PeekAndSend)
{
    // Thread1 で Peek()、Thread2 で Send()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_PEEK,          // Thread1 は PeekMessageQueue()
            THR_CALL_SEND,          // Thread2 は SendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest7, test_PeekAndTrySend)
{
    // Thread1 で Peek()、Thread2 で TrySend()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_PEEK,          // Thread1 は PeekMessageQueue()
            THR_CALL_TRY_SEND,      // Thread2 は TrySendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest7, test_PeekAndTimedSend)
{
    // Thread1 で Peek()、Thread2 で TimedSend()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_PEEK,          // Thread1 は PeekMessageQueue()
            THR_CALL_TIMED_SEND,    // Thread2 は TimedSendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest7, test_TimedPeekAndSend)
{
    // Thread1 で TimedPeek()、Thread2 で Send()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK,    // Thread1 は TimedPeekMessageQueue()
            THR_CALL_SEND,          // Thread2 は SendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest7, test_TimedPeekAndTrySend)
{
    // Thread1 で TimedPeek()、Thread2 で TrySend()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK,    // Thread1 は TimedPeekMessageQueue()
            THR_CALL_TRY_SEND,      // Thread2 は TrySendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest7, test_TimedPeekAndTimedSend)
{
    // Thread1 で TimedPeek()、Thread2 で TimedSend()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK,    // Thread1 は TimedPeekMessageQueue()
            THR_CALL_TIMED_SEND,    // Thread2 は TimedSendMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

//---------------------------------------------------------------------------
// Peek 待機中に Jam する（Try/Timed 含む）

TEST(CombinedMessageQueueTest8, test_PeekAndJam)
{
    // Thread1 で Peek()、Thread2 で Jam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_PEEK,          // Thread1 は PeekMessageQueue()
            THR_CALL_JAM,           // Thread2 は JamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest8, test_PeekAndTryJam)
{
    // Thread1 で Peek()、Thread2 で TryJam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_PEEK,          // Thread1 は PeekMessageQueue()
            THR_CALL_TRY_JAM,       // Thread2 は TryJamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest8, test_PeekAndTimedJam)
{
    // Thread1 で Peek()、Thread2 で TimedJam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_PEEK,          // Thread1 は PeekMessageQueue()
            THR_CALL_TIMED_JAM,     // Thread2 は TimedJamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest8, test_TimedPeekAndJam)
{
    // Thread1 で TimedPeek()、Thread2 で Jam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK,    // Thread1 は TimedPeekMessageQueue()
            THR_CALL_JAM,           // Thread2 は JamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest8, test_TimedPeekAndTryJam)
{
    // Thread1 で TimedPeek()、Thread2 で TryJam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK,    // Thread1 は TimedPeekMessageQueue()
            THR_CALL_TRY_JAM,       // Thread2 は TryJamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

TEST(CombinedMessageQueueTest8, test_TimedPeekAndTimedJam)
{
    // Thread1 で TimedPeek()、Thread2 で TimedJam()
    setExpectValue( valueAdd, 0 );
    doTest2(MQ_NUM(0),              // MessageQueue は 0 個
            THR_WAIT_TIMED_PEEK,    // Thread1 は TimedPeekMessageQueue()
            THR_CALL_TIMED_JAM,     // Thread2 は TimedJamMessageQueue()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            MQ_NUM(1),              // MessageQueue は 1 個
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 も終了
}

//---------------------------------------------------------------------------
//  MessageQueue クラスのテスト
//  ここでは、MessageQueue クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみ行なうテストである。
//  test_MessageQueueClassTestNonBlockTrue では非ブロックな true 系テストを実施
//
TEST(MessageQueueClass, test_MessageQueueClassTestNonBlockTrue)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout     = nn::TimeSpan::FromMicroSeconds(1);
        const int           msgQueSize  = 8;
        uintptr_t           msgQueBuffer[msgQueSize];
        bool                result;

        SEQ_SET(11000);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue インスタンスの生成（最大メッセージ数：%d）\n"), msgQueSize);
        nn::os::MessageQueue   messageQueue(msgQueBuffer, msgQueSize);


        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::     Send(0x104) を発行\n"));
        messageQueue.Send(0x104);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::  TrySend(0x105) を発行"));
        result = messageQueue.TrySend(0x105);
        CheckBool( result );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedSend(0x106) を発行"));
        result = messageQueue.TimedSend(0x106, timeout);
        CheckBool( result );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::      Jam(0x103) を発行\n"));
        messageQueue.Jam(0x103);

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::   TryJam(0x102) を発行"));
        result = messageQueue.TryJam(0x102);
        CheckBool( result );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue:: TimedJam(0x101) を発行"));
        result = messageQueue.TimedJam(0x101, timeout);
        CheckBool( result );

        for (int i = 0; i < 6; i += 3)
        {
            {   // Peek(), Receive()
                const uintptr_t expect = 0x101 + i;
                uintptr_t       data;

                SEQ_NONE();
                NNT_OS_LOG(NN_TEXT("main: MessageQueue::Peek() の発行        （expect=0x%03x）"), expect);
                messageQueue.Peek( &data );
                CheckBool( expect == data );

                SEQ_NONE();
                NNT_OS_LOG(NN_TEXT("main: MessageQueue::Receive() の発行     （expect=0x%03x）"), expect);
                messageQueue.Receive( &data );
                CheckBool( expect == data );
            }

            {   // TryPeek(), TryReceive()
                const uintptr_t expect = 0x102 + i;
                uintptr_t       data;

                SEQ_NONE();
                NNT_OS_LOG(NN_TEXT("main: MessageQueue::TryPeek() の発行     （expect=0x%03x）"), expect);
                EXPECT_TRUE( messageQueue.TryPeek( &data ) );
                CheckBool( expect == data );

                SEQ_NONE();
                NNT_OS_LOG(NN_TEXT("main: MessageQueue::TryReceive() の発行  （expect=0x%03x）"), expect);
                EXPECT_TRUE( messageQueue.TryReceive( &data ) );
                CheckBool( expect == data );
            }

            {   // TimedPeek(), TimedReceive()
                const uintptr_t expect = 0x103 + i;
                uintptr_t       data;

                SEQ_NONE();
                NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedPeek() の発行   （expect=0x%03x）"), expect);
                EXPECT_TRUE( messageQueue.TimedPeek( &data, timeout ) );
                CheckBool( expect == data );

                SEQ_NONE();
                NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedReceive() の発行（expect=0x%03x）"), expect);
                EXPECT_TRUE( messageQueue.TimedReceive( &data, timeout ) );
                CheckBool( expect == data );
            }
        }
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  MessageQueue クラスのテスト
//  ここでは、MessageQueue クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみ行なうテストである。
//  test_MessageQueueClassTestNonBlockFalse では非ブロックな false 系テストを実施
//
TEST(MessageQueueClass, test_MessageQueueClassTestNonBlockFalse)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout     = nn::TimeSpan::FromMicroSeconds(1);
        const int           msgQueSize  = 1;
        uintptr_t           msgQueBuffer[msgQueSize];
        uintptr_t           data;

        SEQ_SET(11100);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue インスタンスの生成（最大メッセージ数：%d）\n"), msgQueSize);
        nn::os::MessageQueue   messageQueue(msgQueBuffer, msgQueSize);


        // まず、メッセージキューを満杯にする
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TrySend(0x200) で true"));
        CheckBool( messageQueue.TrySend(0x200) == true );

        // ここからが送信テスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TrySend(0xFFF) で false"));
        CheckBool( messageQueue.TrySend(0xfff) == false );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedSend(0xFFF) で false"));
        CheckBool( messageQueue.TimedSend(0xfff, timeout) == false );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TryJam(0xFFF) で false"));
        CheckBool( messageQueue.TryJam(0xfff) == false );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedJam(0xFFF) で false"));
        CheckBool( messageQueue.TimedJam(0xfff, timeout) == false );

        // 続いて、メッセージキューを空にする
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TryReceive() で true"));
        CheckBool( messageQueue.TryReceive(&data) == true );
        EXPECT_TRUE( data == 0x200 );

        // ここからが受信テスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TryReceive() で false"));
        CheckBool( messageQueue.TryReceive(&data) == false );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedReceive() で false"));
        CheckBool( messageQueue.TimedReceive(&data, timeout) == false );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TryPeek() で false"));
        CheckBool( messageQueue.TryPeek(&data) == false );

        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedPeek() で false"));
        CheckBool( messageQueue.TimedPeek(&data, timeout) == false );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  MessageQueue クラスのテスト
//  ここでは、MessageQueue クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみ行なうテストである。
//  test_MessageQueueClassTestBlock ではブロック系テストを２スレッドを使って実施
//
NN_ALIGNAS( 4096 ) char g_StackOfThread[ 8192 ];

void MessageQueueClassTestThread(void* arg)
{
    auto& messageQueue = *static_cast<nn::os::MessageQueue*>(arg);
    const nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(100);
    uintptr_t          data;

    for (int i = 0; i < 4; ++i)
    {
        // Step. 1-4
        nn::os::SleepThread( interval );

        SEQ_CHECK(11202 + i * 3);
        NNT_OS_LOG(NN_TEXT("sub:  MessageQueue::Send(0x%03x) でデータ送信\n"), 0x300 + i);
        messageQueue.Send(0x300 + i);
    }

    for (int i = 0; i < 4; ++i)
    {
        // Step. 5-8
        nn::os::SleepThread( interval );

        SEQ_CHECK(11214 + i * 3);
        NNT_OS_LOG(NN_TEXT("sub:  MessageQueue::Receive() でデータ受信\n"));
        messageQueue.Receive(&data);
    }
}

TEST(MessageQueueClass, test_MessageQueueClassTestBlock)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        const nn::TimeSpan  timeout     = nn::TimeSpan::FromSeconds(1);
        const int           msgQueSize  = 1;
        uintptr_t           msgQueBuffer[msgQueSize];
        uintptr_t           data;

        SEQ_SET(11200);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue インスタンスの生成（最大メッセージ数：%d）"), msgQueSize);
        nn::os::MessageQueue   messageQueue(msgQueBuffer, msgQueSize);

        // テスト用の子スレッドを作成
        nn::os::ThreadType  thread;
        {
            auto pri    = nn::os::GetThreadPriority(nn::os::GetCurrentThread());
            auto result = nn::os::CreateThread( &thread,
                                                &MessageQueueClassTestThread,
                                                &messageQueue,
                                                g_StackOfThread,
                                                sizeof(g_StackOfThread),
                                                pri + 1);
            CheckBool( result.IsSuccess() );
        }

        // 子スレッドを起動
        nn::os::StartThread( &thread );

        // Step1: Peek で待機して起床時の参照データを確認
        SEQ_CHECK(11201);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::Peek() で待機\n"));
        messageQueue.Peek(&data);
        SEQ_CHECK(11203);
        NNT_OS_LOG(NN_TEXT("main: -> 起床（data=0x%03x）"), data);
        CheckBool( data == 0x300 );

        // TryReceive でデータを捨てる
        messageQueue.TryReceive(&data);

        // Step2: TimedPeek で待機して起床時の参照データを確認
        SEQ_CHECK(11204);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedPeek() で待機\n"));
        messageQueue.TimedPeek(&data, timeout);
        SEQ_CHECK(11206);
        NNT_OS_LOG(NN_TEXT("main: -> 起床（data=0x%03x）"), data);
        CheckBool( data == 0x301 );

        // TryReceive でデータを捨てる
        messageQueue.TryReceive(&data);

        // Step3: Receive で待機して起床時の受信データを確認
        SEQ_CHECK(11207);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::Receive() で待機\n"));
        messageQueue.Receive(&data);
        SEQ_CHECK(11209);
        NNT_OS_LOG(NN_TEXT("main: -> 起床（data=0x%03x）"), data);
        CheckBool( data == 0x302 );

        // Step4: TimedReceive で待機して起床時の受信データを確認
        SEQ_CHECK(11210);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedReceive() で待機\n"));
        messageQueue.TimedReceive(&data, timeout);
        SEQ_CHECK(11212);
        NNT_OS_LOG(NN_TEXT("main: -> 起床（data=0x%03x）"), data);
        CheckBool( data == 0x303 );

        // データを１つ送信して満杯にする
        messageQueue.Send(0xFFF);

        // Step5: Send で待機して起床
        SEQ_CHECK(11213);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::Send(0x304) で待機\n"));
        messageQueue.Send(0x304);
        SEQ_CHECK(11215);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // Step6: TimedSend で待機して起床
        SEQ_CHECK(11216);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedSend(0x305) で待機\n"));
        messageQueue.TimedSend(0x305, timeout);
        SEQ_CHECK(11218);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // Step7: Jam で待機して起床
        SEQ_CHECK(11219);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::Jam(0x306) で待機\n"));
        messageQueue.Jam(0x306);
        SEQ_CHECK(11221);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // Step8: TimedJam で待機して起床
        SEQ_CHECK(11222);
        NNT_OS_LOG(NN_TEXT("main: MessageQueue::TimedJam(0x307) で待機\n"));
        messageQueue.TimedJam(0x307, timeout);
        SEQ_CHECK(11224);
        NNT_OS_LOG(NN_TEXT("main: -> 起床\n"));

        // 子スレッドの終了待ち
        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  MessageQueue クラスの型変換関数のテスト
//  ここでは、MessageQueue クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator MessageQueueType&() の動作テストを行なう。
//
TEST(MessageQueueClass, test_MessageQueueClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_SET(11300);
        NNT_OS_LOG(NN_TEXT("messageQueue インスタンスの生成\n"));

        const int   msgQueSize  = 1;
        uintptr_t   msgQueBuffer[msgQueSize];
        nn::os::MessageQueue messageQueue(msgQueBuffer, msgQueSize);
        uintptr_t   data;

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("messageQueue.GetBase() で MessageQueueType オブジェクトを取得"));
        nn::os::MessageQueueType*  messageQueueType = messageQueue.GetBase();
        CheckBool( messageQueueType != NULL );

        // 初期バッファ状態が空であることを確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("messageQueue.TryPeek() でバッファが空であることを確認"));
        CheckBool( messageQueue.TryPeek(&data) == false );

        // MessageQueueType にデータを送信し、TryPeek で true を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("messageQueueType にデータを送信"));
        nn::os::SendMessageQueue(messageQueueType, 0x5555);
        CheckBool( messageQueue.TryPeek(&data) == true );

        // MessageQueueType からデータを受信し、TryPeek で false を確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("messageQueueType からデータを受信"));
        nn::os::ReceiveMessageQueue(&data, messageQueueType);
        CheckBool( messageQueue.TryPeek(&data) == false );

        // operator MessageQueueType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator MessageQueueType&() を使ってオブジェクトの参照を取得"));
        nn::os::MessageQueueType& messageQueueTypeRefer = messageQueue;
        CheckBool( &messageQueueTypeRefer == messageQueueType );

        // operator const MessageQueueType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const MessageQueueType&() を使ってオブジェクトの参照を取得"));
        const nn::os::MessageQueueType& messageQueueTypeReferConst = messageQueue;
        CheckBool( &messageQueueTypeReferConst == messageQueueType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

}}} // namespace nnt::os::messageQueue

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of MessageQueue APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of MessageQueue APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
