﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
 * Copyright (c) 2007-2016 NVIDIA Corporation.  All Rights Reserved.
 *
 * NVIDIA Corporation and its licensors retain all intellectual property and
 * proprietary rights in and to this software and related documentation.  Any
 * use, reproduction, disclosure or distribution of this software and related
 * documentation without an express license agreement from NVIDIA Corporation
 * is strictly prohibited.
 */

/*
 * Test the following basic nvrm functionalities:
 * - memory allocation and mapping
 * - memory read/write
 * - access module capabilities
 * - NvRm open and close
 * - clock & voltage control of a module
 * - RTC
 */
#include <cstdarg>
#include <cstring>

#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/tma/tma.h>

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
// For declaration of 'nv::InitializeGraphics' and 'nv::SetGraphicsAllocator'
#include <nv/nv_MemoryManagement.h>
#endif

#include "nvassert.h"
#include "nvcommon.h"
#include "nverror.h"
#include "nvos.h"
#include "nvrm_channel.h"
#include "nvrm_drf.h"
#include "nvrm_init.h"
#include "nvrm_memmgr.h"
#include "nvrm_module.h"
#include "nvrm_surface.h"
#include "nvtest.h"

using namespace nn;

const size_t HeapSize = 256 * 1024 * 1024;

void
TestThunkTime(NvRmDeviceHandle rm)
{
    NvU32 StartTime;
    NvU32 EndTime;
    NvError Status;
    NvRmMemHandle hMem = 0;
    int i;
    NVRM_DEFINE_MEM_HANDLE_ATTR(NvRmMemAttr);

    NVRM_MEM_HANDLE_SET_ATTR(NvRmMemAttr, 4, NvOsMemAttribute_WriteCombined, 100, 0);
    NVRM_MEM_HANDLE_SET_HEAP_ATTR(NvRmMemAttr, NULL, 0);
    Status = NvRmMemHandleAllocAttr(rm, &NvRmMemAttr, &hMem);
    if (Status != NvSuccess)
    {
        NN_LOG("NvRmMemHandleAllocAttr() failed.\n");
        goto clean;
    }

    StartTime = NvOsGetTimeMS();
    for (i=0; i < 10000; ++i)
    {
        NvRmMemGetSize(hMem);
    }
    EndTime = NvOsGetTimeMS();

    NN_LOG("##teamcity[buildStatisticValue key='ms_for_%d_IPC_calls' value='%d']\n", i, EndTime - StartTime);

clean:
    NvRmMemHandleFree(hMem);

}

NvError
NvTestMain( int argc, char *argv[] )
{
    NvError err;
    NvRmDeviceHandle rm;

    // open the Rm
    err = NvRmOpenNew(&rm);
    if (err != NvSuccess)
    {
        NN_LOG("NvRmOpenNew() failed.\n");
        return err;
    }

    // run the tests
    TestThunkTime( rm );
    NvRmClose( rm );
    return err;
}

namespace{

    const int FsHeapSize = 512 * 1024;

    uint8_t              g_FsHeapBuffer[FsHeapSize];
    nn::lmem::HeapHandle g_FsHeap;

    void FsInitHeap()
    {
        g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
    }

    void* FsAllocate(size_t size)
    {
        return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
    }

    void FsDeallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        return nn::lmem::FreeToExpHeap(g_FsHeap, p);
    }

}

extern "C" void nninitStartup()
{
    const size_t MallocMemorySize = 16 * 1024 * 1024;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(HeapSize);
    if (!result.IsSuccess())
    {
        NN_LOG("SetMemoryHeapSize(%d) failed.\n", HeapSize);
        return;
    }

    result = nn::os::AllocateMemoryBlock(&address, MallocMemorySize);
    NN_ASSERT(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void *>(address), MallocMemorySize);

    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#if defined( NN_BUILD_CONFIG_SPEC_NX ) || NN_GFX_IS_TARGET_NVN
//!--------------------------------------------------------------------------------------
//! @brief 共通アロケータ
//!--------------------------------------------------------------------------------------
static void* Allocate( size_t size, size_t alignment, void* userPtr )
{
    NN_UNUSED( userPtr );
    return aligned_alloc( alignment, size );
}

//!--------------------------------------------------------------------------------------
//! @brief 共通デアロケータ
//!--------------------------------------------------------------------------------------
static void Deallocate( void* addr, void* userPtr )
{
    NN_UNUSED( userPtr );
    free( addr );
}

//!--------------------------------------------------------------------------------------
//! @brief 共通リアロケータ
//!--------------------------------------------------------------------------------------
static void* Reallocate( void* addr, size_t newSize, void* userPtr )
{
    NN_UNUSED( userPtr );
    return realloc( addr, newSize );
}
#endif
#endif

//!--------------------------------------------------------------------------------------
//! @brief ペリフェラルセットアップ
//!--------------------------------------------------------------------------------------
static void SetupPeripherals() NN_NOEXCEPT
{
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#if defined( NN_BUILD_CONFIG_SPEC_NX )
    static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ
    // this memory allocation would be used from the nvn graphics systems at runtime.
    nv::SetGraphicsAllocator( Allocate, Deallocate, Reallocate, nullptr );
    nv::InitializeGraphics( ::malloc( GraphicsSystemReservedMemorySize ), GraphicsSystemReservedMemorySize );
#endif
#if NN_GFX_IS_TARGET_NVN
    // this memory allocation interface would be used when compiling of shader code at runtime.
    glslcSetAllocator( Allocate, Deallocate, Reallocate, nullptr );
#endif
#endif
}

extern "C" void nnMain()
{
    tma::Initialize();
    SetupPeripherals();
    NvTestMain(0, NULL);
    tma::Finalize();
}
