﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  ConditionVariable 関連機能のテスト
//---------------------------------------------------------------------------

#include "../Common/test_Pragma.h"

#include <mutex>
#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include "../Common/test_Helper.h"
#include "../Common/test_Calibration.h"
#include "test_ConditionVariableHelper.h"

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>

namespace nnt { namespace os { namespace conditionVariable {

NN_OS_ALIGNAS_THREAD_STACK uint8_t  g_ThreadStack[8192];

//---------------------------------------------------------------------------
// 単独の WaitConditionVariable() テスト

TEST(WaitConditionVariable, test_WaitConditionVariable1)
{
    // WaitConditionVariable（Mutex 獲得なし）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,              // Mutex 初期ロック状態は false
                THR_CALL_WAIT,      // Thread1 は WaitConditionVariable()
                false,              // Thread1 での API 返値
                THR_STATE_EXITED)   // Thread1 は終了
    , "");
}

TEST(WaitConditionVariable, test_WaitConditionVariable2)
{
    // WaitConditionVariable（自スレッドで Mutex 獲得済み）
    doTest1(true,                   // Mutex 初期ロック状態は true
            THR_WAIT_WAIT,          // Thread1 は WaitConditionVariable()
            true,                   // Thread1 での API 返値
            THR_STATE_WAITING);     // Thread1 は待ち状態
}

TEST(WaitConditionVariable, test_WaitConditionVariable3)
{
    // WaitConditionVariable（他スレッドで Mutex 獲得済み）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(false,              // Mutex 初期ロック状態は false
                THR_CALL_TRY_LOCK,  // Thread1 は TryLockMutex()
                THR_CALL_WAIT,      // Thread2 は WaitConditionVariable()
                true,               // Thread1 での API 返値
                false,              // Thread2 での API 返値
                THR_STATE_WAITING2, // Thread1 は Thread2 終了処理待ち
                THR_STATE_EXITED)   // Thread2 は終了
    , "");
}

TEST(WaitConditionVariable, test_WaitConditionVariable4)
{
    // WaitConditionVariable（他スレッドで LockMutex->WaitCond 済み）
    // ロックは解放されているため、Thread2 の WaitCond でエラー
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,               // Mutex 初期ロック状態は true
                THR_WAIT_WAIT,      // Thread1 は WaitConditionVariable()
                THR_CALL_WAIT,      // Thread2 は WaitConditionVariable()
                true,               // Thread1 での API 返値
                false,              // Thread2 での API 返値
                THR_STATE_EXITED,   // Thread1 は待ち状態
                THR_STATE_EXITED)   // Thread2 は終了
    , "");
}

//---------------------------------------------------------------------------
// 単独の TimedWaitConditionVariable() テスト

TEST(TimedWaitConditionVariable, test_TimedWaitConditionVariable1)
{
    // TimedWaitConditionVariable（Mutex 獲得なし）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,              // Mutex 初期ロック状態は false
                THR_CALL_TIMED_WAIT,// Thread1 は TimedWaitConditionVariable()
                false,              // Thread1 での API 返値
                THR_STATE_EXITED)   // Thread1 は終了
    , "");
}

TEST(TimedWaitConditionVariable, test_TimedWaitConditionVariable2)
{
    // TimedWaitConditionVariable（自スレッドで Mutex 獲得済み）
    doTest1(true,                   // Mutex 初期ロック状態は true
            THR_WAIT_TIMED_WAIT,    // Thread1 は TimedWaitConditionVariable()
            false,                  // Thread1 での API 返値
            THR_STATE_TIMEDOUT);    // Thread1 は待ち状態
}

TEST(TimedWaitConditionVariable, test_TimedWaitConditionVariable3)
{
    // TimedWaitConditionVariable（他スレッドで Mutex 獲得済み）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(false,              // Mutex 初期ロック状態は false
                THR_CALL_TRY_LOCK,  // Thread1 は TryLockMutex()
                THR_CALL_TIMED_WAIT,// Thread2 は TimedWaitConditionVariable()
                true,               // Thread1 での API 返値
                false,              // Thread2 での API 返値
                THR_STATE_WAITING2, // Thread1 は Thread2 終了処理待ち
                THR_STATE_EXITED)   // Thread2 は終了
    , "");
}

TEST(TimedWaitConditionVariable, test_TimedWaitConditionVariable4)
{
    // TimedWaitConditionVariable（他スレッドで LockMutex->TimedWaitCond 済み）
    // ロックは解放されているため、Thread2 の TimedWaitCond でエラー
    EXPECT_DEATH_IF_SUPPORTED(
        doTest2(true,               // Mutex 初期ロック状態は true
                THR_WAIT_TIMED_WAIT,// Thread1 は TimedWaitConditionVariable()
                THR_CALL_TIMED_WAIT,// Thread2 は TimedWaitConditionVariable()
                true,               // Thread1 での API 返値
                false,              // Thread2 での API 返値
                THR_STATE_EXITED,   // Thread1 は待ち状態
                THR_STATE_EXITED)   // Thread2 は終了
    , "");
}

// 負のタイムアウト値を許容することになったときのためにテストを残しておく
#if 0
//---------------------------------------------------------------------------
// 単独の TimedWaitConditionVariable() テスト（負のタイムアウト値）

TEST(TimedWaitConditionVariableWithMinusTimeout, test_TimedWaitConditionVariableWithMinusTimeout1)
{
    // TimedWaitConditionVariable（Mutex 獲得なし）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(false,                     // Mutex 初期ロック状態は false
                THR_CALL_TIMED_WAIT_MINUS, // Thread1 は TimedWaitConditionVariable()
                false,                     // Thread1 での API 返値
                THR_STATE_EXITED)          // Thread1 は終了
    , "");
}

TEST(TimedWaitConditionVariableWithMinusTimeout, test_TimedWaitConditionVariableWithMinusTimeout2)
{
    // TimedWaitConditionVariable（自スレッドで Mutex 獲得済み）
    doTest1(true,                         // Mutex 初期ロック状態は true
            THR_WAIT_TIMED_WAIT_MINUS,    // Thread1 は TimedWaitConditionVariable()
            false,                        // Thread1 での API 返値
            THR_STATE_TIMEDOUT);          // Thread1 は待ち状態
}

#else
// 負のタイムアウト値を許容しない場合の DEATH TEST

TEST(TimedWaitConditionVariableWithMinusTimeoutDeathTest, test_TimedWaitConditionVariableWithMinusTimeoutDeathTest)
{
    // TimedWaitConditionVariable（自スレッドで Mutex 獲得済み）
    EXPECT_DEATH_IF_SUPPORTED(
        doTest1(true,                         // Mutex 初期ロック状態は true
                THR_WAIT_TIMED_WAIT_MINUS,    // Thread1 は TimedWaitConditionVariable()
                false,                        // Thread1 での API 返値
                THR_STATE_TIMEDOUT)           // Thread1 は待ち状態
    , "");
}

#endif

//---------------------------------------------------------------------------
// 単独の SignalConditionVariable() テスト

TEST(SignalConditionVariable, test_SignalConditionVariable1)
{
    // SignalConditionVariable（単独）
    doTest1(false,                  // Mutex 初期ロック状態は false
            THR_CALL_SIGNAL,        // Thread1 は SignalConditionVariable()
            true,                   // Thread1 での API 返値
            THR_STATE_EXITED);      // Thread1 は終了
}

TEST(SignalConditionVariable, test_SignalConditionVariable2)
{
    // SignalConditionVariable（signal 済みのところへ再度 signal）
    doTest2(false,                  // Mutex 初期ロック状態は false
            THR_CALL_SIGNAL,        // Thread1 は SignalConditionVariable()
            THR_CALL_SIGNAL,        // Thread2 は SignalConditionVariable()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}

//---------------------------------------------------------------------------
// 単独の BroadcastConditionVariable() テスト

TEST(BroadcastConditionVariable, test_BroadcastConditionVariable1)
{
    // BroadcastConditionVariable（単独）
    doTest1(false,                  // Mutex 初期ロック状態は false
            THR_CALL_BROAD,         // Thread1 は BroadcastConditionVariable()
            true,                   // Thread1 での API 返値
            THR_STATE_EXITED);      // Thread1 は終了
}

TEST(BroadcastConditionVariable, test_BroadcastConditionVariable2)
{
    // BroadcastConditionVariable（signal 済みのところへ再度 signal）
    doTest2(false,                  // Mutex 初期ロック状態は false
            THR_CALL_BROAD,         // Thread1 は BroadcastConditionVariable()
            THR_CALL_BROAD,         // Thread2 は BroadcastConditionVariable()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は終了
            THR_STATE_EXITED);      // Thread2 は終了
}


//---------------------------------------------------------------------------
// WaitCond/TimedWaitCond 中に他スレッドから Signal 発行

TEST(CombinedConditionVariable1, test_WaitAndSignal)
{
    // Thread1 で LockMutex->WaitCond、Thread2 で Signal 発行
    doTest2(true,                   // Mutex 初期ロック状態は true
            THR_WAIT_WAIT,          // Thread1 は WaitConditionVariable()
            THR_CALL_SIGNAL,        // Thread2 は SignalConditionVariable()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は待ち状態
            THR_STATE_EXITED);      // Thread2 は終了
}


TEST(CombinedConditionVariable1, test_TimedWaitAndSignal)
{
    // Thread1 で LockMutex->WaitCond、Thread2 で Signal 発行
    doTest2(true,                   // Mutex 初期ロック状態は true
            THR_WAIT_TIMED_WAIT,    // Thread1 は TimedWaitConditionVariable()
            THR_CALL_SIGNAL,        // Thread2 は SignalConditionVariable()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は待ち状態
            THR_STATE_EXITED);      // Thread2 は終了
}


//---------------------------------------------------------------------------
// WaitCond/TimedWaitCond 中に他スレッドから Broadcast 発行

TEST(CombinedConditionVariable2, test_WaitAndBroadcast)
{
    // Thread1 で LockMutex->WaitCond、Thread2 で BroadCast 発行
    doTest2(true,                   // Mutex 初期ロック状態は true
            THR_WAIT_WAIT,          // Thread1 は WaitConditionVariable()
            THR_CALL_BROAD,         // Thread2 は BroadCastConditionVariable()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は待ち状態
            THR_STATE_EXITED);      // Thread2 は終了
}


TEST(CombinedConditionVariable2, test_TimedWaitAndBroadcast)
{
    // Thread1 で LockMutex->WaitCond、Thread2 で BroadCast 発行
    doTest2(true,                   // Mutex 初期ロック状態は true
            THR_WAIT_TIMED_WAIT,    // Thread1 は TimedWaitConditionVariable()
            THR_CALL_BROAD,         // Thread2 は BroadcastConditionVariable()
            true,                   // Thread1 での API 返値
            true,                   // Thread2 での API 返値
            THR_STATE_EXITED,       // Thread1 は待ち状態
            THR_STATE_EXITED);      // Thread2 は終了
}


//---------------------------------------------------------------------------
// Recursive Mutex を一度だけロックして WaitConditionVariable するテスト
//
nn::os::MutexType               g_Mutex;
nn::os::ConditionVariableType   g_Cond;

void test_WaitCondVarForRecursiveMutexTestThread(void* arg)
{
    NN_UNUSED(arg);

    nn::os::LockMutex(&g_Mutex);
    nn::os::SignalConditionVariable(&g_Cond);
    nn::os::UnlockMutex(&g_Mutex);
}


TEST(WaitConditionVariableTest, test_RecursiveMutexLockOnce)
{
    CLEAR_GOOGLE_TEST();

    // Recursive Mutex として初期化
    nn::os::InitializeMutex(&g_Mutex, true, 0);
    nn::os::InitializeConditionVariable(&g_Cond);

    // スレッドの作成
    nn::os::ThreadType  thread;
    auto result = nn::os::CreateThread( &thread, test_WaitCondVarForRecursiveMutexTestThread, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));
    EXPECT_TRUE( result.IsSuccess() );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("Recursive Mutex を一度だけロックして WaitCondVar() するテスト"));

    // シングルロック（WaitConditionVariable）
    nn::os::LockMutex(&g_Mutex);
    nn::os::StartThread(&thread);
    nn::os::WaitConditionVariable(&g_Cond, &g_Mutex);
    nn::os::UnlockMutex(&g_Mutex);

    // スレッド終了
    nn::os::WaitThread(&thread);
    nn::os::DestroyThread(&thread);

    // シングルロック（TimedWaitConditionVariable）
    nn::os::LockMutex(&g_Mutex);
    auto cvStatus = nn::os::TimedWaitConditionVariable(&g_Cond, &g_Mutex, nn::TimeSpan::FromMilliSeconds(1));
    nn::os::UnlockMutex(&g_Mutex);
    EXPECT_TRUE( cvStatus == nn::os::ConditionVariableStatus_Timeout );

    nn::os::FinalizeMutex(&g_Mutex);
    nn::os::FinalizeConditionVariable(&g_Cond);

    NNT_OS_LOG(" ... OK\n");
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// Recursive Mutex を多重ロックして WaitConditionVariable する DEATH テスト
//
TEST(WaitConditionVariableTest, test_RecursiveMutexLockTwice)
{
    CLEAR_GOOGLE_TEST();

    // Recursive Mutex として初期化
    nn::os::InitializeMutex(&g_Mutex, true, 0);
    nn::os::InitializeConditionVariable(&g_Cond);

    // 多重ロック（WaitConditionVariable）
    nn::os::LockMutex(&g_Mutex);
    nn::os::LockMutex(&g_Mutex);
    EXPECT_DEATH_IF_SUPPORTED( nn::os::WaitConditionVariable(&g_Cond, &g_Mutex), "" );
    nn::os::UnlockMutex(&g_Mutex);
    nn::os::UnlockMutex(&g_Mutex);

    // 多重ロック（TimedWaitConditionVariable）
    nn::os::LockMutex(&g_Mutex);
    nn::os::LockMutex(&g_Mutex);
    EXPECT_DEATH_IF_SUPPORTED( nn::os::TimedWaitConditionVariable(&g_Cond, &g_Mutex, nn::TimeSpan::FromMilliSeconds(1)), "" );
    nn::os::UnlockMutex(&g_Mutex);
    nn::os::UnlockMutex(&g_Mutex);

    nn::os::FinalizeMutex(&g_Mutex);
    nn::os::FinalizeConditionVariable(&g_Cond);

    JUDGE_GOOGLE_TEST();
}


//---------------------------------------------------------------------------
// nn::os::ConditionVariable クラスのテスト
//

class CVTestClass
{
public:
    CVTestClass() : m_WakeupParent(false), m_WakeupChild(false), m_Mutex(false), m_ParentThreadCounter(0), m_ChildThreadCounter(0) {}

    // for Child
    void WaitForParent()
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        while (!m_WakeupChild)
        {
            m_CondChild.Wait(m_Mutex);
        }
        m_WakeupChild = false;
    }

    void SignalToParent()
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_WakeupParent = true;
        m_CondParent.Signal();
    }

    // for Parent
    bool TimedWaitForChild(nn::TimeSpan timeout)
    {
        auto tick = nn::os::GetSystemTick() + nn::os::ConvertToTick(timeout);

        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        while (!m_WakeupParent)
        {
            if (nn::os::GetSystemTick() > tick)
            {
                // Timeout
                return false;
            }
            m_CondParent.TimedWait(m_Mutex, timeout);
        }
        m_WakeupParent = false;
        return true;
    }

    void BroadcastToChild()
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        m_WakeupChild = true;
        m_CondChild.Broadcast();
    }

    // あえてロックしない
    void IncrementParentThreadCounter()
    {
        ++m_ParentThreadCounter;
    }
    void IncrementChildThreadCounter()
    {
        ++m_ChildThreadCounter;
    }
    int  GetParentThreadCounter() const
    {
        return m_ParentThreadCounter;
    }
    int  GetChildThreadCounter()  const
    {
        return m_ChildThreadCounter;
    }

private:
    bool                        m_WakeupParent;
    bool                        m_WakeupChild;
    nn::os::Mutex               m_Mutex;
    nn::os::ConditionVariable   m_CondParent;
    nn::os::ConditionVariable   m_CondChild;
    int                         m_ParentThreadCounter;
    int                         m_ChildThreadCounter;
};


void test_ConditionVariableClassTestThread(void* arg)
{
    CVTestClass& g_CvTestClass = *reinterpret_cast<CVTestClass*>(arg);

    for (int i=0; i<50000; ++i)
    {
        g_CvTestClass.IncrementChildThreadCounter();

        g_CvTestClass.SignalToParent();

        g_CvTestClass.WaitForParent();
    }
}

TEST(ConditionVariableClassTest, test_ConditionVariableClass)
{
    CLEAR_GOOGLE_TEST();

    CVTestClass g_CvTestClass;

    // スレッドの作成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, test_ConditionVariableClassTestThread, &g_CvTestClass, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("ConditionVariable クラスのテスト\n"));
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("親スレッドと子スレッドで同じ条件変数インスタンスを制御\n"));

    // テスト本体（スレッド実行開始）
    nn::os::StartThread( &thread );
    for (int i=0; i<50000; ++i)
    {
        g_CvTestClass.TimedWaitForChild( nn::TimeSpan::FromSeconds(10) );

        g_CvTestClass.IncrementParentThreadCounter();

        // parent と child が必ず交互にそれぞれのカウンタをインクリメント
        // するはずなので、毎ループ毎にカウンタ値をチェックする。
        int parentCounter = g_CvTestClass.GetParentThreadCounter();
        int childCounter  = g_CvTestClass.GetChildThreadCounter();
        if (parentCounter != childCounter)
        {
            SEQ_NONE();
            NNT_OS_LOG(NN_TEXT("条件変数テストのループで同期の確立に失敗しました。\n"));
            FAIL();
        }

        g_CvTestClass.BroadcastToChild();
    }

    // すぐにタイムアウトで返ってくる
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("TimeWait() でタイムアウトが成立する"));
    bool ret = g_CvTestClass.TimedWaitForChild( nn::TimeSpan::FromMilliSeconds(1) );
    CheckBool(ret == false);

    // スレッドの終了待ち
    nn::os::WaitThread( &thread );

    int parentCounter = g_CvTestClass.GetParentThreadCounter();
    int childCounter  = g_CvTestClass.GetChildThreadCounter();

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("親スレッドのカウンタが期待値通りか counter=%d"), parentCounter);
    CheckBool( parentCounter == 50000 );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("子スレッドのカウンタが期待値通りか counter=%d"), childCounter);
    CheckBool( childCounter  == 50000 );

    // スレッドの破棄
    nn::os::DestroyThread( &thread );

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
// nn::os::ConditionVariableType 構造体の静的初期化のテスト
//

nn::os::MutexType             g_StaticMutex    = NN_OS_MUTEX_INITIALIZER(false);
nn::os::ConditionVariableType g_StaticCondVar1 = NN_OS_CONDITION_VARIABLE_INITIALIZER();
nn::os::ConditionVariableType g_StaticCondVar2 = NN_OS_CONDITION_VARIABLE_INITIALIZER();
int     g_CondVarSeqCounter;

inline void CheckAndIncrementCondVarCounter(int expect)
{
    EXPECT_TRUE( g_CondVarSeqCounter == expect );
    ++g_CondVarSeqCounter;
}

void test_StaticConditionVariableTestThread(void* arg)
{
    NN_UNUSED(arg);

    nn::os::LockMutex( &g_StaticMutex );
    {
        CheckAndIncrementCondVarCounter(2);

        nn::os::SignalConditionVariable( &g_StaticCondVar1 );

        nn::os::WaitConditionVariable( &g_StaticCondVar2, &g_StaticMutex );

        CheckAndIncrementCondVarCounter(5);

        nn::os::BroadcastConditionVariable( &g_StaticCondVar1 );
    }
    nn::os::UnlockMutex( &g_StaticMutex );
}

TEST(ConditionVariableTypeStaticInitializerTest, test_ConditionVariableTypeStaticInitializer)
{
    g_CondVarSeqCounter = 0;

    CLEAR_GOOGLE_TEST();

    // 静的初期化された ConditionVariableType オブジェクトのダンプ
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticCondVar1 の内部状態（チェックなし）\n"));
    for (size_t i=0; i<sizeof(nn::os::ConditionVariableType) / sizeof(uint32_t); ++i)
    {
        SEQ_NONE();
        NNT_OS_LOG("g_StaticCondVar1[%d]= 0x%08x\n", i, ((uint32_t*)&g_StaticCondVar1)[i]);
    }

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticCondVar2 の内部状態（チェックなし）\n"));
    for (size_t i=0; i<sizeof(nn::os::ConditionVariableType) / sizeof(uint32_t); ++i)
    {
        SEQ_NONE();
        NNT_OS_LOG("g_StaticCondVar2[%d]= 0x%08x\n", i, ((uint32_t*)&g_StaticCondVar2)[i]);
    }

    // スレッドの作成
    nn::os::ThreadType  thread;
    nn::os::CreateThread( &thread, test_StaticConditionVariableTestThread, NULL, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));

    // 静的初期化された ConditionVariableType オブジェクトを利用するテスト
    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticCondVar が Signal されるのを待機\n"));

    nn::os::LockMutex( &g_StaticMutex );
    {
        CheckAndIncrementCondVarCounter(0);

        nn::os::StartThread( &thread );

        CheckAndIncrementCondVarCounter(1);

        nn::os::WaitConditionVariable( &g_StaticCondVar1, &g_StaticMutex );
    }
    nn::os::UnlockMutex( &g_StaticMutex );


    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された g_StaticCondVar が Broadcast されるのを待機\n"));

    nn::os::LockMutex( &g_StaticMutex );
    {
        CheckAndIncrementCondVarCounter(3);

        nn::os::SignalConditionVariable( &g_StaticCondVar2 );

        CheckAndIncrementCondVarCounter(4);

        nn::os::WaitConditionVariable( &g_StaticCondVar1, &g_StaticMutex );

        CheckAndIncrementCondVarCounter(6);
    }
    nn::os::UnlockMutex( &g_StaticMutex );

    nn::os::WaitThread( &thread );
    nn::os::DestroyThread( &thread );

    SEQ_NONE();
    NNT_OS_LOG(NN_TEXT("静的初期化された ConditionVariableType オブジェクトがファイナライズ可能か"));
    nn::os::FinalizeConditionVariable( &g_StaticCondVar1 );
    nn::os::FinalizeConditionVariable( &g_StaticCondVar2 );
    CheckBool( true );

    // エージングテスト用にもう一度初期化状態に戻しておく
    nn::os::InitializeConditionVariable( &g_StaticCondVar1 );
    nn::os::InitializeConditionVariable( &g_StaticCondVar2 );

    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  ConditionVariable クラスの型変換関数のテスト
//  ここでは、ConditionVariable クラスが OS-API をラッピングしたものであるという
//  前提で各メソッドの簡単な動作確認のみを行なうテストである。
//  ここでは、GetBase() および operator ConditionVariableType&() の動作テストを行なう。
//
nn::os::Mutex   g_TypeExchangeMutex(false);
bool            g_TypeExchangeFlag;

void test_ConditionVariableClassTypeExchangeTestThread(void* arg)
{
    auto*  condVarType = static_cast<nn::os::ConditionVariableType*>(arg);

    // 引数で受取った条件変数オブジェクトに対して Signal する
    // 事前に親スレッドがロックを獲得しているので、子スレッドがこのロック区間に
    // 入った時点で親スレッドは条件変数待ちになっている。
    g_TypeExchangeMutex.Lock();
    {
        g_TypeExchangeFlag = true;
        nn::os::SignalConditionVariable(condVarType);
    }
    g_TypeExchangeMutex.Unlock();
}

TEST(ConditionVariableClass, test_ConditionVariableClassTestTypeExchange)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("condVar インスタンスの生成\n"));
        nn::os::ConditionVariable condVar;

        // ここからがテスト
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("condVar.GetBase() で ConditionVariableType オブジェクトを取得"));
        nn::os::ConditionVariableType*  condVarType = condVar.GetBase();
        CheckBool( condVarType != NULL );

        // 子スレッドを作成
        // 子スレッドには直前で GetBase() した条件変数オブジェクトを渡す
        nn::os::ThreadType  thread;
        nn::os::CreateThread( &thread, test_ConditionVariableClassTypeExchangeTestThread, condVarType, g_ThreadStack, sizeof(g_ThreadStack), nn::os::GetThreadPriority(nn::os::GetCurrentThread()));

        g_TypeExchangeFlag = false;

        // condVar クラスを使って mutex ロック中に条件変数待機→復帰
        // 子スレッドは ConditionVariableType オブジェクトを使って Signal する
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("condVar.Wait() で条件変数を待ち、子スレッドから Signal する"));

        g_TypeExchangeMutex.Lock();
        {
            // ロックを取得してから子スレッドの動作を開始
            nn::os::StartThread( &thread );
            while (g_TypeExchangeFlag == false)
            {
                condVar.Wait(g_TypeExchangeMutex);
            }
        }
        g_TypeExchangeMutex.Unlock();
        CheckBool( true );

        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );

        // operator ConditionVariableType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator ConditionVariableType&() を使ってオブジェクトの参照を取得"));
        nn::os::ConditionVariableType& condVarTypeRefer = condVar;
        CheckBool( &condVarTypeRefer == condVarType );

        // operator const ConditionVariableType&() の確認
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("operator const ConditionVariableType&() を使ってオブジェクトの参照を取得"));
        const nn::os::ConditionVariableType& condVarTypeReferConst = condVar;
        CheckBool( &condVarTypeReferConst == condVarType );
    }

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}


#if defined(NN_BUILD_CONFIG_OS_HORIZON)
//---------------------------------------------------------------------------
//  SIGLO-8567 のテスト
//
//  厳格な優先度ベースのスケジューリングが必要なので、Horizon 専用のテスト
//  として実施する。このテストにある 2 つのスレッドは、同一コア上で動作する
//  ことを前提としており、マルチコア上で同時に動作してはならない。
//
volatile bool   g_TimedWaitConditionVariableTestPassedFlag;

void TimedWaitConditionVariableTimeoutZeroSubThread(void* arg)
{
    auto*   mutex = static_cast<nn::os::MutexType*>(arg);

    // 子スレッドでもロックする（ここでは必ず一旦待機する）
    nn::os::LockMutex( mutex );

    // 通過フラグをセット
    g_TimedWaitConditionVariableTestPassedFlag = true;

    nn::os::UnlockMutex( mutex );
}

TEST(TimedWaitConditionVariable, test_TimedWaitConditionVariableTimeoutZero)
{
    // 個別のテスト集計を開始
    CLEAR_GOOGLE_TEST();

    // 一時的にスレッド優先度を設定する
    int currentPriority = nn::os::ChangeThreadPriority(
                                    nn::os::GetCurrentThread(),
                                    nn::os::DefaultThreadPriority);

    g_TimedWaitConditionVariableTestPassedFlag = false;

    {
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("TimedWaitConditionVariable(timeout=0) で Mutex が一時的に\n"));
        SEQ_NONE();
        NNT_OS_LOG(NN_TEXT("アンロックされるか"));

        nn::os::MutexType               mutex;
        nn::os::ConditionVariableType   cond;

        nn::os::InitializeMutex( &mutex, false, 0 );
        nn::os::InitializeConditionVariable( &cond );

        // 子スレッドを生成（同一コアを指定、親スレッドより優先度が高い）
        nn::os::ThreadType  thread;
        int coreNum = nn::os::GetCurrentCoreNumber();
        auto result = nn::os::CreateThread( &thread, TimedWaitConditionVariableTimeoutZeroSubThread, &mutex, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority - 1, coreNum );
        EXPECT_TRUE( result.IsSuccess() );

        // まず先にロックする
        nn::os::LockMutex( &mutex );

        // 子スレッドを開始
        // ここですぐに子スレッドが動き出し、Mutex ロック待ちに入って
        // またすぐにこちらに制御が戻ってくる。
        nn::os::StartThread( &thread );

        // ここで timeout=0 で条件変数待機する。
        // 一瞬だけアンロックされるようであれば、そこで子スレッドが動くはず。
        nn::os::TimedWaitConditionVariable( &cond, &mutex, nn::TimeSpan(0) );
        CheckBool( g_TimedWaitConditionVariableTestPassedFlag );

        // あとは終了処理
        nn::os::UnlockMutex( &mutex );

        nn::os::WaitThread( &thread );
        nn::os::DestroyThread( &thread );

        nn::os::FinalizeMutex( &mutex );
        nn::os::FinalizeConditionVariable( &cond );
    }

    // スレッド優先度を元に戻す
    nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), currentPriority);

    // 個別のテスト集計を通知
    JUDGE_GOOGLE_TEST();
}

//---------------------------------------------------------------------------
//  SIGLO-11040 のテスト
//
//  非キャッシュ領域に同期オブジェクトを配置し、チェックに引っ掛かるかを確認します。
//  Debug か Develop ビルドでのみチェックは行われます。
//
#if 0
NN_ALIGNAS(4096) char g_DataForPuttingSyncObject[4096];

TEST(PutSyncObjectOnUncachedAreaTest, test_PutSyncObjectOnUncachedArea)
{
    // Unchaced 領域に変更する
    nn::os::SetMemoryAttribute(
        reinterpret_cast<uintptr_t>(g_DataForPuttingSyncObject),
        sizeof(g_DataForPuttingSyncObject),
        nn::os::MemoryAttribute_Uncached );

    // placement new で上記領域に同期オブジェクトを設置
    nn::os::ConditionVariable* pCond = new( g_DataForPuttingSyncObject ) nn::os::ConditionVariable();

    NN_UNUSED( pCond );
}
#endif

#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON)

}}} // namespace nnt::os::conditionVariable

//---------------------------------------------------------------------------
//  Test Main 関数
//---------------------------------------------------------------------------

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    int result;

    NNT_CALIBRATE_INITIALIZE();
    SEQ_INITIALIZE();
    INITIALIZE_TEST_COUNT();

    // テスト開始
    SEQ_CHECK(0);
    NNT_OS_LOG("=== Start Test of ConditionVariable APIs\n");

    // GoogleTest おまじない
    ::testing::InitGoogleTest(&argc, argv);
    result = RUN_ALL_TESTS();

    // テスト終了
    NNT_OS_LOG("\n=== End Test of ConditionVariable APIs\n");

    // 集計結果の表示
    g_Result.Show();

    nnt::Exit(result);
}
