﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"

#include <string>
#include <cstdlib>

#include <nn/os/os_Config.h>
#include <nn/nn_SdkText.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_FormatString.h>
#include <nn/os.h>
#include <nn/nn_Windows.h>

#include "test_Helper.h"
#include "test_NamedPipe.h"

namespace nnt { namespace os { namespace detail {

namespace {

void CreateWin32PipeName(char* pipeName, const char* portName)
NN_NOEXCEPT
{
    std::strcpy(pipeName, "\\\\.\\pipe\\");
    std::strcat(pipeName, portName);
}

}   // namespace

void CreateUniquePortName(char* portName, size_t size)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size >= 12,
        "CreateUniquePortName(): size=%d is too small.", size);

    // portName のアドレスをプロセス内でユニークなものとして名前に変換する。
    // Horizon の svc ではポート名は 12 文字までなのでそれに収める。
    //
    // 理想的には、プロセス内はもちろん、DEATH テストの子プロセス、
    // 頻繁なポートの Create/Destroy を繰り返した場合でも、
    // 常にユニークな名前になることが望ましい。
    // そのため、portName のアドレスに加え、Tick 値を名前として付加し、
    // 同一 portName バッファが使いまわされた場合でも、
    // ある程度ユニークになるようにしておく。
    int count = nn::os::GetSystemTick().GetInt64Value() & 0xfff;
    nn::util::SNPrintf(portName, size, "P_%X_%03X", portName, count);
}

void CreateUniqueInterruptName(char* interruptName, size_t size)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(size >= 32,
        "CreateUniqueInterruptName(): size=%d is too small.", size);

    // interruptName のアドレスをプロセス内でユニークなものとして名前に使う。
    int64_t count = nn::os::GetSystemTick().GetInt64Value();
    nn::util::SNPrintf(interruptName, size, "I_%X_%016x", interruptName, count);
}


//=============================================================================
// NamedPipeServerImplByWin32
//=============================================================================

//-----------------------------------------------------------------------------
// コンストラクタ
//
//  既定の名前で Win32 名前付きパイプを開き、以後 ReplyValue() と RecvHandle()
//  が利用可能となる。これらは、プロセスを超えて通信を行なうことが出来る。
//
NamedPipeServerImplByWin32::NamedPipeServerImplByWin32(const char* portName)
NN_NOEXCEPT
{
    char    pipeName[64];
    CreateWin32PipeName(pipeName, portName);
    NNT_OS_LOG("Server: CreatedPipeName=\x22%s\x22\n", pipeName);

    // ハンドル値を一旦無効化
    m_Handle = nullptr;

    HANDLE  handle = CreateNamedPipeA(
                        pipeName,                       // パイプの名前
                        PIPE_ACCESS_DUPLEX,             // オープンモード
                        PIPE_TYPE_BYTE | PIPE_WAIT,     // パイプ固有モード
                        4,                              // インスタンス最大数
                        256,                            // 出力バッファサイズ
                        256,                            // 入力バッファサイズ
                        1000,                           // タイムアウト（msec）
                        NULL);                          // セキュリティ記述子

    if (handle == INVALID_HANDLE_VALUE)
    {
        NNT_OS_LOG("errNum= %d\n", GetLastError());
        NN_ABORT(NN_TEXT("Server: Pipe を作成できませんでした。"));
    }

    // 作成されたパイプのハンドルを設定
    m_Handle = handle;
}

//-----------------------------------------------------------------------------
// デストラクタ
//
//  既定の名前で作成した Win32 名前付きパイプを閉じる。
//
NamedPipeServerImplByWin32::~NamedPipeServerImplByWin32()
NN_NOEXCEPT
{
    if (!m_Handle)
    {
        return;
    }

    FlushFileBuffers( m_Handle );
    DisconnectNamedPipe( m_Handle );
    CloseHandle( m_Handle );
    m_Handle = nullptr;
}


//-----------------------------------------------------------------------------
// 接続
//
//  サーバ側でパイプに接続する
//
void NamedPipeServerImplByWin32::Connect()
NN_NOEXCEPT
{
    if (!m_Handle)
    {
        NN_ABORT(NN_TEXT("Server: Pipe が作成されていません。"));
    }

    BOOL ret = ConnectNamedPipe( m_Handle, NULL );
    if (ret)
    {
        // 接続に成功ならリターン
        return;
    }

    // 接続失敗
    CloseHandle( m_Handle );
    m_Handle = 0;
    NN_ABORT(NN_TEXT("Server: Pipe への接続に失敗しました。"));
}

//-----------------------------------------------------------------------------
// 接続解除
//
void NamedPipeServerImplByWin32::Disconnect()
NN_NOEXCEPT
{
    if (!m_Handle)
    {
        return;
    }

    DisconnectNamedPipe( m_Handle );
}

//-----------------------------------------------------------------------------
// 符号なし整数値の返信
//
void NamedPipeServerImplByWin32::ReplyValue(uint32_t value)
NN_NOEXCEPT
{
    if (!m_Handle)
    {
        NN_ABORT(NN_TEXT("Server: Pipe が作成されていません。"));
    }

    DWORD   writtenByte;
    WriteFile( m_Handle, &value, sizeof(value), &writtenByte, NULL);
}

//-----------------------------------------------------------------------------
// ハンドルの受信
//
//  nn::os::NativeHandle 型の値を受信して返す。失敗時には 0 を返す。
//  受信待ち状態で Client 側が切断した場合、一旦 Disconnect() した上で、
//  再度 Connect() を行ない、セッションを張り替える。
//
nn::os::NativeHandle NamedPipeServerImplByWin32::RecvHandle()
NN_NOEXCEPT
{
    if (!m_Handle)
    {
        NN_ABORT(NN_TEXT("Server: Pipe が作成されていません。"));
    }

    for (;;)
    {
        nn::os::NativeHandle   handle;
        DWORD       readByte;

        BOOL ret = ReadFile( m_Handle, &handle, sizeof(handle), &readByte, NULL);
        if (ret == 0)
        {
            // 一旦解除した上で、再度接続を試みる
            Disconnect();
            Connect();
            continue;
        }

        if (readByte != sizeof(nn::os::NativeHandle))
        {
            // 受信に失敗
            return 0;
        }

        // 受信した値を返す
        return handle;
    }
}

//-----------------------------------------------------------------------------
// ハンドルの破棄
//
void NamedPipeServerImplByWin32::DestroyHandle(nn::os::NativeHandle handle) NN_NOEXCEPT
{
    // Win32 ではハンドルの IPC 受信で参照カウンタを増やさないので
    // ここでは何もしない
    NN_UNUSED(handle);
}

//=============================================================================
// NamedPipeClientImplByWin32
//=============================================================================

//-----------------------------------------------------------------------------
// コンストラクタ
//
//  既定の名前で Win32 名前付きパイプを開き、以後 SendHandle() と RecvValue()
//  が利用可能となる。これらは、プロセスを超えて通信を行なうことが出来る。
//
NamedPipeClientImplByWin32::NamedPipeClientImplByWin32(const char* portName)
NN_NOEXCEPT
{
    char    pipeName[64];
    CreateWin32PipeName(pipeName, portName);
    NNT_OS_LOG("Client: ConnectToPipeName=\x22%s\x22\n", pipeName);

    // ハンドル値を一旦無効化
    m_Handle = 0;

    int count = 0;
    while (count < 50)
    {
        // パイプを開く
        HANDLE handle = CreateFileA(
                         pipeName,                      // パイプの名前
                         GENERIC_READ | GENERIC_WRITE,  // アクセスモード
                         0,                             // 共有モード
                         NULL,                          // セキュリティ記述子
                         OPEN_EXISTING,                 // 作成方法
                         FILE_ATTRIBUTE_NORMAL,         // ファイル属性
                         NULL);                         // テンプレートファイル

        if (handle != INVALID_HANDLE_VALUE)
        {
            // 作成されたパイプのハンドルを設定してリターン
            m_Handle = handle;
            return;
        }

        // クライアントからサーバへの Pipe 接続に失敗
        // サーバ側で disconnect -> connect 処理中なので少し待ってリトライ
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100) );
        ++count;
    }

    NN_ABORT(NN_TEXT("Client: Pipe への接続に失敗しました。"));
}

//-----------------------------------------------------------------------------
// デストラクタ
//
//  既定の名前で作成した Win32 名前付きパイプを閉じる。
//
NamedPipeClientImplByWin32::~NamedPipeClientImplByWin32()
NN_NOEXCEPT
{
    if (!m_Handle)
    {
        return;
    }

    CloseHandle( m_Handle );
    m_Handle = 0;
}


//-----------------------------------------------------------------------------
// ハンドルの送信
//
void NamedPipeClientImplByWin32::SendHandle(int cmd, nn::os::NativeHandle handle)
NN_NOEXCEPT
{
    NN_UNUSED(cmd);

    if (!m_Handle)
    {
        NN_ABORT(NN_TEXT("Client: Pipe がオープンされていません。"));
    }

    DWORD   writtenByte;
    WriteFile( m_Handle, &handle, sizeof(handle), &writtenByte, NULL);
}

//-----------------------------------------------------------------------------
// 符号なし整数値の受信
//
uint32_t NamedPipeClientImplByWin32::RecvValue()
NN_NOEXCEPT
{
    if (!m_Handle)
    {
        NN_ABORT(NN_TEXT("Client: Pipe がオープンされていません。"));
    }

    uint32_t    value;
    DWORD       readByte;
    ReadFile( m_Handle, &value, sizeof(value), &readByte, NULL);

    if (readByte != sizeof(value))
    {
        return 0;
    }

    return value;
}

//-----------------------------------------------------------------------------

}}} // namespace nnt::os::detail


