﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//---------------------------------------------------------------------------
//  Calibration 機能
//
//  テストを実装する上で必要となる以下の機能を提供する。
//  ・実動作時間を計測し、テスト実行時の待ち時間をキャリブレーションする
//---------------------------------------------------------------------------

#pragma once

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>

//---------------------------------------------------------------------------
//【テストシーケンス】
//
//【単発実行】
//
//  th1 :   +-+Wait/Exit
//  th2 :   | |
// main :---+ |                                        +--
// idle :     +........................................+
//
//
//【単発実行×２】
//
//  th1 :   +-+Wait/Exit
//  th2 :   | |        +-+Wait/Exit
// main :---+ |        | |                             +--
// idle :     +........+ +.............................+
//
//
//【単発実行解除系×２】
//
//  th1 :   +-+Wait      +-+Exit
//  th2 :   | |        +-+ |Exit
// main :---+ |        |   |                           +--
// idle :     +........+   +...........................+
//                       ↑Signaled
//
//【単発タイムアウト】
//
//  th1 :   +-+Timed              +-+Exit
//  th2 :   | |                   | |
// main :---+ |                   | |                  +---
// idle :     +...................+ +..................+
//
//
//【単発タイムアウト解除】
//
//  th1 :   +-+Timed     +-+Exit
//  th2 :   | |        +-+ |Exit
// main :---+ |        |   |                            +---
// idle :     +........+   +............................+
//                       ↑Signaled
//
//【単発タイムアウト×２（成功＋待機）】
//
//  th1 :   +-+Exit
//  th2 :   | |        +-+Timed              +-+Exit
// main :---+ |        | |                   | |        +---
// idle :     +........+ +...................+ +........+
//
//
//【単発タイムアウト×２（待機＋待機）】
//
//  th1 :   +-+Timed              +-+Exit
//  th2 :   | |        +-+Timed   | |        +-+Exit
// main :---+ |        | |        | |        | |        +---
// idle :     +........+ +........+ +........+ +........+
//
//---------------------------------------------------------------------------

const int   nntMeasureCount = 4;

//---------------------------------------------------------------------------
//  同期 API を使わない劣等同期クラス
//---------------------------------------------------------------------------
class   Calibration
{
public:
    Calibration()    {}
    ~Calibration()   {}

    void Initialize()   NN_NOEXCEPT;

    int64_t GetBaseInterval()   const NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON) && defined(NN_BUILD_CONFIG_HARDWARE_SMMA53)
        return m_calibratedBase / 20;
#else
        return m_calibratedBase;
#endif
    }

private:
    nn::TimeSpan GetMinimumCalibratedTime() NN_NOEXCEPT
    {
        return nn::TimeSpan::FromMicroSeconds(500);
    }

private:
    int64_t m_calibratedBase;
};

extern Calibration   g_Calibration;

#define NNT_CALIBRATE_INITIALIZE()  g_Calibration.Initialize()
#define NNT_CALIBRATE_RATE()        g_Calibration.GetBaseInterval()

//---------------------------------------------------------------------------

