﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>

#include <nn/account/account_ApiForAdministrators.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/database/olsc_TransferTaskContextDatabase.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabaseManager.h>
#include <nn/olsc/srv/database/olsc_TransferTaskDatabase.h>
#include <nn/olsc/srv/database/olsc_ErrorHistoryDatabase.h>
#include <nn/olsc/srv/olsc_SystemEventManager.h>
#include <nn/olsc/srv/olsc_TransferTask.h>
#include <nn/olsc/srv/olsc_TransferTaskAgent.h>
#include <nn/olsc/srv/util/olsc_MountManager.h>
#include <nn/olsc/srv/util/olsc_File.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/util/util_FormatString.h>

#include "testOlsc_TransferTaskManagement.h"
#include "testOlsc_SaveDataUtil.h"

#include <vector>

#define NNT_OLSC_TTA_BASIC
#define NNT_OLSC_TTA_CANCEL

using namespace nn;
using namespace nn::olsc;
using namespace nn::olsc::srv;
using namespace nn::olsc::srv::database;

namespace {

    class OlscTransferTaskDatabaseManagerTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            NNT_ASSERT_RESULT_SUCCESS(InitializeSaveData());
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            fs::SetEnabledAutoAbort(false);
            fs::DisableAutoSaveDataCreation();
            NNT_ASSERT_RESULT_SUCCESS(time::Initialize());

#if defined( NN_BUILD_CONFIG_OS_WIN )
            nn::account::InitializeForAdministrator();

            nnt::olsc::CleanupUsers();
            nn::account::Uid uid;
            nnt::olsc::CreateUsers(&uid, 1);
#else
            account::InitializeForSystemService();
#endif
            NNT_ASSERT_RESULT_SUCCESS(nifm::InitializeSystem());
        }

        static void TearDownTestCase()
        {
            NNT_ASSERT_RESULT_SUCCESS(time::Finalize());
        }
    private:

        static Result InitializeSaveData()
        {
            fs::SystemSaveDataId toDelete[] = {
                nnt::olsc::MountInfoForTestDeviceSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSettingSave.systemSaveDataId,
                nnt::olsc::MountInfoForTestUserSeriesInfoSave.systemSaveDataId,
            };
            for (auto& id : toDelete)
            {
                NN_RESULT_TRY(fs::DeleteSystemSaveData(fs::SaveDataSpaceId::System, id, {}))
                    NN_RESULT_CATCH(fs::ResultTargetNotFound)
                {
                }
                NN_RESULT_END_TRY;
            }

            NN_RESULT_SUCCESS;
        }
    };

    struct TaskParam
    {
        account::Uid        uid;
        ApplicationId       appId;
        TransferTaskKind    kind;
    };

    TaskParam CreateTaskParamByIndex(int index) NN_NOEXCEPT
    {
        return TaskParam
        {
            nnt::olsc::GetFirstUserId(),
            { static_cast<Bit64>(index) },
            index % 2 == 0 ? TransferTaskKind::Download : TransferTaskKind::Upload
        };
    }
}

#if defined(NNT_OLSC_TTA_BASIC)
TEST_F(OlscTransferTaskDatabaseManagerTest, Basic)
{
    olsc::srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave);

    TransferTaskDatabaseManager ttdm(mountManager);

    const int MaxTaskCount = 10;
    std::vector<TransferTaskId> expectedDoneList;
    for (int i = 0; i < MaxTaskCount; ++i)
    {
        auto p = CreateTaskParamByIndex(i);
        TransferTaskConfig config = {};
        config.kind = p.kind;
        TransferTaskId id;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ttdm.RegisterTransferTask(&id, p.uid, p.appId, config, TransferTaskRank::Basic));
        expectedDoneList.push_back(id);
    }

    std::vector<TransferTaskId> taskDoneList;

    static TransferTaskAgent::ThreadResource threadResource(
        NN_SYSTEM_THREAD_NAME(olsc, TransferTask),
        NN_SYSTEM_THREAD_PRIORITY(olsc, TransferTask)
    );

    static nn::olsc::srv::TransferTaskBase::TransferTaskExecutionResource executionResource = {};

    // appId が奇数のタスクは失敗扱いにする。
    auto taskResultArbiter = [](const TransferTaskDetailInfo& info) -> Result
    {
        if (info.appId.value % 2 == 0)
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW(fs::ResultPathNotFound());
    };

    nnt::olsc::TransferTaskFactoryForTest factory(ttdm, taskDoneList, 0, taskResultArbiter);
    TransferTaskCompleteEventManager completeEventManager;
    TransferTaskStartEventManager startEventManager;
    SeriesInfoDatabaseManager seriesInfoDatabaseManager(mountManager);
    SaveDataArchiveInfoCacheManager sdaInfoCacheManager(mountManager);
    ErrorHistoryDatabase errorHistoryDatabase(mountManager);

    // appId が偶数のタスクは成功し、エラー履歴が消去される。
    // これを確認するためにあらかじめ偶数の appId に対してエラー履歴を登録しておく
    for (int i = 0; i < MaxTaskCount; ++i)
    {
        auto p = CreateTaskParamByIndex(i);
        if (p.appId.value % 2 == 0)
        {
            EXPECT_TRUE(errorHistoryDatabase.SetLastError(p.uid, p.appId, p.kind, false, fs::ResultPathAlreadyExists()));
        }
    }

    TransferTaskAgent agent(ttdm, threadResource, factory, startEventManager, completeEventManager, seriesInfoDatabaseManager, sdaInfoCacheManager, errorHistoryDatabase, executionResource);

    os::SystemEventType finishedEvent;
    NNT_ASSERT_RESULT_SUCCESS(completeEventManager.AcquireSystemEvent(&finishedEvent));

    os::SystemEventType startedEvent;
    NNT_ASSERT_RESULT_SUCCESS(startEventManager.AcquireSystemEvent(&startedEvent));

    agent.Start();
    EXPECT_TRUE(nnt::olsc::WaitTaskEvent(&startedEvent, [&]() -> bool {
        return true;
    }, 10));

    nnt::olsc::WaitTaskEvent(&finishedEvent, [&]() -> bool {
        return expectedDoneList == taskDoneList;
    }, 60);

    agent.Stop();

    EXPECT_EQ(expectedDoneList, taskDoneList);

    // エラー履歴が残っていることを確認
    for (int i = 0; i < MaxTaskCount; ++i)
    {
        auto p = CreateTaskParamByIndex(i);
        auto lastError = errorHistoryDatabase.GetLastErrorInfo(p.uid, p.appId);
        if (p.appId.value % 2 == 0)
        {
            EXPECT_FALSE(lastError);
        }
        else
        {
            ASSERT_TRUE(lastError);
            NNT_EXPECT_RESULT_FAILURE(fs::ResultPathNotFound, lastError->result);
        }
    }
}
#endif

#if defined(NNT_OLSC_TTA_CANCEL)
TEST_F(OlscTransferTaskDatabaseManagerTest, Cancel)
{
    olsc::srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTestDeviceSave, nnt::olsc::MountInfoForTestUserSettingSave, nnt::olsc::MountInfoForTestUserSeriesInfoSave);

    srv::database::TransferTaskDatabaseManager ttdm(mountManager);

    // 2つのタスクを積むが、2つ目はキャンセルさせるため、ExpectedDoneList には1つ目だけの情報を記録
    const int MaxTaskCount = 2;
    std::vector<TransferTaskId> expectedDoneList;
    for (int i = 0; i < MaxTaskCount; ++i)
    {
        auto p = CreateTaskParamByIndex(i);
        TransferTaskConfig config = {};
        config.kind = p.kind;
        TransferTaskId id;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ttdm.RegisterTransferTask(&id, p.uid, p.appId, config, TransferTaskRank::Basic));
        if (i == 0)
        {
            expectedDoneList.push_back(id);
        }
    }

    std::vector<TransferTaskId> taskDoneList;

    static TransferTaskAgent::ThreadResource threadResource(
        NN_SYSTEM_THREAD_NAME(olsc, TransferTask),
        NN_SYSTEM_THREAD_PRIORITY(olsc, TransferTask)
    );

    static nn::olsc::srv::TransferTaskBase::TransferTaskExecutionResource executionResource;

    auto taskResultArbiter = [](const TransferTaskDetailInfo& info) -> Result
    {
        NN_RESULT_SUCCESS;
    };

    nnt::olsc::TransferTaskFactoryForTest factory(ttdm, taskDoneList, 5, taskResultArbiter);
    TransferTaskCompleteEventManager completeEventManager;
    TransferTaskCompleteEventManager startEventManager;
    SeriesInfoDatabaseManager seriesInfoDatabaseManager(mountManager);
    SaveDataArchiveInfoCacheManager sdaInfoCacheManager(mountManager);
    ErrorHistoryDatabase errorHistoryDatabase(mountManager);
    TransferTaskAgent agent(ttdm, threadResource, factory, startEventManager, completeEventManager, seriesInfoDatabaseManager, sdaInfoCacheManager, errorHistoryDatabase, executionResource);

    os::SystemEventType startedEvent;
    NNT_ASSERT_RESULT_SUCCESS(startEventManager.AcquireSystemEvent(&startedEvent));


    agent.Start();
    EXPECT_TRUE(nnt::olsc::WaitTaskEvent(&startedEvent, [&]() -> bool {
        return true;
    }, 10));

    // 1つ目のタスクが終わり、2つ目のタスクの完了待ち中にキャンセルする

    os::SystemEventType finishedEvent;
    NNT_ASSERT_RESULT_SUCCESS(completeEventManager.AcquireSystemEvent(&finishedEvent));
    nnt::olsc::WaitTaskEvent(&finishedEvent, [&]() -> bool {
        return taskDoneList.size() >= 1;
    }, 60);

    agent.CancelCurrentTask();

    EXPECT_EQ(expectedDoneList, taskDoneList);
}
#endif
