﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/base/testBase_Exit.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

#include <nn/http/json/http_RapidJsonApi.h>
#include <nn/http/json/http_JsonErrorMap.h>
#include <nn/olsc/olsc_Result.h>
#include <nn/olsc/srv/olsc_InternalTypes.h>
#include <olsc_SaveDataArchiveInfoAdaptor.h>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_UserAccountSystemSaveData.h>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ApiForAdministrators.h>

#include <olsc_OutputStream.h>
#include "testOlsc_InputStream.h"
#include "testOlsc_SaveDataUtil.h"

using namespace nn;
using namespace nn::olsc;

namespace save_data_archive_success {

const char SaveDataArchiveJson[] =
"{"
    "\"save_data_archives\" : ["
    "{"
        // 全部最大値
        "\"id\"                      : 9223372036854775807,"
        "\"nsa_id\"                  : \"ffffffffffffffff\","
        "\"application_id\"          : \"ffffffffffffffff\","
        "\"device_id\"               : \"ffffffffffffffff\","
        "\"data_size\"               : 4294967295,"
        "\"auto_backup\"             : true,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"uploading\","
        "\"has_thumbnail\"           : true,"
        "\"launch_required_version\" : 4294967295,"
        "\"num_of_partitions\"   : 4294967295,"
        "\"series_id\"               : 18446744073709551615,"
        "\"saved_at_as_unixtime\"    : 9223372036854775807,"
        "\"timeout_at_as_unixtime\"  : 9223372036854775807,"
        "\"finished_at_as_unixtime\" : 9223372036854775807,"
        "\"created_at_as_unixtime\"  : 9223372036854775807,"
        "\"updated_at_as_unixtime\"  : 9223372036854775807"
    "},"
    "{"
        // 現実的な値、かつ値が混同されていないことが確認できる
        "\"id\"                      : 1234,"
        "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
        "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
        "\"device_id\"               : \"dddddddddddddddd\","
        "\"data_size\"               : 123456789,"
        "\"auto_backup\"             : false,"
        "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
        "\"status\"                  : \"fixed\","
        "\"has_thumbnail\"           : false,"
        "\"launch_required_version\" : 5678,"
        "\"num_of_partitions\"   : 16,"
        "\"series_id\"               : 100,"
        "\"saved_at_as_unixtime\"    : 1514732400,"
        "\"timeout_at_as_unixtime\"  : 1514732401,"
        "\"finished_at_as_unixtime\" : 1514732402,"
        "\"created_at_as_unixtime\"  : 1514732403,"
        "\"updated_at_as_unixtime\"  : 1514732404"
    "}"
    "]"
"}";

struct Cancelable
{
    bool IsCancelled() const { return false; }
};

static const fs::SaveDataCommitId TestSaveDataCommitId = 0x4e728abeca603d6c;

TEST(OlscJsonAdaptorTest, SaveDataMeta)
{
    auto uid = nnt::olsc::GetFirstUserId();
    nn::fs::DeleteSystemSaveData(nnt::olsc::SystemSaveDataIdForTest, uid);
    srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest);
    srv::database::SaveDataArchiveInfoCache sdaInfoCache(uid, mountManager);

    olsc::srv::adaptor::SaveDataArchiveInfoStorageOutputStream outputStream(sdaInfoCache);
    olsc::srv::adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);

    nnt::olsc::MemoryInputStreamForRapidJson inputStream;
    inputStream.Set(SaveDataArchiveJson, sizeof(SaveDataArchiveJson));

    const Cancelable c;
    NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, inputStream, &c));
    NNT_EXPECT_RESULT_SUCCESS(adaptor.GetResult());

    EXPECT_TRUE(sdaInfoCache.GetCount() == 2);

    srv::SaveDataArchiveInfo sda[2];

    sdaInfoCache.List(sda, 2, 0);

    // appId でソートされるため順番は逆になる
    {
        EXPECT_EQ(1234, sda[0].id);
        EXPECT_EQ(0xaaaaaaaaaaaaaaaaull, sda[0].nsaId.id);
        EXPECT_EQ(0xbbbbbbbbbbbbbbbbull, sda[0].applicationId.value);
        EXPECT_EQ(123456789, sda[0].dataSize);
        EXPECT_EQ(0xddddddddddddddddull, sda[0].deviceId);
        EXPECT_EQ(TestSaveDataCommitId, sda[0].seriesInfo.commitId);
        EXPECT_EQ(olsc::srv::SaveDataArchiveStatus::Fixed, sda[0].status);
        EXPECT_EQ(false, sda[0].autoBackup);
        EXPECT_EQ(false, sda[0].hasThumbnail);
        EXPECT_EQ(5678, sda[0].launchRequiredVersion);
        EXPECT_EQ(16, sda[0].numOfPartitions);
        EXPECT_EQ(100, sda[0].seriesInfo.seriesId);
        EXPECT_EQ(1514732400, sda[0].savedAt.value);
        EXPECT_EQ(1514732401, sda[0].timeoutAt.value);
        EXPECT_EQ(1514732402, sda[0].finishedAt.value);
        EXPECT_EQ(1514732403, sda[0].createdAt.value);
        EXPECT_EQ(1514732404, sda[0].updatedAt.value);
    }


    {
        EXPECT_EQ(INT64_MAX, sda[1].id);
        EXPECT_EQ(UINT64_MAX, sda[1].nsaId.id);
        EXPECT_EQ(UINT64_MAX, sda[1].applicationId.value);
        EXPECT_EQ(UINT32_MAX, sda[1].dataSize);
        EXPECT_EQ(UINT64_MAX, sda[1].deviceId);
        EXPECT_EQ(TestSaveDataCommitId, sda[1].seriesInfo.commitId);
        EXPECT_EQ(olsc::srv::SaveDataArchiveStatus::Uploading, sda[1].status);
        EXPECT_EQ(true, sda[1].autoBackup);
        EXPECT_EQ(true, sda[1].hasThumbnail);
        EXPECT_EQ(UINT32_MAX, sda[1].launchRequiredVersion);
        EXPECT_EQ(UINT32_MAX, sda[1].numOfPartitions);
        EXPECT_EQ(UINT64_MAX, sda[1].seriesInfo.seriesId);
        EXPECT_EQ(INT64_MAX, sda[1].savedAt.value);
        EXPECT_EQ(INT64_MAX, sda[1].timeoutAt.value);
        EXPECT_EQ(INT64_MAX, sda[1].finishedAt.value);
        EXPECT_EQ(INT64_MAX, sda[1].createdAt.value);
        EXPECT_EQ(INT64_MAX, sda[1].updatedAt.value);
    }
}

} // namespace save_data_archive

namespace save_data_archive_fail {
    const int JsonLen = 1024;

    const char InvalidDigest[][JsonLen] =
    {
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"auto_backup\"             : false,"
            "\"encoded_digest\"          : \"cbD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"has_thumbnail\"           : false,"
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 4294967296,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"timeout_at_as_unixtime\"  : 1514732401,"
            "\"finished_at_as_unixtime\" : 1514732402,"
            "\"created_at_as_unixtime\"  : 1514732403,"
            "\"updated_at_as_unixtime\"  : 1514732404"
        "}"
        "]}",
         "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"auto_backup\"             : false,"
            "\"encoded_digest\"          : \"D1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"has_thumbnail\"           : false,"
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 4294967296,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"timeout_at_as_unixtime\"  : 1514732401,"
            "\"finished_at_as_unixtime\" : 1514732402,"
            "\"created_at_as_unixtime\"  : 1514732403,"
            "\"updated_at_as_unixtime\"  : 1514732404"
        "}"
        "]}"
    };

    const char InvalidNumOfPartitions[][JsonLen] =
    {
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"auto_backup\"             : false,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"has_thumbnail\"           : false,"
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 4294967296,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"timeout_at_as_unixtime\"  : 1514732401,"
            "\"finished_at_as_unixtime\" : 1514732402,"
            "\"created_at_as_unixtime\"  : 1514732403,"
            "\"updated_at_as_unixtime\"  : 1514732404"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"auto_backup\"             : false,"
            "\"encoded_digest\"          : \"D1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"has_thumbnail\"           : false,"
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 0,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"timeout_at_as_unixtime\"  : 1514732401,"
            "\"finished_at_as_unixtime\" : 1514732402,"
            "\"created_at_as_unixtime\"  : 1514732403,"
            "\"updated_at_as_unixtime\"  : 1514732404"
        "}"
        "]}"
    };

    const char InvalidDataSize[][JsonLen] =
    {
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 4294967296,"
            "\"auto_backup\"             : false,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"has_thumbnail\"           : false,"
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"timeout_at_as_unixtime\"  : 1514732401,"
            "\"finished_at_as_unixtime\" : 1514732402,"
            "\"created_at_as_unixtime\"  : 1514732403,"
            "\"updated_at_as_unixtime\"  : 1514732404"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 0,"
            "\"auto_backup\"             : false,"
            "\"encoded_digest\"          : \"D1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"has_thumbnail\"           : false,"
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 4294967296,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"timeout_at_as_unixtime\"  : 1514732401,"
            "\"finished_at_as_unixtime\" : 1514732402,"
            "\"created_at_as_unixtime\"  : 1514732403,"
            "\"updated_at_as_unixtime\"  : 1514732404"
        "}"
        "]}"
    };

    const char InvalidApplicationIdJson[][JsonLen] =
    {
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"0\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"auto_backup\"             : false,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"has_thumbnail\"           : false,"
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"timeout_at_as_unixtime\"  : 1514732401,"
            "\"finished_at_as_unixtime\" : 1514732402,"
            "\"created_at_as_unixtime\"  : 1514732403,"
            "\"updated_at_as_unixtime\"  : 1514732404"
        "}"
        "]}"
    };

    const char InvalidStatus[][JsonLen] =
    {
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"auto_backup\"             : false,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixee\","
            "\"has_thumbnail\"           : false,"
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"timeout_at_as_unixtime\"  : 1514732401,"
            "\"finished_at_as_unixtime\" : 1514732402,"
            "\"created_at_as_unixtime\"  : 1514732403,"
            "\"updated_at_as_unixtime\"  : 1514732404"
        "}"
        "]}"
    };

    const char LackOfNecessaryFieldJson[][JsonLen] =
    {
        "{ \"save_data_archives\": ["
        "{"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
         "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"series_id\"               : 100,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"saved_at_as_unixtime\"    : 1514732400,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}",
        "{ \"save_data_archives\": ["
        "{"
            "\"id\"                      : 1234,"
            "\"nsa_id\"                  : \"aaaaaaaaaaaaaaaa\","
            "\"application_id\"          : \"bbbbbbbbbbbbbbbb\","
            "\"device_id\"               : \"dddddddddddddddd\","
            "\"data_size\"               : 123456789,"
            "\"encoded_digest\"          : \"bD1gyr6Kck4t7RExh_MPDPfVK-3aoCYb7Cbp0FuBrX8=\","
            "\"status\"                  : \"fixed\","
            "\"launch_required_version\" : 5678,"
            "\"num_of_partitions\"   : 16,"
            "\"series_id\"               : 100,"
            "\"finished_at_as_unixtime\" : 1514732402"
        "}"
        "]}"
    };

    struct Cancelable
    {
        bool IsCancelled() const { return false; }
    };

    // 不正な入力を正しく無視するテスト
    void RunTestIgnore(const char json[][JsonLen], int count)
    {
        auto uid = nnt::olsc::GetFirstUserId();
        nn::fs::DeleteSystemSaveData(nnt::olsc::SystemSaveDataIdForTest, uid);
        srv::util::DefaultMountManager mountManager(nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest, nnt::olsc::MountInfoForTest);
        srv::database::SaveDataArchiveInfoCache sdaInfoCache(uid, mountManager);

        for (int i = 0; i < count; ++i)
        {
            olsc::srv::adaptor::SaveDataArchiveInfoStorageOutputStream outputStream(sdaInfoCache);
            olsc::srv::adaptor::SaveDataArchiveInfoAdaptor adaptor(&outputStream);

            nnt::olsc::MemoryInputStreamForRapidJson inputStream;
            inputStream.Set(json[i], JsonLen);

            const Cancelable c;
            NNT_EXPECT_RESULT_SUCCESS(http::json::ImportJsonByRapidJson<http::json::DefaultJsonErrorMap>(adaptor, inputStream, &c));
            NNT_EXPECT_RESULT_SUCCESS(adaptor.GetResult());

            EXPECT_TRUE(sdaInfoCache.GetCount() == 0);
        }
    }

    TEST(OlscJsonAdaptorTest, SaveDataMetaFail)
    {
        RunTestIgnore(InvalidDigest, sizeof(InvalidDigest) / JsonLen);
        RunTestIgnore(InvalidNumOfPartitions, sizeof(InvalidNumOfPartitions) / JsonLen);
        RunTestIgnore(InvalidDataSize, sizeof(InvalidDataSize) / JsonLen);
        RunTestIgnore(InvalidApplicationIdJson, sizeof(InvalidApplicationIdJson) / JsonLen);
        RunTestIgnore(InvalidStatus, sizeof(InvalidStatus) / JsonLen);
        RunTestIgnore(LackOfNecessaryFieldJson, sizeof(LackOfNecessaryFieldJson) / JsonLen);
    }

} // namespace save_data_archive_fail

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    nnt::olsc::Initialize();

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::olsc::Finalize();

    nnt::Exit(ret);
}
